/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.node.VersionInformation;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.core.Predicates;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.ConnectionManager;
import org.elasticsearch.transport.LinkedProjectConfig;
import org.elasticsearch.transport.NoSeedNodeLeftException;
import org.elasticsearch.transport.RemoteConnectionInfo;
import org.elasticsearch.transport.RemoteConnectionManager;
import org.elasticsearch.transport.RemoteConnectionStrategy;
import org.elasticsearch.transport.TransportService;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;

public class ProxyConnectionStrategy
extends RemoteConnectionStrategy {
    static final int CHANNELS_PER_CONNECTION = 1;
    private static final int MAX_CONNECT_ATTEMPTS_PER_RUN = 3;
    private final int maxNumConnections;
    private final String configuredAddress;
    private final String configuredServerName;
    private final Supplier<TransportAddress> address;
    private final AtomicReference<ClusterName> remoteClusterName = new AtomicReference();
    private final ConnectionManager.ConnectionValidator clusterNameValidator;

    ProxyConnectionStrategy(LinkedProjectConfig.ProxyLinkedProjectConfig config, TransportService transportService, RemoteConnectionManager connectionManager) {
        this(config, () -> ProxyConnectionStrategy.resolveAddress(config.proxyAddress()), transportService, connectionManager);
    }

    ProxyConnectionStrategy(LinkedProjectConfig.ProxyLinkedProjectConfig config, Supplier<TransportAddress> address, TransportService transportService, RemoteConnectionManager connectionManager) {
        super(config, transportService, connectionManager);
        this.maxNumConnections = config.maxNumConnections();
        this.configuredAddress = config.proxyAddress();
        this.configuredServerName = config.serverName();
        assert (!org.elasticsearch.common.Strings.isEmpty(this.configuredAddress)) : "Cannot use proxy connection strategy with no configured addresses";
        this.address = address;
        this.clusterNameValidator = (newConnection, actualProfile, listener) -> {
            assert (actualProfile.getTransportProfile().equals(connectionManager.getConnectionProfile().getTransportProfile())) : "transport profile must be consistent between the connection manager and the actual profile";
            transportService.handshake(RemoteConnectionManager.wrapConnectionWithRemoteClusterInfo(newConnection, this.clusterAlias, connectionManager.getCredentialsManager()), actualProfile.getHandshakeTimeout(), Predicates.always(), listener.map(resp -> {
                ClusterName remote = resp.getClusterName();
                if (this.remoteClusterName.compareAndSet(null, remote)) {
                    return null;
                }
                if (!this.remoteClusterName.get().equals(remote)) {
                    DiscoveryNode node = newConnection.getNode();
                    throw new ConnectTransportException(node, "handshake failed. unexpected remote cluster name " + String.valueOf(remote));
                }
                return null;
            }));
        };
    }

    static Writeable.Reader<RemoteConnectionInfo.ModeInfo> infoReader() {
        return ProxyModeInfo::new;
    }

    @Override
    protected boolean shouldOpenMoreConnections() {
        return this.connectionManager.size() < this.maxNumConnections;
    }

    @Override
    protected boolean strategyMustBeRebuilt(LinkedProjectConfig config) {
        assert (config instanceof LinkedProjectConfig.ProxyLinkedProjectConfig) : "expected config to be of type " + String.valueOf(ProxyConnectionStrategy.class);
        LinkedProjectConfig.ProxyLinkedProjectConfig proxyConfig = (LinkedProjectConfig.ProxyLinkedProjectConfig)config;
        return proxyConfig.maxNumConnections() != this.maxNumConnections || !this.configuredAddress.equals(proxyConfig.proxyAddress()) || !Objects.equals(proxyConfig.serverName(), this.configuredServerName);
    }

    @Override
    protected RemoteConnectionStrategy.ConnectionStrategy strategyType() {
        return RemoteConnectionStrategy.ConnectionStrategy.PROXY;
    }

    @Override
    protected void connectImpl(ActionListener<Void> listener) {
        this.performProxyConnectionProcess(listener);
    }

    @Override
    public RemoteConnectionInfo.ModeInfo getModeInfo() {
        return new ProxyModeInfo(this.configuredAddress, this.configuredServerName, this.maxNumConnections, this.connectionManager.size());
    }

    private void performProxyConnectionProcess(ActionListener<Void> listener) {
        this.openConnections(listener, 1);
    }

    private void openConnections(final ActionListener<Void> finished, final int attemptNumber) {
        if (attemptNumber <= 3) {
            TransportAddress resolved = this.address.get();
            final int remaining = this.maxNumConnections - this.connectionManager.size();
            ActionListener<Void> compositeListener = new ActionListener<Void>(){
                private final AtomicInteger successfulConnections = new AtomicInteger(0);
                private final CountDown countDown = new CountDown(remaining);
                private final Map<Tuple<Class<?>, String>, Exception> exceptions = new ConcurrentHashMap();

                @Override
                public void onResponse(Void v) {
                    this.successfulConnections.incrementAndGet();
                    if (this.countDown.countDown()) {
                        if (ProxyConnectionStrategy.this.shouldOpenMoreConnections()) {
                            ProxyConnectionStrategy.this.openConnections(finished, attemptNumber + 1);
                        } else {
                            assert (ProxyConnectionStrategy.this.connectionManager.size() > 0) : "must have at least one opened connection";
                            finished.onResponse(v);
                        }
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    this.exceptions.put(new Tuple(e.getClass(), e.getMessage()), e);
                    if (this.countDown.countDown()) {
                        if (attemptNumber >= 3 && ProxyConnectionStrategy.this.connectionManager.size() == 0) {
                            if (this.exceptions.values().stream().allMatch(RemoteConnectionStrategy::isRetryableException)) {
                                finished.onFailure(ProxyConnectionStrategy.this.getNoSeedNodeLeftException(this.exceptions.values()));
                            } else {
                                this.exceptions.values().stream().filter(e1 -> e1 != e).forEach(e::addSuppressed);
                                finished.onFailure(e);
                            }
                        } else {
                            ProxyConnectionStrategy.this.openConnections(finished, attemptNumber + 1);
                        }
                    }
                }
            };
            for (int i = 0; i < remaining; ++i) {
                String id = this.clusterAlias + "#" + String.valueOf(resolved);
                Map<Object, Object> attributes = org.elasticsearch.common.Strings.isNullOrEmpty(this.configuredServerName) ? Collections.emptyMap() : Collections.singletonMap("server_name", this.configuredServerName);
                DiscoveryNode node = new DiscoveryNode(null, id, resolved, attributes, DiscoveryNodeRole.roles(), new VersionInformation(Version.CURRENT.minimumCompatibilityVersion(), IndexVersions.MINIMUM_COMPATIBLE, IndexVersions.MINIMUM_READONLY_COMPATIBLE, IndexVersion.current()));
                this.connectionManager.connectToRemoteClusterNode(node, this.clusterNameValidator, compositeListener.delegateResponse((l, e) -> {
                    this.logger.debug(() -> Strings.format("failed to open remote connection [remote cluster: %s, address: %s]", this.clusterAlias, resolved), (Throwable)e);
                    l.onFailure((Exception)e);
                }));
            }
        } else {
            this.logger.debug("unable to open maximum number of connections [remote cluster: {}, opened: {}, maximum: {}]", (Object)this.clusterAlias, (Object)this.connectionManager.size(), (Object)this.maxNumConnections);
            finished.onResponse(null);
        }
    }

    private NoSeedNodeLeftException getNoSeedNodeLeftException(Collection<Exception> suppressedExceptions) {
        NoSeedNodeLeftException e = new NoSeedNodeLeftException("Unable to open any proxy connections to cluster [" + this.clusterAlias + "] at address [" + String.valueOf(this.address.get()) + "]");
        suppressedExceptions.forEach(e::addSuppressed);
        return e;
    }

    private static TransportAddress resolveAddress(String address) {
        return new TransportAddress(ProxyConnectionStrategy.parseConfiguredAddress(address));
    }

    public static class ProxyModeInfo
    implements RemoteConnectionInfo.ModeInfo {
        private final String address;
        private final String serverName;
        private final int maxSocketConnections;
        private final int numSocketsConnected;

        public ProxyModeInfo(String address, String serverName, int maxSocketConnections, int numSocketsConnected) {
            this.address = address;
            this.serverName = serverName;
            this.maxSocketConnections = maxSocketConnections;
            this.numSocketsConnected = numSocketsConnected;
        }

        private ProxyModeInfo(StreamInput input) throws IOException {
            this.address = input.readString();
            this.serverName = input.readString();
            this.maxSocketConnections = input.readVInt();
            this.numSocketsConnected = input.readVInt();
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.field("proxy_address", this.address);
            builder.field("server_name", this.serverName);
            builder.field("num_proxy_sockets_connected", this.numSocketsConnected);
            builder.field("max_proxy_socket_connections", this.maxSocketConnections);
            return builder;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.address);
            out.writeString(this.serverName);
            out.writeVInt(this.maxSocketConnections);
            out.writeVInt(this.numSocketsConnected);
        }

        @Override
        public boolean isConnected() {
            return this.numSocketsConnected > 0;
        }

        @Override
        public String modeName() {
            return "proxy";
        }

        @Override
        public RemoteConnectionStrategy.ConnectionStrategy modeType() {
            return RemoteConnectionStrategy.ConnectionStrategy.PROXY;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ProxyModeInfo otherProxy = (ProxyModeInfo)o;
            return this.maxSocketConnections == otherProxy.maxSocketConnections && this.numSocketsConnected == otherProxy.numSocketsConnected && Objects.equals(this.address, otherProxy.address) && Objects.equals(this.serverName, otherProxy.serverName);
        }

        public int hashCode() {
            return Objects.hash(this.address, this.serverName, this.maxSocketConnections, this.numSocketsConnected);
        }
    }
}

