"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AbuseIPDBConnector = void 0;
var _zod = require("@kbn/zod");
var _connector_spec_ui = require("../connector_spec_ui");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Example: AbuseIPDB Connector
 *
 * This demonstrates a threat intelligence connector with:
 * - IP reputation checking
 * - Abuse reporting
 * - Geolocation and ISP data
 * - Bulk IP checking
 *
 * MVP implementation focusing on core IP reputation actions.
 */

const AbuseIPDBConnector = exports.AbuseIPDBConnector = {
  metadata: {
    id: '.abuseipdb',
    displayName: 'AbuseIPDB',
    description: 'IP reputation checking and abuse reporting',
    minimumLicense: 'gold',
    supportedFeatureIds: ['alerting', 'siem']
  },
  schema: _zod.z.discriminatedUnion('method', [_zod.z.object({
    method: _zod.z.literal('headers'),
    headers: _zod.z.object({
      Key: _connector_spec_ui.UISchemas.secret().describe('API Key')
    })
  })]),
  actions: {
    checkIp: {
      isTool: true,
      input: _zod.z.object({
        ipAddress: _zod.z.string().ip().describe('IP address to check'),
        maxAgeInDays: _zod.z.number().int().min(1).max(365).optional().default(90).describe('Maximum age of reports in days')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.get('https://api.abuseipdb.com/api/v2/check', {
          params: {
            ipAddress: typedInput.ipAddress,
            maxAgeInDays: typedInput.maxAgeInDays || 90
          }
        });
        return {
          ipAddress: response.data.data.ipAddress,
          abuseConfidenceScore: response.data.data.abuseConfidenceScore,
          usageType: response.data.data.usageType,
          isp: response.data.data.isp,
          countryCode: response.data.data.countryCode,
          totalReports: response.data.data.totalReports
        };
      }
    },
    reportIp: {
      isTool: true,
      input: _zod.z.object({
        ip: _zod.z.string().ip().describe('IP address to report'),
        categories: _zod.z.array(_zod.z.number().int()).min(1).describe('Abuse category IDs'),
        comment: _zod.z.string().optional().describe('Additional details')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.post('https://api.abuseipdb.com/api/v2/report', new URLSearchParams({
          ip: typedInput.ip,
          categories: typedInput.categories.join(','),
          ...(typedInput.comment && {
            comment: typedInput.comment
          })
        }), {
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
          }
        });
        return {
          ipAddress: response.data.data.ipAddress,
          abuseConfidenceScore: response.data.data.abuseConfidenceScore
        };
      }
    },
    getIpInfo: {
      isTool: true,
      input: _zod.z.object({
        ipAddress: _zod.z.string().ip().describe('IP address to lookup')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.get('https://api.abuseipdb.com/api/v2/check', {
          params: {
            ipAddress: typedInput.ipAddress,
            verbose: true
          }
        });
        return {
          ipAddress: response.data.data.ipAddress,
          isPublic: response.data.data.isPublic,
          ipVersion: response.data.data.ipVersion,
          isWhitelisted: response.data.data.isWhitelisted,
          abuseConfidenceScore: response.data.data.abuseConfidenceScore,
          countryCode: response.data.data.countryCode,
          usageType: response.data.data.usageType,
          isp: response.data.data.isp,
          domain: response.data.data.domain
        };
      }
    },
    bulkCheck: {
      isTool: true,
      input: _zod.z.object({
        network: _zod.z.string().describe('Network in CIDR notation'),
        maxAgeInDays: _zod.z.number().int().min(1).max(365).optional().default(30).describe('Maximum age of reports in days')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.get('https://api.abuseipdb.com/api/v2/check-block', {
          params: {
            network: typedInput.network,
            maxAgeInDays: typedInput.maxAgeInDays || 30
          }
        });
        return {
          networkAddress: response.data.data.networkAddress,
          netmask: response.data.data.netmask,
          reportedAddress: response.data.data.reportedAddress
        };
      }
    }
  },
  test: {
    handler: async ctx => {
      try {
        await ctx.client.get('https://api.abuseipdb.com/api/v2/check', {
          params: {
            ipAddress: '8.8.8.8'
          }
        });
        return {
          ok: true,
          message: 'Successfully connected to AbuseIPDB API'
        };
      } catch (error) {
        return {
          ok: false,
          message: `Failed to connect: ${error}`
        };
      }
    },
    description: 'Verifies AbuseIPDB API key'
  }
};