"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertMigrationFunction = convertMigrationFunction;
exports.downgradeRequired = downgradeRequired;
exports.transformComparator = transformComparator;
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
var _migration_logger = require("../core/migration_logger");
var _utils = require("./pipelines/utils");
var _transform_saved_object_document_error = require("../core/transform_saved_object_document_error");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const TRANSFORM_PRIORITY = [_types.TransformType.Core, _types.TransformType.Reference, _types.TransformType.Convert, _types.TransformType.Migrate];

/**
 * If a specific transform function fails, this tacks on a bit of information
 * about the document and transform that caused the failure.
 */
function convertMigrationFunction(version, type, migration, log) {
  const context = Object.freeze({
    log: new _migration_logger.MigrationLogger(log),
    migrationVersion: version,
    convertToMultiNamespaceTypeVersion: type.convertToMultiNamespaceTypeVersion,
    isSingleNamespaceType: type.namespaceType === 'single'
  });
  return function tryTransformDoc(doc) {
    try {
      const transformFn = (0, _lodash.isFunction)(migration) ? migration : migration.transform;
      const result = transformFn(doc, context);

      // A basic check to help migration authors detect basic errors
      // (e.g. forgetting to return the transformed doc)
      if (!result || !result.type) {
        throw new Error(`Invalid saved object returned from migration ${type.name}:${version}.`);
      }
      return {
        transformedDoc: result,
        additionalDocs: []
      };
    } catch (error) {
      log.error(`Error trying to transform document: ${error.message}`);
      throw new _transform_saved_object_document_error.TransformSavedObjectDocumentError(error, version);
    }
  };
}

/**
 * Transforms are sorted in ascending order by version depending on their type:
 *  - `core` transforms always run first no matter version;
 *  - `reference` transforms have priority in case of the same version;
 *  - `convert` transforms run after in case of the same version;
 *  - 'migrate' transforms always run last.
 * This is because:
 *  1. 'convert' transforms get rid of the `namespace` field, which must be present for 'reference' transforms to function correctly.
 *  2. 'migrate' transforms are defined by the consumer, and may change the object type or `migrationVersion` which resets the migration loop
 *     and could cause any remaining transforms for this version to be skipped.One version may contain multiple transforms.
 */
function transformComparator(a, b) {
  const aPriority = TRANSFORM_PRIORITY.indexOf(a.transformType);
  const bPriority = TRANSFORM_PRIORITY.indexOf(b.transformType);
  if (aPriority !== bPriority && (a.transformType === _types.TransformType.Core || b.transformType === _types.TransformType.Core)) {
    return aPriority - bPriority;
  }
  return _semver.default.compare(a.version, b.version) || aPriority - bPriority;
}

/**
 * Returns true if the given document has an higher version that the last known version, false otherwise
 */
function downgradeRequired(doc, latestVersions, targetTypeVersion) {
  var _doc$typeMigrationVer, _doc$migrationVersion;
  const docTypeVersion = (_doc$typeMigrationVer = doc.typeMigrationVersion) !== null && _doc$typeMigrationVer !== void 0 ? _doc$typeMigrationVer : (_doc$migrationVersion = doc.migrationVersion) === null || _doc$migrationVersion === void 0 ? void 0 : _doc$migrationVersion[doc.type];
  const latestMigrationVersion = targetTypeVersion !== null && targetTypeVersion !== void 0 ? targetTypeVersion : (0, _utils.maxVersion)(latestVersions[_types.TransformType.Migrate], latestVersions[_types.TransformType.Convert]);
  if (!docTypeVersion || !latestMigrationVersion) {
    return false;
  }
  return _semver.default.gt(docTypeVersion, latestMigrationVersion);
}