"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateColorRangeValue = exports.updateColorRangeColor = exports.deleteColorRange = exports.addColorRange = void 0;
var _palettes = require("../../../../palettes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Allows to update a ColorRange
 * @internal
 */
const updateColorRangeItem = (colorRanges, index, payload) => {
  const ranges = [...colorRanges];
  ranges[index] = {
    ...ranges[index],
    ...payload
  };
  return ranges;
};

/**
 * Add new color range after the last item
 * @internal
 */
const addColorRange = (colorRanges, rangeType, dataBounds) => {
  let newColorRanges = [...colorRanges];
  const lastIndex = newColorRanges.length - 1;
  const lastStart = newColorRanges[lastIndex].start;
  const lastEnd = newColorRanges[lastIndex].end;
  const lastColor = newColorRanges[lastIndex].color;
  const {
    max: dataMax
  } = (0, _palettes.getDataMinMax)(rangeType, dataBounds);
  const max = Math.max(dataMax, lastEnd);
  let insertEnd = (0, _palettes.roundValue)(Math.min(lastStart + 1, max));
  if (insertEnd === Number.NEGATIVE_INFINITY) {
    insertEnd = 1;
  }
  newColorRanges = updateColorRangeItem(newColorRanges, lastIndex, {
    end: insertEnd
  });
  newColorRanges.push({
    color: lastColor,
    start: insertEnd,
    end: lastEnd === insertEnd ? lastEnd + 1 : lastEnd
  });
  return newColorRanges;
};

/**
 * Delete ColorRange
 * @internal
 */
exports.addColorRange = addColorRange;
const deleteColorRange = (index, colorRanges) => {
  const lastIndex = colorRanges.length - 1;
  let ranges = colorRanges;
  if (index !== 0) {
    if (index !== lastIndex) {
      ranges = updateColorRangeItem(ranges, index - 1, {
        end: ranges[index + 1].start
      });
    }
    if (index === lastIndex) {
      ranges = updateColorRangeItem(ranges, index - 1, {
        end: colorRanges[index].end
      });
    }
  }
  return ranges.filter((item, i) => i !== index);
};

/**
 * Update ColorRange value
 * @internal
 */
exports.deleteColorRange = deleteColorRange;
const updateColorRangeValue = (index, value, accessor, colorRanges) => {
  const parsedValue = value ? parseFloat(value) : Number.NaN;
  let ranges = colorRanges;
  if (accessor === 'end') {
    ranges = updateColorRangeItem(ranges, index, {
      end: parsedValue
    });
  } else {
    ranges = updateColorRangeItem(ranges, index, {
      start: parsedValue
    });
    if (index > 0) {
      ranges = updateColorRangeItem(ranges, index - 1, {
        end: parsedValue
      });
    }
  }
  return ranges;
};

/**
 * Update ColorRange color
 * @internal
 */
exports.updateColorRangeValue = updateColorRangeValue;
const updateColorRangeColor = (index, color, colorRanges) => updateColorRangeItem(colorRanges, index, {
  color
});
exports.updateColorRangeColor = updateColorRangeColor;