"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.determineDelayedAlerts = determineDelayedAlerts;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function determineDelayedAlerts({
  newAlerts,
  activeAlerts,
  trackedActiveAlerts,
  recoveredAlerts,
  trackedRecoveredAlerts,
  alertDelay,
  startedAt,
  ruleRunMetricsStore
}) {
  let delayedAlertsCount = 0;
  for (const id of (0, _lodash.keys)(activeAlerts)) {
    const alert = activeAlerts[id];
    alert.incrementActiveCount();
    // do not trigger an action notification if the number of consecutive
    // active alerts is less than the rule alertDelay threshold
    if (alert.getActiveCount() < alertDelay) {
      // remove from new alerts and active alerts
      delete newAlerts[id];
      delete activeAlerts[id];
      delayedAlertsCount += 1;
    } else {
      // if the active count is equal to the alertDelay it is considered a new alert
      if (alert.getActiveCount() === alertDelay) {
        const currentTime = startedAt !== null && startedAt !== void 0 ? startedAt : new Date().toISOString();
        const state = alert.getState();
        // keep the state and update the start time and duration
        alert.replaceState({
          ...state,
          start: currentTime,
          duration: '0'
        });
        newAlerts[id] = alert;
      }
    }
  }
  for (const id of (0, _lodash.keys)(recoveredAlerts)) {
    const alert = recoveredAlerts[id];
    // if alert has not reached the alertDelay threshold don't recover the alert
    const activeCount = alert.getActiveCount();
    if (activeCount > 0 && activeCount < alertDelay) {
      // remove from recovered alerts
      delete recoveredAlerts[id];
      delete trackedRecoveredAlerts[id];
    }
    alert.resetActiveCount();
  }
  ruleRunMetricsStore.setNumberOfDelayedAlerts(delayedAlertsCount);
  return {
    newAlerts,
    activeAlerts,
    trackedActiveAlerts,
    recoveredAlerts,
    trackedRecoveredAlerts
  };
}