"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionTypeRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _i18n = require("@kbn/i18n");
var _server = require("@kbn/task-manager-plugin/server");
var _common = require("../common");
var _lib = require("./lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ActionTypeRegistry {
  constructor(constructorParams) {
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "actionTypes", new Map());
    (0, _defineProperty2.default)(this, "taskRunnerFactory", void 0);
    (0, _defineProperty2.default)(this, "actionsConfigUtils", void 0);
    (0, _defineProperty2.default)(this, "licenseState", void 0);
    (0, _defineProperty2.default)(this, "inMemoryConnectors", void 0);
    (0, _defineProperty2.default)(this, "licensing", void 0);
    /**
     * Returns true if the action type is a system action type
     */
    (0, _defineProperty2.default)(this, "isSystemActionType", actionTypeId => {
      var _this$actionTypes$get;
      return Boolean((_this$actionTypes$get = this.actionTypes.get(actionTypeId)) === null || _this$actionTypes$get === void 0 ? void 0 : _this$actionTypes$get.isSystemActionType);
    });
    /**
     * Returns true if the connector type has a sub-feature type defined
     */
    (0, _defineProperty2.default)(this, "hasSubFeature", actionTypeId => {
      var _this$actionTypes$get2;
      return Boolean((_this$actionTypes$get2 = this.actionTypes.get(actionTypeId)) === null || _this$actionTypes$get2 === void 0 ? void 0 : _this$actionTypes$get2.subFeature);
    });
    this.taskManager = constructorParams.taskManager;
    this.taskRunnerFactory = constructorParams.taskRunnerFactory;
    this.actionsConfigUtils = constructorParams.actionsConfigUtils;
    this.licenseState = constructorParams.licenseState;
    this.inMemoryConnectors = constructorParams.inMemoryConnectors;
    this.licensing = constructorParams.licensing;
  }

  /**
   * Returns if the action type registry has the given action type registered
   */
  has(id) {
    return this.actionTypes.has(id);
  }

  /**
   * Throws error if action type is not enabled.
   */
  ensureActionTypeEnabled(id) {
    this.actionsConfigUtils.ensureActionTypeEnabled(id);
    // Important to happen last because the function will notify of feature usage at the
    // same time and it shouldn't notify when the action type isn't enabled
    this.licenseState.ensureLicenseForActionType(this.get(id));
  }

  /**
   * Returns true if action type is enabled in the config and a valid license is used.
   */
  isActionTypeEnabled(id, options = {
    notifyUsage: false
  }) {
    return this.actionsConfigUtils.isActionTypeEnabled(id) && this.licenseState.isLicenseValidForActionType(this.get(id), options).isValid === true;
  }

  /**
   * Returns true if action type is enabled or preconfigured.
   * An action type can be disabled but used with a preconfigured action.
   * This does not apply to system actions as those can be disabled.
   */
  isActionExecutable(actionId, actionTypeId, options = {
    notifyUsage: false
  }) {
    const validLicense = this.licenseState.isLicenseValidForActionType(this.get(actionTypeId), options).isValid;
    if (validLicense === false) return false;
    const actionTypeEnabled = this.isActionTypeEnabled(actionTypeId, options);
    const inMemoryConnector = this.inMemoryConnectors.find(connector => connector.id === actionId);
    return actionTypeEnabled || !actionTypeEnabled && ((inMemoryConnector === null || inMemoryConnector === void 0 ? void 0 : inMemoryConnector.isPreconfigured) === true || (inMemoryConnector === null || inMemoryConnector === void 0 ? void 0 : inMemoryConnector.isSystemAction) === true);
  }
  /**
   * Returns the kibana privileges
   */
  getActionKibanaPrivileges(actionTypeId, params, source) {
    var _actionType$getKibana, _actionType$getKibana2;
    const actionType = this.actionTypes.get(actionTypeId);
    if (!(actionType !== null && actionType !== void 0 && actionType.isSystemActionType) && !(actionType !== null && actionType !== void 0 && actionType.subFeature)) {
      return [];
    }
    return (_actionType$getKibana = actionType === null || actionType === void 0 ? void 0 : (_actionType$getKibana2 = actionType.getKibanaPrivileges) === null || _actionType$getKibana2 === void 0 ? void 0 : _actionType$getKibana2.call(actionType, {
      params,
      source
    })) !== null && _actionType$getKibana !== void 0 ? _actionType$getKibana : [];
  }

  /**
   * Registers an action type to the action type registry
   */
  register(actionType) {
    if (this.has(actionType.id)) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.duplicateActionTypeErrorMessage', {
        defaultMessage: 'Action type "{id}" is already registered.',
        values: {
          id: actionType.id
        }
      }));
    }
    if (!actionType.supportedFeatureIds || actionType.supportedFeatureIds.length === 0) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.missingSupportedFeatureIds', {
        defaultMessage: 'At least one "supportedFeatureId" value must be supplied for connector type "{connectorTypeId}".',
        values: {
          connectorTypeId: actionType.id
        }
      }));
    }
    if (!(0, _common.areValidFeatures)(actionType.supportedFeatureIds)) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.invalidConnectorFeatureIds', {
        defaultMessage: 'Invalid feature ids "{ids}" for connector type "{connectorTypeId}".',
        values: {
          connectorTypeId: actionType.id,
          ids: actionType.supportedFeatureIds.join(',')
        }
      }));
    }
    if (!actionType.isSystemActionType && !actionType.subFeature && actionType.getKibanaPrivileges) {
      throw new Error(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.register.invalidKibanaPrivileges', {
        defaultMessage: 'Kibana privilege authorization is only supported for system actions and action types that are registered under a sub-feature'
      }));
    }
    const maxAttempts = this.actionsConfigUtils.getMaxAttempts({
      actionTypeId: actionType.id,
      actionTypeMaxAttempts: actionType.maxAttempts
    });
    this.actionTypes.set(actionType.id, {
      ...actionType
    });
    this.taskManager.registerTaskDefinitions({
      [`actions:${actionType.id}`]: {
        title: actionType.name,
        maxAttempts,
        cost: _server.TaskCost.Tiny,
        createTaskRunner: context => this.taskRunnerFactory.create(context)
      }
    });
    // No need to notify usage on basic action types
    if (actionType.minimumLicenseRequired !== 'basic') {
      this.licensing.featureUsage.register((0, _lib.getActionTypeFeatureUsageName)(actionType), actionType.minimumLicenseRequired);
    }
  }

  /**
   * Returns an action type, throws if not registered
   */
  get(id) {
    if (!this.has(id)) {
      throw _boom.default.badRequest(_i18n.i18n.translate('xpack.actions.actionTypeRegistry.get.missingActionTypeErrorMessage', {
        defaultMessage: 'Action type "{id}" is not registered.',
        values: {
          id
        }
      }));
    }
    return this.actionTypes.get(id);
  }

  /**
   * Returns a list of registered action types [{ id, name, enabled }], filtered by featureId if provided.
   */
  list({
    featureId,
    exposeValidation
  } = {
    exposeValidation: false
  }) {
    return Array.from(this.actionTypes).filter(([_, actionType]) => {
      return featureId ? actionType.supportedFeatureIds.includes(featureId) : true;
    }).map(([actionTypeId, actionType]) => ({
      id: actionTypeId,
      name: actionType.name,
      minimumLicenseRequired: actionType.minimumLicenseRequired,
      enabled: this.isActionTypeEnabled(actionTypeId),
      enabledInConfig: this.actionsConfigUtils.isActionTypeEnabled(actionTypeId),
      enabledInLicense: !!this.licenseState.isLicenseValidForActionType(actionType).isValid,
      supportedFeatureIds: actionType.supportedFeatureIds,
      isSystemActionType: !!actionType.isSystemActionType,
      subFeature: actionType.subFeature,
      ...(exposeValidation === true ? {
        validate: {
          params: actionType.validate.params
        }
      } : {}),
      isDeprecated: !!actionType.isDeprecated
    }));
  }

  /**
   * Returns the actions configuration utilities
   */
  getUtils() {
    return this.actionsConfigUtils;
  }
  getAllTypes() {
    return [...this.list().map(({
      id
    }) => id)];
  }
  isDeprecated(actionTypeId) {
    var _this$actionTypes$get3;
    return Boolean((_this$actionTypes$get3 = this.actionTypes.get(actionTypeId)) === null || _this$actionTypes$get3 === void 0 ? void 0 : _this$actionTypes$get3.isDeprecated);
  }
}
exports.ActionTypeRegistry = ActionTypeRegistry;