/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.BackoffPolicy;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.BlobStoreActionStats;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.repositories.RepositoriesMetrics;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobContainer;
import org.elasticsearch.repositories.s3.S3RepositoriesMetrics;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.S3Service;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.ThreadPool;
import software.amazon.awssdk.awscore.AwsRequest;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryUtils;
import software.amazon.awssdk.http.HttpMetric;
import software.amazon.awssdk.metrics.MetricCollection;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.ObjectCannedACL;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.S3Error;
import software.amazon.awssdk.services.s3.model.StorageClass;

class S3BlobStore
implements BlobStore {
    public static final String CUSTOM_QUERY_PARAMETER_COPY_SOURCE = "x-amz-copy-source";
    public static final String CUSTOM_QUERY_PARAMETER_PURPOSE = "x-purpose";
    static final int MAX_BULK_DELETES = 1000;
    static final int MAX_DELETE_EXCEPTIONS = 10;
    private static final Logger logger = LogManager.getLogger(S3BlobStore.class);
    @Nullable
    private final ProjectId projectId;
    private final S3Service service;
    private final BigArrays bigArrays;
    private final String bucket;
    private final ByteSizeValue bufferSize;
    private final ByteSizeValue maxCopySizeBeforeMultipart;
    private final boolean serverSideEncryption;
    private final ObjectCannedACL cannedACL;
    private final StorageClass storageClass;
    private final boolean supportsConditionalWrites;
    private final RepositoryMetadata repositoryMetadata;
    private final ThreadPool threadPool;
    private final Executor snapshotExecutor;
    private final S3RepositoriesMetrics s3RepositoriesMetrics;
    private final StatsCollectors statsCollectors = new StatsCollectors();
    private final int bulkDeletionBatchSize;
    private final BackoffPolicy retryThrottledDeleteBackoffPolicy;
    private final TimeValue getRegisterRetryDelay;
    private final boolean addPurposeCustomQueryParameter;

    S3BlobStore(@Nullable ProjectId projectId, S3Service service, String bucket, boolean serverSideEncryption, ByteSizeValue bufferSize, ByteSizeValue maxCopySizeBeforeMultipart, String cannedACL, String storageClass, boolean supportConditionalWrites, RepositoryMetadata repositoryMetadata, BigArrays bigArrays, ThreadPool threadPool, S3RepositoriesMetrics s3RepositoriesMetrics, BackoffPolicy retryThrottledDeleteBackoffPolicy) {
        this.projectId = projectId;
        this.service = service;
        this.bigArrays = bigArrays;
        this.bucket = bucket;
        this.serverSideEncryption = serverSideEncryption;
        this.bufferSize = bufferSize;
        this.maxCopySizeBeforeMultipart = maxCopySizeBeforeMultipart;
        this.cannedACL = S3BlobStore.initCannedACL(cannedACL);
        this.storageClass = S3BlobStore.initStorageClass(storageClass);
        this.supportsConditionalWrites = supportConditionalWrites;
        this.repositoryMetadata = repositoryMetadata;
        this.threadPool = threadPool;
        this.snapshotExecutor = threadPool.executor("snapshot");
        this.s3RepositoriesMetrics = s3RepositoriesMetrics;
        this.bulkDeletionBatchSize = (Integer)S3Repository.DELETION_BATCH_SIZE_SETTING.get(repositoryMetadata.settings());
        this.retryThrottledDeleteBackoffPolicy = retryThrottledDeleteBackoffPolicy;
        this.getRegisterRetryDelay = (TimeValue)S3Repository.GET_REGISTER_RETRY_DELAY.get(repositoryMetadata.settings());
        this.addPurposeCustomQueryParameter = service.settings((ProjectId)projectId, (RepositoryMetadata)repositoryMetadata).addPurposeCustomQueryParameter;
    }

    MetricPublisher getMetricPublisher(Operation operation, OperationPurpose purpose) {
        return this.statsCollectors.getMetricPublisher(operation, purpose);
    }

    public Executor getSnapshotExecutor() {
        return this.snapshotExecutor;
    }

    public TimeValue getCompareAndExchangeTimeToLive() {
        return this.service.compareAndExchangeTimeToLive;
    }

    public TimeValue getCompareAndExchangeAntiContentionDelay() {
        return this.service.compareAndExchangeAntiContentionDelay;
    }

    public String toString() {
        return this.bucket;
    }

    public AmazonS3Reference clientReference() {
        return this.service.client(this.projectId, this.repositoryMetadata);
    }

    final int getMaxRetries() {
        return this.service.settings((ProjectId)this.projectId, (RepositoryMetadata)this.repositoryMetadata).maxRetries;
    }

    public String bucket() {
        return this.bucket;
    }

    public BigArrays bigArrays() {
        return this.bigArrays;
    }

    public boolean serverSideEncryption() {
        return this.serverSideEncryption;
    }

    public long bufferSizeInBytes() {
        return this.bufferSize.getBytes();
    }

    public long maxCopySizeBeforeMultipart() {
        return this.maxCopySizeBeforeMultipart.getBytes();
    }

    public RepositoryMetadata getRepositoryMetadata() {
        return this.repositoryMetadata;
    }

    public S3RepositoriesMetrics getS3RepositoriesMetrics() {
        return this.s3RepositoriesMetrics;
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new S3BlobContainer(path, this);
    }

    void deleteBlobs(OperationPurpose purpose, Iterator<String> blobNames) throws IOException {
        if (!blobNames.hasNext()) {
            return;
        }
        ArrayList<ObjectIdentifier> partition = new ArrayList<ObjectIdentifier>();
        try {
            DeletionExceptions deletionExceptions = new DeletionExceptions();
            blobNames.forEachRemaining(key -> {
                partition.add((ObjectIdentifier)ObjectIdentifier.builder().key(key).build());
                if (partition.size() == this.bulkDeletionBatchSize) {
                    this.deletePartition(purpose, partition, deletionExceptions);
                    partition.clear();
                }
            });
            if (!partition.isEmpty()) {
                this.deletePartition(purpose, partition, deletionExceptions);
            }
            if (deletionExceptions.exception != null) {
                throw deletionExceptions.exception;
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to delete blobs " + String.valueOf(partition.stream().limit(10L).toList()), e);
        }
    }

    private void deletePartition(OperationPurpose purpose, List<ObjectIdentifier> partition, DeletionExceptions deletionExceptions) {
        Iterator retries = this.retryThrottledDeleteBackoffPolicy.iterator();
        int retryCounter = 0;
        while (true) {
            try (AmazonS3Reference clientReference = this.clientReference();){
                DeleteObjectsResponse response = clientReference.client().deleteObjects(S3BlobStore.bulkDelete(purpose, this, partition));
                if (response.hasErrors()) {
                    ElasticsearchException exception = new ElasticsearchException(this.buildDeletionErrorMessage(response.errors()), new Object[0]);
                    logger.warn(exception.getMessage(), (Throwable)exception);
                    deletionExceptions.useOrMaybeSuppress((Exception)exception);
                    return;
                }
                this.s3RepositoriesMetrics.retryDeletesHistogram().record((long)retryCounter);
                return;
            }
            catch (SdkException e) {
                if (this.shouldRetryDelete(purpose) && RetryUtils.isThrottlingException((SdkException)e)) {
                    if (this.maybeDelayAndRetryDelete(retries)) {
                        ++retryCounter;
                        continue;
                    }
                    this.s3RepositoriesMetrics.retryDeletesHistogram().record((long)retryCounter);
                    deletionExceptions.useOrMaybeSuppress((Exception)((Object)e));
                    return;
                }
                deletionExceptions.useOrMaybeSuppress((Exception)((Object)e));
                return;
            }
            break;
        }
    }

    private String buildDeletionErrorMessage(List<S3Error> errors) {
        StringBuilder sb = new StringBuilder("Failed to delete some blobs ");
        for (int i = 0; i < errors.size() && i < 10; ++i) {
            S3Error err = errors.get(i);
            sb.append("[").append(err.key()).append("][").append(err.code()).append("][").append(err.message()).append("]");
            if (i >= errors.size() - 1) continue;
            sb.append(",");
        }
        if (errors.size() > 10) {
            sb.append("... (").append(errors.size()).append(" in total, ").append(errors.size() - 10).append(" omitted)");
        }
        return sb.toString();
    }

    private boolean maybeDelayAndRetryDelete(Iterator<TimeValue> retries) {
        if (retries.hasNext()) {
            try {
                Thread.sleep(retries.next().millis());
                return true;
            }
            catch (InterruptedException iex) {
                Thread.currentThread().interrupt();
                logger.warn("Aborting tenacious snapshot delete retries due to interrupt");
            }
        } else {
            logger.warn("Exceeded maximum tenacious snapshot delete retries, aborting. Using back-off policy " + String.valueOf(this.retryThrottledDeleteBackoffPolicy) + ", see the throttled_delete_retry.* S3 repository properties to configure the back-off parameters");
        }
        return false;
    }

    private boolean shouldRetryDelete(OperationPurpose operationPurpose) {
        return operationPurpose == OperationPurpose.SNAPSHOT_DATA || operationPurpose == OperationPurpose.SNAPSHOT_METADATA;
    }

    private static DeleteObjectsRequest bulkDelete(OperationPurpose purpose, S3BlobStore blobStore, List<ObjectIdentifier> blobs) {
        DeleteObjectsRequest.Builder requestBuilder = DeleteObjectsRequest.builder().bucket(blobStore.bucket()).delete(b -> b.quiet(Boolean.valueOf(true)).objects((Collection)blobs));
        S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)requestBuilder, blobStore, Operation.DELETE_OBJECTS, purpose);
        return (DeleteObjectsRequest)requestBuilder.build();
    }

    public void close() throws IOException {
        this.service.onBlobStoreClose(this.projectId);
    }

    public Map<String, BlobStoreActionStats> stats() {
        return this.statsCollectors.statsMap(this.service.isStateless);
    }

    StatsCollectors getStatsCollectors() {
        return this.statsCollectors;
    }

    public ObjectCannedACL getCannedACL() {
        return this.cannedACL;
    }

    public StorageClass getStorageClass() {
        return this.storageClass;
    }

    public TimeValue getGetRegisterRetryDelay() {
        return this.getRegisterRetryDelay;
    }

    public static StorageClass initStorageClass(String storageClassName) {
        StorageClass storageClass;
        if (storageClassName == null || storageClassName.equals("")) {
            return StorageClass.STANDARD;
        }
        try {
            storageClass = StorageClass.fromValue((String)storageClassName.toUpperCase(Locale.ENGLISH));
        }
        catch (Exception e) {
            throw new BlobStoreException("`" + storageClassName + "` is not a valid S3 Storage Class.", (Throwable)e);
        }
        if (storageClass.equals((Object)StorageClass.GLACIER)) {
            throw new BlobStoreException("Glacier storage class is not supported");
        }
        if (storageClass.equals((Object)StorageClass.UNKNOWN_TO_SDK_VERSION)) {
            throw new BlobStoreException("`" + storageClassName + "` is not a known S3 Storage Class.");
        }
        return storageClass;
    }

    public static ObjectCannedACL initCannedACL(String cannedACL) {
        if (cannedACL == null || cannedACL.equals("")) {
            return ObjectCannedACL.PRIVATE;
        }
        for (ObjectCannedACL cur : ObjectCannedACL.values()) {
            if (!cur.toString().equalsIgnoreCase(cannedACL)) continue;
            return cur;
        }
        throw new BlobStoreException("cannedACL is not valid: [" + cannedACL + "]");
    }

    ThreadPool getThreadPool() {
        return this.threadPool;
    }

    static void configureRequestForMetrics(AwsRequest.Builder request, S3BlobStore blobStore, Operation operation, OperationPurpose purpose) {
        request.overrideConfiguration(builder -> {
            builder.metricPublishers(List.of(blobStore.getMetricPublisher(operation, purpose)));
            blobStore.addPurposeQueryParameter(purpose, (AwsRequestOverrideConfiguration.Builder)builder);
        });
    }

    public void addPurposeQueryParameter(OperationPurpose purpose, AwsRequestOverrideConfiguration.Builder builder) {
        if (this.addPurposeCustomQueryParameter || purpose == OperationPurpose.REPOSITORY_ANALYSIS) {
            builder.putRawQueryParameter(CUSTOM_QUERY_PARAMETER_PURPOSE, purpose.getKey());
        }
    }

    public boolean supportsConditionalWrites(OperationPurpose purpose) {
        return this.supportsConditionalWrites || purpose == OperationPurpose.REPOSITORY_ANALYSIS;
    }

    class StatsCollectors {
        final Map<StatsKey, ElasticsearchS3MetricsCollector> collectors = new ConcurrentHashMap<StatsKey, ElasticsearchS3MetricsCollector>();

        StatsCollectors() {
        }

        MetricPublisher getMetricPublisher(Operation operation, OperationPurpose purpose) {
            return this.collectors.computeIfAbsent(new StatsKey(operation, purpose), k -> this.buildMetricPublisher(k.operation(), k.purpose()));
        }

        Map<String, BlobStoreActionStats> statsMap(boolean isStateless) {
            if (isStateless) {
                return this.collectors.entrySet().stream().collect(Collectors.toUnmodifiableMap(entry -> ((StatsKey)entry.getKey()).toString(), entry -> ((ElasticsearchS3MetricsCollector)entry.getValue()).getEndpointStats()));
            }
            Map<String, BlobStoreActionStats> m = Arrays.stream(Operation.values()).collect(Collectors.toMap(Operation::getKey, e -> BlobStoreActionStats.ZERO));
            this.collectors.forEach((sk, v) -> m.compute(sk.operation().getKey(), (k, c) -> Objects.requireNonNull(c).add(v.getEndpointStats())));
            return Map.copyOf(m);
        }

        ElasticsearchS3MetricsCollector buildMetricPublisher(Operation operation, OperationPurpose purpose) {
            return new ElasticsearchS3MetricsCollector(operation, purpose);
        }
    }

    static final class Operation
    extends Enum<Operation> {
        public static final /* enum */ Operation HEAD_OBJECT = new Operation("HeadObject");
        public static final /* enum */ Operation GET_OBJECT = new Operation("GetObject");
        public static final /* enum */ Operation LIST_OBJECTS = new Operation("ListObjects");
        public static final /* enum */ Operation PUT_OBJECT = new Operation("PutObject");
        public static final /* enum */ Operation PUT_MULTIPART_OBJECT = new Operation("PutMultipartObject");
        public static final /* enum */ Operation DELETE_OBJECTS = new Operation("DeleteObjects");
        public static final /* enum */ Operation ABORT_MULTIPART_OBJECT = new Operation("AbortMultipartObject");
        public static final /* enum */ Operation COPY_OBJECT = new Operation("CopyObject");
        public static final /* enum */ Operation COPY_MULTIPART_OBJECT = new Operation("CopyMultipartObject");
        private final String key;
        private static final Predicate<String> IS_PUT_MULTIPART_OPERATION;
        private static final Predicate<String> IS_COPY_MULTIPART_OPERATION;
        private static final Predicate<String> IS_LIST_OPERATION;
        private static final /* synthetic */ Operation[] $VALUES;

        public static Operation[] values() {
            return (Operation[])$VALUES.clone();
        }

        public static Operation valueOf(String name) {
            return Enum.valueOf(Operation.class, name);
        }

        String getKey() {
            return this.key;
        }

        private Operation(String key) {
            this.key = Objects.requireNonNull(key);
        }

        static Operation parse(String s) {
            for (Operation operation : Operation.values()) {
                if (!operation.key.equals(s)) continue;
                return operation;
            }
            throw new IllegalArgumentException(Strings.format((String)"invalid operation [%s] expected one of [%s]", (Object[])new Object[]{s, Strings.arrayToCommaDelimitedString((Object[])Operation.values())}));
        }

        boolean assertConsistentOperationName(MetricCollection metricCollection) {
            Predicate<String> expectedOperationPredicate;
            List operationNameMetrics = metricCollection.metricValues(CoreMetric.OPERATION_NAME);
            assert (operationNameMetrics.size() == 1) : operationNameMetrics;
            switch (this.ordinal()) {
                case 2: {
                    Predicate<String> predicate = IS_LIST_OPERATION;
                    break;
                }
                case 4: {
                    Predicate<String> predicate = IS_PUT_MULTIPART_OPERATION;
                    break;
                }
                case 6: {
                    Predicate<String> predicate = "AbortMultipartUpload"::equals;
                    break;
                }
                case 8: {
                    Predicate<String> predicate = IS_COPY_MULTIPART_OPERATION;
                    break;
                }
                default: {
                    Predicate<String> predicate = expectedOperationPredicate = this.key::equals;
                }
            }
            assert (expectedOperationPredicate.test((String)operationNameMetrics.get(0))) : String.valueOf((Object)this) + " vs " + String.valueOf(operationNameMetrics);
            return true;
        }

        private static /* synthetic */ Operation[] $values() {
            return new Operation[]{HEAD_OBJECT, GET_OBJECT, LIST_OBJECTS, PUT_OBJECT, PUT_MULTIPART_OBJECT, DELETE_OBJECTS, ABORT_MULTIPART_OBJECT, COPY_OBJECT, COPY_MULTIPART_OBJECT};
        }

        static {
            $VALUES = Operation.$values();
            IS_PUT_MULTIPART_OPERATION = Set.of("CreateMultipartUpload", "UploadPart", "CompleteMultipartUpload")::contains;
            IS_COPY_MULTIPART_OPERATION = Set.of("CreateMultipartUpload", "UploadPartCopy", "CompleteMultipartUpload")::contains;
            IS_LIST_OPERATION = Set.of("ListObjects", "ListObjectsV2", "ListMultipartUploads")::contains;
        }
    }

    private static class DeletionExceptions {
        Exception exception = null;
        private int count = 0;

        private DeletionExceptions() {
        }

        void useOrMaybeSuppress(Exception e) {
            if (this.count < 10) {
                this.exception = (Exception)ExceptionsHelper.useOrSuppress((Throwable)this.exception, (Throwable)e);
                ++this.count;
            }
        }
    }

    record StatsKey(Operation operation, OperationPurpose purpose) {
        @Override
        public String toString() {
            return this.purpose.getKey() + "_" + this.operation.getKey();
        }
    }

    class ElasticsearchS3MetricsCollector
    implements MetricPublisher {
        final LongAdder requests = new LongAdder();
        final LongAdder operations = new LongAdder();
        private final Operation operation;
        private final Map<String, Object> attributes;

        private ElasticsearchS3MetricsCollector(Operation operation, OperationPurpose purpose) {
            this.operation = operation;
            this.attributes = RepositoriesMetrics.createAttributesMap((RepositoryMetadata)S3BlobStore.this.repositoryMetadata, (OperationPurpose)purpose, (String)operation.getKey());
        }

        BlobStoreActionStats getEndpointStats() {
            return new BlobStoreActionStats(this.operations.sum(), this.requests.sum());
        }

        public void publish(MetricCollection metricCollection) {
            assert (this.operation.assertConsistentOperationName(metricCollection));
            boolean overallSuccess = false;
            for (Boolean successMetricValue : metricCollection.metricValues(CoreMetric.API_CALL_SUCCESSFUL)) {
                if (Boolean.TRUE.equals(successMetricValue)) {
                    overallSuccess = true;
                    continue;
                }
                overallSuccess = false;
                break;
            }
            long totalTimeNanoseconds = 0L;
            for (Duration durationMetricValue : metricCollection.metricValues(CoreMetric.API_CALL_DURATION)) {
                totalTimeNanoseconds += durationMetricValue.toNanos();
            }
            long requestCount = 0L;
            long responseCount = 0L;
            long awsErrorCount = 0L;
            long throttleCount = 0L;
            long http416ResponseCount = 0L;
            for (MetricCollection apiCallAttemptMetrics : metricCollection.children()) {
                List httpResponses;
                if (!"ApiCallAttempt".equals(apiCallAttemptMetrics.name())) continue;
                ++requestCount;
                List errorTypes = apiCallAttemptMetrics.metricValues(CoreMetric.ERROR_TYPE);
                if (errorTypes != null && errorTypes.size() > 0) {
                    ++awsErrorCount;
                    if (errorTypes.contains("Throttling")) {
                        ++throttleCount;
                    }
                }
                if ((httpResponses = apiCallAttemptMetrics.metricValues(HttpMetric.HTTP_STATUS_CODE)) == null || httpResponses.size() <= 0) continue;
                ++responseCount;
                if (!httpResponses.contains(RestStatus.REQUESTED_RANGE_NOT_SATISFIED.getStatus())) continue;
                ++http416ResponseCount;
            }
            this.requests.add(responseCount);
            S3BlobStore.this.s3RepositoriesMetrics.common().operationCounter().incrementBy(1L, this.attributes);
            this.operations.increment();
            if (!overallSuccess) {
                S3BlobStore.this.s3RepositoriesMetrics.common().unsuccessfulOperationCounter().incrementBy(1L, this.attributes);
            }
            S3BlobStore.this.s3RepositoriesMetrics.common().requestCounter().incrementBy(requestCount, this.attributes);
            if (awsErrorCount > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().exceptionCounter().incrementBy(awsErrorCount, this.attributes);
                S3BlobStore.this.s3RepositoriesMetrics.common().exceptionHistogram().record(awsErrorCount, this.attributes);
            }
            if (throttleCount > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().throttleCounter().incrementBy(throttleCount, this.attributes);
                S3BlobStore.this.s3RepositoriesMetrics.common().throttleHistogram().record(throttleCount, this.attributes);
            }
            if (http416ResponseCount > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().requestRangeNotSatisfiedExceptionCounter().incrementBy(http416ResponseCount, this.attributes);
            }
            if (totalTimeNanoseconds > 0L) {
                S3BlobStore.this.s3RepositoriesMetrics.common().httpRequestTimeInMillisHistogram().record(TimeUnit.NANOSECONDS.toMillis(totalTimeNanoseconds), this.attributes);
            }
        }

        public void close() {
        }
    }
}

