/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.datafeed.extractor.scroll;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionRequestBuilder;
import org.elasticsearch.action.search.ClearScrollRequest;
import org.elasticsearch.action.search.ClearScrollResponse;
import org.elasticsearch.action.search.SearchPhaseExecutionException;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollRequestBuilder;
import org.elasticsearch.action.search.TransportClearScrollAction;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.ElasticsearchClient;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.datafeed.SearchInterval;
import org.elasticsearch.xpack.ml.datafeed.DatafeedTimingStatsReporter;
import org.elasticsearch.xpack.ml.datafeed.extractor.DataExtractor;
import org.elasticsearch.xpack.ml.datafeed.extractor.DataExtractorUtils;
import org.elasticsearch.xpack.ml.datafeed.extractor.scroll.ScrollDataExtractorContext;
import org.elasticsearch.xpack.ml.datafeed.extractor.scroll.SearchHitToJsonProcessor;
import org.elasticsearch.xpack.ml.extractor.ExtractedField;
import org.elasticsearch.xpack.ml.extractor.SourceSupplier;

class ScrollDataExtractor
implements DataExtractor {
    private static final TimeValue SCROLL_TIMEOUT = new TimeValue(30L, TimeUnit.MINUTES);
    private static final Logger logger = LogManager.getLogger(ScrollDataExtractor.class);
    private final Client client;
    private final ScrollDataExtractorContext context;
    private final DatafeedTimingStatsReporter timingStatsReporter;
    private String scrollId;
    private boolean isCancelled;
    private boolean hasNext;
    private Long timestampOnCancel;
    protected Long lastTimestamp;
    private boolean searchHasShardFailure;

    ScrollDataExtractor(Client client, ScrollDataExtractorContext dataExtractorContext, DatafeedTimingStatsReporter timingStatsReporter) {
        this.client = Objects.requireNonNull(client);
        this.context = Objects.requireNonNull(dataExtractorContext);
        this.timingStatsReporter = Objects.requireNonNull(timingStatsReporter);
        this.hasNext = true;
        this.searchHasShardFailure = false;
    }

    @Override
    public boolean hasNext() {
        return this.hasNext;
    }

    @Override
    public boolean isCancelled() {
        return this.isCancelled;
    }

    @Override
    public void cancel() {
        logger.trace("[{}] Data extractor received cancel request", (Object)this.context.jobId);
        this.isCancelled = true;
    }

    @Override
    public void destroy() {
        this.cancel();
        this.clearScroll();
    }

    @Override
    public long getEndTime() {
        return this.context.queryContext.end;
    }

    @Override
    public DataExtractor.Result next() throws IOException {
        if (!this.hasNext()) {
            throw new NoSuchElementException();
        }
        Optional<InputStream> stream = this.tryNextStream();
        if (!stream.isPresent()) {
            this.hasNext = false;
        }
        return new DataExtractor.Result(new SearchInterval(this.context.queryContext.start, this.context.queryContext.end), stream);
    }

    private Optional<InputStream> tryNextStream() throws IOException {
        try {
            return this.scrollId == null ? Optional.ofNullable(this.initScroll(this.context.queryContext.start)) : Optional.ofNullable(this.continueScroll());
        }
        catch (Exception e) {
            this.scrollId = null;
            if (this.searchHasShardFailure) {
                throw e;
            }
            logger.debug("[{}] Resetting scroll search after shard failure", (Object)this.context.jobId);
            this.markScrollAsErrored();
            return Optional.ofNullable(this.initScroll(this.lastTimestamp == null ? this.context.queryContext.start : this.lastTimestamp));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected InputStream initScroll(long startTimestamp) throws IOException {
        logger.debug("[{}] Initializing scroll with start time [{}]", (Object)this.context.jobId, (Object)startTimestamp);
        SearchResponse searchResponse = this.executeSearchRequest((ActionRequestBuilder<?, SearchResponse>)this.buildSearchRequest(startTimestamp));
        try {
            logger.debug("[{}] Search response was obtained", (Object)this.context.jobId);
            this.timingStatsReporter.reportSearchDuration(searchResponse.getTook());
            this.scrollId = searchResponse.getScrollId();
            InputStream inputStream = this.processAndConsumeSearchHits(searchResponse.getHits());
            return inputStream;
        }
        finally {
            searchResponse.decRef();
        }
    }

    protected SearchResponse executeSearchRequest(ActionRequestBuilder<?, SearchResponse> searchRequestBuilder) {
        return this.checkForSkippedClusters((SearchResponse)ClientHelper.executeWithHeaders(this.context.queryContext.headers, (String)"ml", (Client)this.client, () -> searchRequestBuilder.get()));
    }

    private SearchResponse checkForSkippedClusters(SearchResponse searchResponse) {
        boolean success = false;
        try {
            DataExtractorUtils.checkForSkippedClusters(searchResponse);
            success = true;
        }
        catch (ResourceNotFoundException e) {
            this.clearScrollLoggingExceptions(searchResponse.getScrollId());
            throw e;
        }
        finally {
            if (!success) {
                searchResponse.decRef();
            }
        }
        return searchResponse;
    }

    private SearchRequestBuilder buildSearchRequest(long start) {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder().size(this.context.scrollSize).sort(this.context.extractedFields.timeField(), SortOrder.ASC).query(DataExtractorUtils.wrapInTimeRangeQuery(this.context.queryContext.query, this.context.extractedFields.timeField(), start, this.context.queryContext.end)).runtimeMappings(this.context.queryContext.runtimeMappings);
        SearchRequestBuilder searchRequestBuilder = new SearchRequestBuilder((ElasticsearchClient)this.client).setScroll(SCROLL_TIMEOUT).setIndices(this.context.queryContext.indices).setIndicesOptions(this.context.queryContext.indicesOptions).setAllowPartialSearchResults(false).setSource(searchSourceBuilder);
        for (ExtractedField docValueField : this.context.extractedFields.getDocValueFields()) {
            searchRequestBuilder.addDocValueField(docValueField.getSearchField(), docValueField.getDocValueFormat());
        }
        String[] sourceFields = this.context.extractedFields.getSourceFields();
        if (sourceFields.length == 0) {
            searchRequestBuilder.setFetchSource(false);
            searchRequestBuilder.storedFields(new String[]{"_none_"});
        } else {
            searchRequestBuilder.setFetchSource(sourceFields, null);
        }
        this.context.scriptFields.forEach(f -> searchRequestBuilder.addScriptField(f.fieldName(), f.script()));
        return searchRequestBuilder;
    }

    private InputStream processAndConsumeSearchHits(SearchHits hits) throws IOException {
        if (hits.getHits().length == 0) {
            this.hasNext = false;
            this.clearScroll();
            return null;
        }
        BytesStreamOutput outputStream = new BytesStreamOutput();
        SearchHit lastHit = hits.getAt(hits.getHits().length - 1);
        this.lastTimestamp = this.context.extractedFields.timeFieldValue(lastHit, new SourceSupplier(lastHit));
        try (SearchHitToJsonProcessor hitProcessor = new SearchHitToJsonProcessor(this.context.extractedFields, (OutputStream)outputStream);){
            for (SearchHit hit : hits) {
                Long timestamp;
                SourceSupplier sourceSupplier = new SourceSupplier(hit);
                if (this.isCancelled && (timestamp = this.context.extractedFields.timeFieldValue(hit, sourceSupplier)) != null) {
                    if (this.timestampOnCancel == null) {
                        this.timestampOnCancel = timestamp;
                    } else if (!timestamp.equals(this.timestampOnCancel)) {
                        this.hasNext = false;
                        this.clearScroll();
                        break;
                    }
                }
                hitProcessor.process(hit, sourceSupplier);
            }
        }
        return outputStream.bytes().streamInput();
    }

    private InputStream continueScroll() throws IOException {
        logger.debug("[{}] Continuing scroll with id [{}]", (Object)this.context.jobId, (Object)this.scrollId);
        SearchResponse searchResponse = null;
        try {
            try {
                searchResponse = this.executeSearchScrollRequest(this.scrollId);
            }
            catch (SearchPhaseExecutionException searchExecutionException) {
                if (this.searchHasShardFailure) {
                    throw searchExecutionException;
                }
                logger.debug("[{}] search failed due to SearchPhaseExecutionException. Will attempt again with new scroll", (Object)this.context.jobId);
                this.markScrollAsErrored();
                searchResponse = this.executeSearchRequest((ActionRequestBuilder<?, SearchResponse>)this.buildSearchRequest(this.lastTimestamp == null ? this.context.queryContext.start : this.lastTimestamp));
            }
            logger.debug("[{}] Search response was obtained", (Object)this.context.jobId);
            this.timingStatsReporter.reportSearchDuration(searchResponse.getTook());
            this.scrollId = searchResponse.getScrollId();
            InputStream inputStream = this.processAndConsumeSearchHits(searchResponse.getHits());
            return inputStream;
        }
        finally {
            if (searchResponse != null) {
                searchResponse.decRef();
            }
        }
    }

    void markScrollAsErrored() {
        this.scrollId = null;
        if (this.lastTimestamp != null) {
            Long l = this.lastTimestamp;
            this.lastTimestamp = this.lastTimestamp + 1L;
        }
        this.searchHasShardFailure = true;
    }

    protected SearchResponse executeSearchScrollRequest(String scrollId) {
        return this.executeSearchRequest((ActionRequestBuilder<?, SearchResponse>)new SearchScrollRequestBuilder((ElasticsearchClient)this.client).setScroll(SCROLL_TIMEOUT).setScrollId(scrollId));
    }

    private void clearScroll() {
        this.innerClearScroll(this.scrollId);
        this.scrollId = null;
    }

    private void clearScrollLoggingExceptions(String scrollId) {
        try {
            this.innerClearScroll(scrollId);
        }
        catch (Exception e) {
            logger.error(() -> "[" + this.context.jobId + "] Failed to clear scroll", (Throwable)e);
        }
    }

    private void innerClearScroll(String scrollId) {
        if (scrollId != null) {
            ClearScrollRequest request = new ClearScrollRequest();
            request.addScrollId(scrollId);
            ClientHelper.executeWithHeaders(this.context.queryContext.headers, (String)"ml", (Client)this.client, () -> (ClearScrollResponse)this.client.execute(TransportClearScrollAction.TYPE, (ActionRequest)request).actionGet());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public DataExtractor.DataSummary getSummary() {
        SearchRequestBuilder searchRequestBuilder = DataExtractorUtils.getSearchRequestBuilderForSummary(this.client, this.context.queryContext);
        SearchResponse searchResponse = this.executeSearchRequest((ActionRequestBuilder<?, SearchResponse>)searchRequestBuilder);
        try {
            logger.debug("[{}] Scrolling Data summary response was obtained", (Object)this.context.jobId);
            this.timingStatsReporter.reportSearchDuration(searchResponse.getTook());
            DataExtractor.DataSummary dataSummary = DataExtractorUtils.getDataSummary(searchResponse);
            return dataSummary;
        }
        finally {
            searchResponse.decRef();
        }
    }
}

