/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.snapshots;

import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardSnapshotStatus;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class SnapshotShutdownProgressTracker {
    public static final Setting<TimeValue> SNAPSHOT_PROGRESS_DURING_SHUTDOWN_LOG_INTERVAL_SETTING = Setting.timeSetting("snapshots.shutdown.progress.interval", TimeValue.timeValueSeconds((long)5L), TimeValue.MINUS_ONE, Setting.Property.NodeScope, Setting.Property.Dynamic);
    private static final Logger logger = LogManager.getLogger(SnapshotShutdownProgressTracker.class);
    private static final DateFormatter DATE_TIME_FORMATTER = DateFormatter.forPattern("strict_date_optional_time");
    private final Supplier<String> getLocalNodeId;
    private final Consumer<Logger> logIndexShardSnapshotStatuses;
    private final ThreadPool threadPool;
    private volatile TimeValue progressLoggerInterval;
    private Scheduler.Cancellable scheduledProgressLoggerFuture;
    private volatile long shutdownStartMillis = -1L;
    private volatile long shutdownFinishedSignallingPausingMillis = -1L;
    private final AtomicLong numberOfShardSnapshotsInProgressOnDataNode = new AtomicLong();
    private final Map<String, Long> shardSnapshotRequests = ConcurrentCollections.newConcurrentMap();
    private final AtomicLong doneCount = new AtomicLong();
    private final AtomicLong failureCount = new AtomicLong();
    private final AtomicLong abortedCount = new AtomicLong();
    private final AtomicLong pausedCount = new AtomicLong();

    public SnapshotShutdownProgressTracker(Supplier<String> localNodeIdSupplier, Consumer<Logger> logShardStatuses, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.getLocalNodeId = localNodeIdSupplier;
        this.logIndexShardSnapshotStatuses = logShardStatuses;
        clusterSettings.initializeAndWatch(SNAPSHOT_PROGRESS_DURING_SHUTDOWN_LOG_INTERVAL_SETTING, value -> {
            this.progressLoggerInterval = value;
        });
        this.threadPool = threadPool;
    }

    private void scheduleProgressLogger() {
        if (this.progressLoggerInterval.millis() > 0L) {
            this.scheduledProgressLoggerFuture = this.threadPool.scheduleWithFixedDelay(this::logProgressReport, this.progressLoggerInterval, this.threadPool.executor("generic"));
            logger.debug(() -> Strings.format("Starting shutdown snapshot progress logging on node [%s], runs every [%s]", this.getLocalNodeId.get(), this.progressLoggerInterval));
        } else {
            logger.debug("Snapshot progress logging during shutdown is disabled");
        }
    }

    private void cancelProgressLogger() {
        assert (this.scheduledProgressLoggerFuture != null) : "Somehow shutdown mode was removed before it was added.";
        this.scheduledProgressLoggerFuture.cancel();
        if (this.progressLoggerInterval.millis() > 0L) {
            logger.debug(() -> Strings.format("Cancelling shutdown snapshot progress logging on node [%s]", this.getLocalNodeId.get()));
        }
    }

    private void logProgressReport() {
        logger.info("Current active shard snapshot stats on data node [{}]. Node shutdown cluster state update received at [{} UTC]. Finished signalling shard snapshots to pause at [{} UTC]. Time between the node shutdown cluster state update and signalling shard snapshots to pause is [{} millis]. Number shard snapshots running [{}]. Number shard snapshots waiting for master node reply to status update request [{}] Shard snapshot completion stats since shutdown began: Done [{}]; Failed [{}]; Aborted [{}]; Paused [{}]", (Object)this.getLocalNodeId.get(), (Object)DATE_TIME_FORMATTER.formatMillis(this.shutdownStartMillis), (Object)DATE_TIME_FORMATTER.formatMillis(this.shutdownFinishedSignallingPausingMillis), (Object)(this.shutdownFinishedSignallingPausingMillis - this.shutdownStartMillis), (Object)this.numberOfShardSnapshotsInProgressOnDataNode.get(), (Object)this.shardSnapshotRequests.size(), (Object)this.doneCount.get(), (Object)this.failureCount.get(), (Object)this.abortedCount.get(), (Object)this.pausedCount.get());
        this.logIndexShardSnapshotStatuses.accept(logger);
    }

    public void onClusterStateAddShutdown() {
        assert (this.shutdownStartMillis == -1L) : "Expected not to be tracking anything. Call shutdown remove before adding shutdown again";
        this.doneCount.set(0L);
        this.failureCount.set(0L);
        this.abortedCount.set(0L);
        this.pausedCount.set(0L);
        this.shutdownStartMillis = this.threadPool.relativeTimeInMillis();
        this.scheduleProgressLogger();
    }

    public void onClusterStatePausingSetForAllShardSnapshots() {
        assert (this.shutdownStartMillis != -1L) : "Should not have left shutdown mode before finishing processing the cluster state update with shutdown";
        this.shutdownFinishedSignallingPausingMillis = this.threadPool.relativeTimeInMillis();
        logger.debug(() -> Strings.format("Pause signals have been set for all shard snapshots on data node [%s]", this.getLocalNodeId.get()));
    }

    public void onClusterStateRemoveShutdown() {
        assert (this.shutdownStartMillis != -1L) : "Expected a call to add shutdown mode before a call to remove shutdown mode.";
        this.shutdownStartMillis = -1L;
        this.shutdownFinishedSignallingPausingMillis = -1L;
        this.cancelProgressLogger();
    }

    public void incNumberOfShardSnapshotsInProgress(ShardId shardId, Snapshot snapshot) {
        logger.debug(() -> Strings.format("Started shard (shard ID: [%s]) in snapshot ([%s])", shardId, snapshot));
        this.numberOfShardSnapshotsInProgressOnDataNode.incrementAndGet();
    }

    public void decNumberOfShardSnapshotsInProgress(ShardId shardId, Snapshot snapshot, IndexShardSnapshotStatus shardSnapshotStatus) {
        logger.debug(() -> Strings.format("Finished shard (shard ID: [%s]) in snapshot ([%s]) with status ([%s]): ", shardId, snapshot, shardSnapshotStatus.toString()));
        this.numberOfShardSnapshotsInProgressOnDataNode.decrementAndGet();
        if (this.shutdownStartMillis != -1L) {
            switch (shardSnapshotStatus.getStage()) {
                case DONE: {
                    this.doneCount.incrementAndGet();
                    break;
                }
                case FAILURE: {
                    this.failureCount.incrementAndGet();
                    break;
                }
                case ABORTED: {
                    this.abortedCount.incrementAndGet();
                    break;
                }
                case PAUSED: {
                    this.pausedCount.incrementAndGet();
                    break;
                }
                default: {
                    assert (false) : "unexpected shard snapshot stage transition during shutdown: " + String.valueOf((Object)shardSnapshotStatus.getStage());
                    break;
                }
            }
        }
    }

    public void trackRequestSentToMaster(Snapshot snapshot, ShardId shardId) {
        logger.debug(() -> Strings.format("Tracking shard (shard ID: [%s]) snapshot ([%s]) request to master", shardId, snapshot));
        this.shardSnapshotRequests.put(snapshot.toString() + shardId.getIndexName() + shardId.getId(), this.threadPool.relativeTimeInNanos());
    }

    public void releaseRequestSentToMaster(Snapshot snapshot, ShardId shardId) {
        Long masterRequestStartTime = this.shardSnapshotRequests.remove(snapshot.toString() + shardId.getIndexName() + shardId.getId());
        if (masterRequestStartTime != null) {
            logger.debug(() -> Strings.format("Finished shard (shard ID: [%s]) snapshot ([%s]) update request to master in [%s]", shardId, snapshot, new TimeValue(this.threadPool.relativeTimeInNanos() - masterRequestStartTime, TimeUnit.NANOSECONDS)));
        }
    }

    void assertStatsForTesting(long done, long failure, long aborted, long paused) {
        assert (this.doneCount.get() == done) : "doneCount is " + this.doneCount.get() + ", expected count was " + done;
        assert (this.failureCount.get() == failure) : "failureCount is " + this.doneCount.get() + ", expected count was " + failure;
        assert (this.abortedCount.get() == aborted) : "abortedCount is " + this.doneCount.get() + ", expected count was " + aborted;
        assert (this.pausedCount.get() == paused) : "pausedCount is " + this.doneCount.get() + ", expected count was " + paused;
    }
}

