/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchWrapperException;
import org.elasticsearch.action.NoShardAvailableActionException;
import org.elasticsearch.action.ShardOperationFailedException;
import org.elasticsearch.action.UnavailableShardsException;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.discovery.MasterNotDiscoveredException;
import org.elasticsearch.index.Index;
import org.elasticsearch.node.NodeClosedException;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.tasks.TaskCancelledException;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.NoSeedNodeLeftException;
import org.elasticsearch.transport.NoSuchRemoteClusterException;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.NodeNotConnectedException;
import org.elasticsearch.xcontent.XContentParseException;

public final class ExceptionsHelper {
    private static final Logger logger = LogManager.getLogger(ExceptionsHelper.class);
    private static final String CAUSE_CAPTION = "Caused by: ";
    private static final String SUPPRESSED_CAPTION = "Suppressed: ";
    private static final List<Class<? extends IOException>> CORRUPTION_EXCEPTIONS = List.of(CorruptIndexException.class, IndexFormatTooOldException.class, IndexFormatTooNewException.class);

    public static RuntimeException convertToRuntime(Exception e) {
        if (e instanceof RuntimeException) {
            return (RuntimeException)e;
        }
        return new ElasticsearchException(e);
    }

    public static ElasticsearchException convertToElastic(Exception e) {
        if (e instanceof ElasticsearchException) {
            return (ElasticsearchException)e;
        }
        return new ElasticsearchException(e);
    }

    public static RestStatus status(Throwable t) {
        if (t != null) {
            if (t instanceof ElasticsearchException) {
                return ((ElasticsearchException)t).status();
            }
            if (t instanceof IllegalArgumentException) {
                return RestStatus.BAD_REQUEST;
            }
            if (t instanceof XContentParseException) {
                return RestStatus.BAD_REQUEST;
            }
            if (t instanceof EsRejectedExecutionException) {
                return RestStatus.TOO_MANY_REQUESTS;
            }
        }
        return RestStatus.INTERNAL_SERVER_ERROR;
    }

    public static Throwable unwrapCause(Throwable t) {
        int counter = 0;
        Throwable result = t;
        while (result instanceof ElasticsearchWrapperException) {
            if (result.getCause() == null) {
                return result;
            }
            if (result.getCause() == result) {
                return result;
            }
            if (counter++ > 10) {
                logger.warn("Exception cause unwrapping ran for 10 levels...", t);
                return result;
            }
            result = result.getCause();
        }
        return result;
    }

    public static String stackTrace(Throwable e) {
        StringWriter stackTraceStringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stackTraceStringWriter);
        e.printStackTrace(printWriter);
        return stackTraceStringWriter.toString();
    }

    public static String limitedStackTrace(Throwable e, int traceDepth) {
        assert (traceDepth >= 0) : "Cannot print stacktraces with negative trace depths";
        StringWriter stackTraceStringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stackTraceStringWriter);
        ExceptionsHelper.printLimitedStackTrace(e, printWriter, traceDepth);
        return stackTraceStringWriter.toString();
    }

    private static void printLimitedStackTrace(Throwable e, PrintWriter s, int maxLines) {
        Set<Throwable> dejaVu = Collections.newSetFromMap(new IdentityHashMap());
        dejaVu.add(e);
        s.println(ExceptionsHelper.compressExceptionMessage(e));
        StackTraceElement[] trace = e.getStackTrace();
        int linesPrinted = 0;
        for (StackTraceElement stackTraceElement : trace) {
            if (linesPrinted >= maxLines) break;
            s.println(ExceptionsHelper.compressStackTraceElement(new StringBuilder("\tat "), stackTraceElement));
            ++linesPrinted;
        }
        if (trace.length > linesPrinted) {
            s.println("\t... " + (trace.length - linesPrinted) + " more");
        }
        for (Serializable serializable : e.getSuppressed()) {
            ExceptionsHelper.limitAndPrintEnclosedStackTrace((Throwable)serializable, s, trace, SUPPRESSED_CAPTION, "\t", maxLines, dejaVu);
        }
        Throwable ourCause = e.getCause();
        if (ourCause != null) {
            ExceptionsHelper.limitAndPrintEnclosedStackTrace(ourCause, s, trace, CAUSE_CAPTION, "", maxLines, dejaVu);
        }
    }

    private static void limitAndPrintEnclosedStackTrace(Throwable e, PrintWriter s, StackTraceElement[] enclosingTrace, String caption, String prefix, int maxLines, Set<Throwable> dejaVu) {
        if (dejaVu.contains(e)) {
            s.println(prefix + caption + "[CIRCULAR REFERENCE: " + ExceptionsHelper.compressExceptionMessage(e) + "]");
        } else {
            dejaVu.add(e);
            StackTraceElement[] trace = e.getStackTrace();
            int m = trace.length - 1;
            for (int n = enclosingTrace.length - 1; m >= 0 && n >= 0 && trace[m].equals(enclosingTrace[n]); --m, --n) {
            }
            int framesInCommon = trace.length - 1 - m;
            int linesToPrint = m + 1;
            if (linesToPrint > maxLines) {
                m = maxLines - 1;
                framesInCommon = trace.length - 1 - m;
            }
            s.println(prefix + caption + ExceptionsHelper.compressExceptionMessage(e));
            for (int i = 0; i <= m; ++i) {
                s.println(ExceptionsHelper.compressStackTraceElement(new StringBuilder(prefix).append("\tat "), trace[i]));
            }
            if (framesInCommon != 0) {
                s.println(prefix + "\t... " + framesInCommon + " more");
            }
            for (Throwable se : e.getSuppressed()) {
                ExceptionsHelper.limitAndPrintEnclosedStackTrace(se, s, trace, SUPPRESSED_CAPTION, prefix + "\t", maxLines, dejaVu);
            }
            Throwable ourCause = e.getCause();
            if (ourCause != null) {
                ExceptionsHelper.limitAndPrintEnclosedStackTrace(ourCause, s, trace, CAUSE_CAPTION, prefix, maxLines, dejaVu);
            }
        }
    }

    private static String compressExceptionMessage(Throwable e) {
        StringBuilder msg = new StringBuilder();
        ExceptionsHelper.compressPackages(msg, e.getClass().getName());
        String message = e.getLocalizedMessage();
        if (message != null) {
            msg.append(": ").append(message);
        }
        return msg.toString();
    }

    private static StringBuilder compressStackTraceElement(StringBuilder s, StackTraceElement stackTraceElement) {
        String declaringClass = stackTraceElement.getClassName();
        ExceptionsHelper.compressPackages(s, declaringClass);
        String methodName = stackTraceElement.getMethodName();
        s.append(".").append(methodName).append("(");
        if (stackTraceElement.isNativeMethod()) {
            s.append("Native Method)");
        } else {
            String fileName = stackTraceElement.getFileName();
            int lineNumber = stackTraceElement.getLineNumber();
            if (fileName != null && lineNumber >= 0) {
                s.append(fileName).append(":").append(lineNumber).append(")");
            } else if (fileName != null) {
                s.append(fileName).append(")");
            } else {
                s.append("Unknown Source)");
            }
        }
        return s;
    }

    static void compressPackages(StringBuilder s, String className) {
        assert (s != null) : "s cannot be null";
        assert (className != null) : "className cannot be null";
        int finalDot = className.lastIndexOf(46);
        if (finalDot < 0) {
            s.append(className);
            return;
        }
        int lastPackageName = className.lastIndexOf(46, finalDot - 1);
        if (lastPackageName < 0) {
            if (finalDot >= 1) {
                s.append(className.charAt(0)).append('.');
            }
            s.append(className.substring(finalDot + 1));
            return;
        }
        boolean firstChar = true;
        char[] charArray = className.toCharArray();
        for (int idx = 0; idx <= lastPackageName + 1; ++idx) {
            char c = charArray[idx];
            if (firstChar && '.' != c) {
                s.append(c).append('.');
            }
            firstChar = '.' == c;
        }
        s.append(className.substring(finalDot + 1));
    }

    public static String formatStackTrace(StackTraceElement[] stackTrace) {
        return Arrays.stream(stackTrace).skip(1L).map(e -> "\tat " + String.valueOf(e)).collect(Collectors.joining("\n"));
    }

    public static <T extends Throwable> void rethrowAndSuppress(List<T> exceptions) throws T {
        Throwable main = null;
        for (Throwable ex : exceptions) {
            main = ExceptionsHelper.useOrSuppress(main, ex);
        }
        if (main != null) {
            throw main;
        }
    }

    public static <T extends Throwable> void maybeThrowRuntimeAndSuppress(List<T> exceptions) {
        Throwable main = null;
        for (Throwable ex : exceptions) {
            main = ExceptionsHelper.useOrSuppress(main, ex);
        }
        if (main != null) {
            throw new ElasticsearchException(main);
        }
    }

    public static <T extends Throwable> T useOrSuppress(T first, T second) {
        if (first == null) {
            return second;
        }
        first.addSuppressed(second);
        return first;
    }

    public static IOException unwrapCorruption(Throwable t) {
        return t == null ? null : (IOException)ExceptionsHelper.unwrapCausesAndSuppressed(t, cause -> {
            for (Class<? extends IOException> clazz : CORRUPTION_EXCEPTIONS) {
                if (!clazz.isInstance(cause)) continue;
                return true;
            }
            return false;
        }).orElse(null);
    }

    public static Throwable unwrap(Throwable t, Class<?> ... clazzes) {
        if (t != null) {
            Set seen = Collections.newSetFromMap(new IdentityHashMap());
            do {
                if (!seen.add(t)) {
                    return null;
                }
                for (Class<?> clazz : clazzes) {
                    if (!clazz.isInstance(t)) continue;
                    return t;
                }
            } while ((t = t.getCause()) != null);
        }
        return null;
    }

    public static boolean reThrowIfNotNull(@Nullable Throwable e) {
        if (e != null) {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException(e);
        }
        return true;
    }

    public static <T extends Throwable> Optional<T> unwrapCausesAndSuppressed(Throwable cause, Predicate<Throwable> predicate) {
        if (predicate.test(cause)) {
            return Optional.of(cause);
        }
        LinkedList<Throwable> queue = new LinkedList<Throwable>();
        queue.add(cause);
        Set seen = Collections.newSetFromMap(new IdentityHashMap());
        while (!queue.isEmpty()) {
            Throwable current = (Throwable)queue.remove();
            if (!seen.add(current)) continue;
            if (predicate.test(current)) {
                return Optional.of(current);
            }
            Collections.addAll(queue, current.getSuppressed());
            if (current.getCause() == null) continue;
            queue.add(current.getCause());
        }
        return Optional.empty();
    }

    public static Optional<Error> maybeError(Throwable cause) {
        return ExceptionsHelper.unwrapCausesAndSuppressed(cause, t -> t instanceof Error);
    }

    public static void maybeDieOnAnotherThread(Throwable throwable) {
        ExceptionsHelper.maybeError(throwable).ifPresent(error -> {
            try {
                String formatted = ExceptionsHelper.formatStackTrace(Thread.currentThread().getStackTrace());
                logger.error("fatal error {}: {}\n{}", (Object)error.getClass().getCanonicalName(), (Object)error.getMessage(), (Object)formatted);
            }
            finally {
                new Thread(() -> {
                    throw error;
                }, "elasticsearch-error-rethrower").start();
            }
        });
    }

    public static ShardOperationFailedException[] groupBy(ShardOperationFailedException[] failures) {
        if (failures == null || failures.length == 0) {
            return failures;
        }
        ArrayList<ShardOperationFailedException> uniqueFailures = new ArrayList<ShardOperationFailedException>();
        HashSet<GroupBy> reasons = new HashSet<GroupBy>();
        for (ShardOperationFailedException failure : failures) {
            GroupBy reason = new GroupBy(failure);
            if (reasons.contains(reason)) continue;
            reasons.add(reason);
            uniqueFailures.add(failure);
        }
        return uniqueFailures.toArray(new ShardOperationFailedException[0]);
    }

    public static boolean isNodeOrShardUnavailableTypeException(Throwable t) {
        return t instanceof NoShardAvailableActionException || t instanceof UnavailableShardsException || t instanceof NodeClosedException || t instanceof NodeDisconnectedException || t instanceof MasterNotDiscoveredException || t instanceof NodeNotConnectedException || t instanceof ClusterBlockException;
    }

    public static boolean isRemoteUnavailableException(Exception e) {
        Throwable unwrap = ExceptionsHelper.unwrap(e, ConnectTransportException.class, NoSuchRemoteClusterException.class, NoSeedNodeLeftException.class);
        if (unwrap != null) {
            return true;
        }
        Throwable ill = ExceptionsHelper.unwrap(e, IllegalStateException.class, IllegalArgumentException.class);
        return ill != null && (ill.getMessage().contains("Unable to open any connections") || ill.getMessage().contains("unknown host"));
    }

    public static boolean isTaskCancelledException(Exception e) {
        return ExceptionsHelper.unwrapCausesAndSuppressed(e, ex -> ex instanceof TaskCancelledException).isPresent();
    }

    private static class GroupBy {
        final String reason;
        final String index;
        final Class<? extends Throwable> causeType;

        GroupBy(ShardOperationFailedException failure) {
            Index index;
            Throwable cause = failure.getCause();
            String indexName = failure.index();
            if (indexName == null && cause instanceof ElasticsearchException && (index = ((ElasticsearchException)cause).getIndex()) != null) {
                indexName = index.getName();
            }
            this.index = indexName;
            this.reason = cause.getMessage();
            this.causeType = cause.getClass();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            GroupBy groupBy = (GroupBy)o;
            return Objects.equals(this.reason, groupBy.reason) && Objects.equals(this.index, groupBy.index) && Objects.equals(this.causeType, groupBy.causeType);
        }

        public int hashCode() {
            return Objects.hash(this.reason, this.index, this.causeType);
        }
    }
}

