/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch.action.admin.cluster.node.reload;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.FailedNodeException;
import org.elasticsearch.action.support.ActionFilters;
import org.elasticsearch.action.support.nodes.TransportNodesAction;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.settings.KeyStoreWrapper;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.injection.guice.Inject;
import org.elasticsearch.plugins.PluginsService;
import org.elasticsearch.plugins.ReloadablePlugin;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;

public class TransportNodesReloadSecureSettingsAction extends TransportNodesAction<
    NodesReloadSecureSettingsRequest,
    NodesReloadSecureSettingsResponse,
    NodesReloadSecureSettingsRequest.NodeRequest,
    NodesReloadSecureSettingsResponse.NodeResponse,
    Void> {

    public static final ActionType<NodesReloadSecureSettingsResponse> TYPE = new ActionType<>("cluster:admin/nodes/reload_secure_settings");

    private static final Logger logger = LogManager.getLogger(TransportNodesReloadSecureSettingsAction.class);

    private final Environment environment;
    private final PluginsService pluginsService;

    @Inject
    public TransportNodesReloadSecureSettingsAction(
        ThreadPool threadPool,
        ClusterService clusterService,
        TransportService transportService,
        ActionFilters actionFilters,
        Environment environment,
        PluginsService pluginService
    ) {
        super(
            TYPE.name(),
            clusterService,
            transportService,
            actionFilters,
            NodesReloadSecureSettingsRequest.NodeRequest::new,
            threadPool.executor(ThreadPool.Names.GENERIC)
        );
        this.environment = environment;
        this.pluginsService = pluginService;
    }

    @Override
    protected NodesReloadSecureSettingsResponse newResponse(
        NodesReloadSecureSettingsRequest request,
        List<NodesReloadSecureSettingsResponse.NodeResponse> responses,
        List<FailedNodeException> failures
    ) {
        return new NodesReloadSecureSettingsResponse(clusterService.getClusterName(), responses, failures);
    }

    @Override
    protected NodesReloadSecureSettingsRequest.NodeRequest newNodeRequest(NodesReloadSecureSettingsRequest request) {
        return request.newNodeRequest();
    }

    @Override
    protected NodesReloadSecureSettingsResponse.NodeResponse newNodeResponse(StreamInput in, DiscoveryNode node) throws IOException {
        return new NodesReloadSecureSettingsResponse.NodeResponse(in);
    }

    @Override
    protected DiscoveryNode[] resolveRequest(NodesReloadSecureSettingsRequest request, ClusterState clusterState) {
        final var concreteNodes = super.resolveRequest(request, clusterState);
        final var isNodeLocal = concreteNodes.length == 1 && concreteNodes[0].getId().equals(clusterState.nodes().getLocalNodeId());
        if (request.hasPassword() && isNodeLocal == false && isNodeTransportTLSEnabled() == false) {
            throw new ElasticsearchException("""
                Secure settings cannot be updated cluster wide when TLS for the transport layer is not enabled. Enable TLS or use the API \
                with a `_local` filter on each node.""");
        } else {
            return concreteNodes;
        }
    }

    @Override
    protected NodesReloadSecureSettingsResponse.NodeResponse nodeOperation(
        NodesReloadSecureSettingsRequest.NodeRequest request,
        Task task
    ) {
        // We default to using an empty string as the keystore password so that we mimic pre 7.3 API behavior
        try (KeyStoreWrapper keystore = KeyStoreWrapper.load(environment.configDir())) {
            // reread keystore from config file
            if (keystore == null) {
                return new NodesReloadSecureSettingsResponse.NodeResponse(
                    clusterService.localNode(),
                    new IllegalStateException("Keystore is missing"),
                    null,
                    null,
                    null,
                    null
                );
            }
            // decrypt the keystore using the password from the request
            keystore.decrypt(request.hasPassword() ? request.getSecureSettingsPassword().getChars() : new char[0]);
            // add the keystore to the original node settings object
            final Settings settingsWithKeystore = Settings.builder().put(environment.settings(), false).setSecureSettings(keystore).build();
            clusterService.getClusterSettings().validate(settingsWithKeystore, true);

            final List<Exception> exceptions = new ArrayList<>();
            // broadcast the new settings object (with the open embedded keystore) to all reloadable plugins
            pluginsService.filterPlugins(ReloadablePlugin.class).forEach(p -> {
                logger.debug("Reloading plugin [" + p.getClass().getSimpleName() + "]");
                try {
                    p.reload(settingsWithKeystore);
                } catch (final Exception e) {
                    logger.warn(() -> "Reload failed for plugin [" + p.getClass().getSimpleName() + "]", e);
                    exceptions.add(e);
                }
            });
            ExceptionsHelper.rethrowAndSuppress(exceptions);
            Path keystorePath = KeyStoreWrapper.keystorePath(environment.configDir());
            return new NodesReloadSecureSettingsResponse.NodeResponse(
                clusterService.localNode(),
                null,
                keystore.getSettingNames().toArray(String[]::new),
                keystorePath.toString(),
                failsafeSha256Digest(keystorePath),
                failsafeLastModifiedTime(keystorePath)
            );
        } catch (final Exception e) {
            return new NodesReloadSecureSettingsResponse.NodeResponse(clusterService.localNode(), e, null, null, null, null);
        }
    }

    private static Long failsafeLastModifiedTime(Path path) {
        try {
            return Files.readAttributes(path, BasicFileAttributes.class).lastModifiedTime().toMillis();
        } catch (IOException e) {
            logger.warn("Failed to read last modified time of [" + path + "]", e);
            return null;
        }
    }

    private static String failsafeSha256Digest(Path path) {
        try {
            return MessageDigests.toHexString(MessageDigests.sha256().digest(Files.readAllBytes(path)));
        } catch (IOException e) {
            logger.warn("Failed to compute SHA-256 digest of [" + path + "]", e);
            return null;
        }
    }

    /**
     * Returns true if the node is configured for TLS on the transport layer
     */
    private boolean isNodeTransportTLSEnabled() {
        return transportService.isTransportSecure();
    }
}
