/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch;

import org.elasticsearch.core.Assertions;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

/**
 * Legacy container for transport version constants.
 *
 *   ____   ___    _   _  ___ _____   _____ ____ ___ _____   _____ _   _ ___ ____    _____ ___ _     _____
 *  |  _ \ / _ \  | \ | |/ _ \_   _| | ____|  _ \_ _|_   _| |_   _| | | |_ _/ ___|  |  ___|_ _| |   | ____|
 *  | | | | | | | |  \| | | | || |   |  _| | | | | |  | |     | | | |_| || |\___ \  | |_   | || |   |  _|
 *  | |_| | |_| | | |\  | |_| || |   | |___| |_| | |  | |     | | |  _  || | ___) | |  _|  | || |___| |___
 *  |____/ \___/  |_| \_|\___/ |_|   |_____|____/___| |_|     |_| |_| |_|___|____/  |_|   |___|_____|_____|
 *
 * DO NOT EDIT THIS FILE! New transport versions should be added using the new transport version system.
 * See docs/internal/Versioning.md.
 */
public class TransportVersions {

    /*
     * NOTE: IntelliJ lies!
     * This map is used during class construction, referenced by the registerTransportVersion method.
     * When all the transport version constants have been registered, the map is cleared & never touched again.
     */
    static TreeSet<Integer> IDS = new TreeSet<>();

    static TransportVersion def(int id) {
        if (IDS == null) throw new IllegalStateException("The IDS map needs to be present to call this method");

        if (IDS.add(id) == false) {
            throw new IllegalArgumentException("Version id " + id + " defined twice");
        }
        if (id < IDS.last()) {
            throw new IllegalArgumentException("Version id " + id + " is not defined in the right location. Keep constants sorted");
        }
        return new TransportVersion(id);
    }

    // TODO: ES-10337 we can remove all transport versions earlier than 8.18
    /*
     * READ THE COMMENT BELOW THIS BLOCK OF DECLARATIONS BEFORE ADDING NEW TRANSPORT VERSIONS
     * Detached transport versions added below here.
     */
    public static final TransportVersion V_8_11_X = def(8_512_0_01);
    public static final TransportVersion V_8_12_0 = def(8_560_0_00);
    public static final TransportVersion V_8_12_1 = def(8_560_0_01);
    public static final TransportVersion V_8_13_0 = def(8_595_0_00);
    public static final TransportVersion V_8_13_4 = def(8_595_0_01);
    public static final TransportVersion V_8_14_0 = def(8_636_0_01);
    public static final TransportVersion V_8_15_0 = def(8_702_0_02);
    public static final TransportVersion V_8_15_2 = def(8_702_0_03);
    public static final TransportVersion V_8_16_0 = def(8_772_0_01);
    public static final TransportVersion V_8_16_1 = def(8_772_0_04);
    public static final TransportVersion INITIAL_ELASTICSEARCH_8_16_5 = def(8_772_0_05);
    public static final TransportVersion INITIAL_ELASTICSEARCH_8_16_6 = def(8_772_0_06);
    public static final TransportVersion V_8_17_0 = def(8_797_0_02);
    public static final TransportVersion INITIAL_ELASTICSEARCH_8_17_3 = def(8_797_0_03);
    public static final TransportVersion INITIAL_ELASTICSEARCH_8_17_4 = def(8_797_0_04);
    public static final TransportVersion INITIAL_ELASTICSEARCH_8_17_5 = def(8_797_0_05);

    /*
     * STOP! READ THIS FIRST! No, really,
     *        ____ _____ ___  ____  _        ____  _____    _    ____    _____ _   _ ___ ____    _____ ___ ____  ____ _____ _
     *       / ___|_   _/ _ \|  _ \| |      |  _ \| ____|  / \  |  _ \  |_   _| | | |_ _/ ___|  |  ___|_ _|  _ \/ ___|_   _| |
     *       \___ \ | || | | | |_) | |      | |_) |  _|   / _ \ | | | |   | | | |_| || |\___ \  | |_   | || |_) \___ \ | | | |
     *        ___) || || |_| |  __/|_|      |  _ <| |___ / ___ \| |_| |   | | |  _  || | ___) | |  _|  | ||  _ < ___) || | |_|
     *       |____/ |_| \___/|_|   (_)      |_| \_\_____/_/   \_\____/    |_| |_| |_|___|____/  |_|   |___|_| \_\____/ |_| (_)
     *
     * DO NOT EDIT THIS FILE.
     *
     * Transport version must now be added with the new transport version system. See docs/internal/Versioning.md
     */

    /**
     * Sorted list of all versions defined in this class
     */
    static final List<TransportVersion> DEFINED_VERSIONS = collectAllVersionIdsDefinedInClass(TransportVersions.class);

    public static List<TransportVersion> collectAllVersionIdsDefinedInClass(Class<?> cls) {
        Map<Integer, String> versionIdFields = new HashMap<>();
        List<TransportVersion> definedTransportVersions = new ArrayList<>();

        for (Field declaredField : cls.getFields()) {
            if (declaredField.getType().equals(TransportVersion.class)) {
                String fieldName = declaredField.getName();

                TransportVersion version;
                try {
                    version = (TransportVersion) declaredField.get(null);
                } catch (IllegalAccessException e) {
                    throw new AssertionError(e);
                }
                definedTransportVersions.add(version);

                if (Assertions.ENABLED) {
                    // check the version number is unique
                    var sameVersionNumber = versionIdFields.put(version.id(), fieldName);
                    assert sameVersionNumber == null
                        : "Versions ["
                            + sameVersionNumber
                            + "] and ["
                            + fieldName
                            + "] have the same version number ["
                            + version.id()
                            + "]. Each TransportVersion should have a different version number";
                }
            }
        }

        Collections.sort(definedTransportVersions);

        return List.copyOf(definedTransportVersions);
    }

    // no instance
    private TransportVersions() {}
}
