/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.blobstore;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.repositories.blobstore.RequestedRangeNotSatisfiedException;

public abstract class RetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(RetryingInputStream.class);
    public static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final BlobStoreServices blobStoreServices;
    private final OperationPurpose purpose;
    private final long start;
    private final long end;
    private final List<Exception> failures;
    protected SingleAttemptInputStream currentStream;
    private long offset = 0L;
    private int attempt = 1;
    private int failuresAfterMeaningfulProgress = 0;
    private boolean closed = false;

    protected RetryingInputStream(BlobStoreServices blobStoreServices, OperationPurpose purpose) throws IOException {
        this(blobStoreServices, purpose, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    protected RetryingInputStream(BlobStoreServices blobStoreServices, OperationPurpose purpose, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobStoreServices = blobStoreServices;
        this.purpose = purpose;
        this.failures = new ArrayList<Exception>(10);
        this.start = start;
        this.end = end;
        int initialAttempt = this.attempt;
        this.openStreamWithRetry();
        this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "open");
    }

    private void openStreamWithRetry() throws IOException {
        while (true) {
            if (this.offset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) assert (this.start + this.offset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.offset + " end=" + this.end;
            try {
                this.currentStream = this.blobStoreServices.getInputStream(Math.addExact(this.start, this.offset), this.end);
                return;
            }
            catch (NoSuchFileException | RequestedRangeNotSatisfiedException e) {
                throw e;
            }
            catch (RuntimeException e) {
                if (this.attempt == 1) {
                    this.blobStoreServices.onRetryStarted("open");
                }
                long delayInMillis = this.maybeLogAndComputeRetryDelay("opening", e);
                this.delayBeforeRetry(delayInMillis);
                continue;
            }
            break;
        }
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int result = this.currentStream.read();
                if (result != -1) {
                    ++this.offset;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "read");
                return result;
            }
            catch (IOException e) {
                if (this.attempt == initialAttempt) {
                    this.blobStoreServices.onRetryStarted("read");
                }
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead != -1) {
                    this.offset += (long)bytesRead;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "read");
                return bytesRead;
            }
            catch (IOException e) {
                if (this.attempt == initialAttempt) {
                    this.blobStoreServices.onRetryStarted("read");
                }
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using RetryingInputStream after close";
            throw new IllegalStateException("Stream is closed");
        }
    }

    private void reopenStreamOrFail(IOException e) throws IOException {
        long meaningfulProgressSize = this.blobStoreServices.getMeaningfulProgressSize();
        if (this.currentStreamProgress() >= meaningfulProgressSize) {
            ++this.failuresAfterMeaningfulProgress;
        }
        long delayInMillis = this.maybeLogAndComputeRetryDelay("reading", e);
        IOUtils.closeWhileHandlingException((Closeable)this.currentStream);
        this.delayBeforeRetry(delayInMillis);
        this.openStreamWithRetry();
    }

    private <T extends Exception> long maybeLogAndComputeRetryDelay(String action, T e) throws T {
        if (!this.shouldRetry(this.attempt)) {
            T finalException = this.addSuppressedExceptions(e);
            this.logForFailure(action, finalException);
            throw finalException;
        }
        this.logForRetry(Integer.bitCount(this.attempt) == 1 ? Level.INFO : Level.DEBUG, action, e);
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        long delayInMillis = this.getRetryDelayInMillis();
        ++this.attempt;
        return delayInMillis;
    }

    private void logForFailure(String action, Exception e) {
        logger.warn(() -> Strings.format((String)"failed %s [%s] at offset [%s] with purpose [%s]", (Object[])new Object[]{action, this.blobStoreServices.getBlobDescription(), this.start + this.offset, this.purpose.getKey()}), (Throwable)e);
    }

    private void logForRetry(Level level, String action, Exception e) {
        logger.log(level, () -> Strings.format((String)"failed %s [%s] at offset [%s] with purpose [%s]; this was attempt [%s] to read this blob which yielded [%s] bytes; in total [%s] of the attempts to read this blob have made meaningful progress and do not count towards the maximum number of retries; the maximum number of read attempts which do not make meaningful progress is [%s]", (Object[])new Object[]{action, this.blobStoreServices.getBlobDescription(), this.start + this.offset, this.purpose.getKey(), this.attempt, this.currentStreamProgress(), this.failuresAfterMeaningfulProgress, this.maxRetriesForNoMeaningfulProgress()}), (Throwable)e);
    }

    private void maybeLogAndRecordMetricsForSuccess(int initialAttempt, String action) {
        if (this.attempt > initialAttempt) {
            int numberOfRetries = this.attempt - initialAttempt;
            logger.info("successfully {} input stream for [{}] with purpose [{}] after [{}] retries", (Object)action, (Object)this.blobStoreServices.getBlobDescription(), (Object)this.purpose.getKey(), (Object)numberOfRetries);
            this.blobStoreServices.onRetrySucceeded(action, numberOfRetries);
        }
    }

    private long currentStreamProgress() {
        if (this.currentStream == null) {
            return 0L;
        }
        return Math.subtractExact(Math.addExact(this.start, this.offset), this.currentStream.getFirstOffset());
    }

    private boolean shouldRetry(int attempt) {
        if (this.purpose == OperationPurpose.REPOSITORY_ANALYSIS) {
            return false;
        }
        if (this.purpose == OperationPurpose.INDICES) {
            return true;
        }
        int maxAttempts = this.blobStoreServices.getMaxRetries() + 1;
        return attempt < maxAttempts + this.failuresAfterMeaningfulProgress;
    }

    private int maxRetriesForNoMeaningfulProgress() {
        return this.purpose == OperationPurpose.INDICES ? Integer.MAX_VALUE : this.blobStoreServices.getMaxRetries() + 1;
    }

    private void delayBeforeRetry(long delayInMillis) {
        try {
            assert (this.shouldRetry(this.attempt - 1)) : "should not have retried";
            Thread.sleep(delayInMillis);
        }
        catch (InterruptedException e) {
            logger.info("retrying input stream delay interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    protected long getRetryDelayInMillis() {
        return 10L << Math.min(this.attempt - 1, 10);
    }

    @Override
    public void close() throws IOException {
        try {
            this.currentStream.close();
        }
        finally {
            this.closed = true;
        }
    }

    @Override
    public long skip(long n) throws IOException {
        this.ensureOpen();
        return this.currentStream.skip(n);
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (Exception failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    protected static interface BlobStoreServices {
        public SingleAttemptInputStream getInputStream(long var1, long var3) throws IOException;

        public void onRetryStarted(String var1);

        public void onRetrySucceeded(String var1, long var2);

        public long getMeaningfulProgressSize();

        public int getMaxRetries();

        public String getBlobDescription();
    }

    protected static abstract class SingleAttemptInputStream
    extends InputStream {
        protected SingleAttemptInputStream() {
        }

        protected abstract long getFirstOffset();
    }
}

