/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.NoSuchFileException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.support.RefCountingListener;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.action.support.ThreadedActionListener;
import org.elasticsearch.cluster.service.MasterService;
import org.elasticsearch.common.BackoffPolicy;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.blobstore.OptionalBytesReference;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.BlobContainerUtils;
import org.elasticsearch.common.blobstore.support.BlobMetadata;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Iterators;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.blobstore.ChunkedBlobOutputStream;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobStore;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.S3RetryingInputStream;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.threadpool.ThreadPool;
import software.amazon.awssdk.awscore.AwsRequest;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CommonPrefix;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.ListMultipartUploadsRequest;
import software.amazon.awssdk.services.s3.model.ListMultipartUploadsResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.MultipartUpload;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Exception;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.model.SdkPartType;
import software.amazon.awssdk.services.s3.model.ServerSideEncryption;
import software.amazon.awssdk.services.s3.model.UploadPartCopyRequest;
import software.amazon.awssdk.services.s3.model.UploadPartCopyResponse;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.model.UploadPartResponse;

class S3BlobContainer
extends AbstractBlobContainer {
    private static final Logger logger = LogManager.getLogger(S3BlobContainer.class);
    private final S3BlobStore blobStore;
    private final String keyPath;

    S3BlobContainer(BlobPath path, S3BlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.keyPath = path.buildAsString();
    }

    public boolean blobExists(OperationPurpose purpose, String blobName) {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        try {
            boolean bl = this.doesObjectExist(purpose, clientReference, this.blobStore.bucket(), this.buildKey(blobName));
            if (clientReference != null) {
                clientReference.close();
            }
            return bl;
        }
        catch (Throwable throwable) {
            try {
                if (clientReference != null) {
                    try {
                        clientReference.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new BlobStoreException("Failed to check if blob [" + blobName + "] exists", (Throwable)e);
            }
        }
    }

    public InputStream readBlob(OperationPurpose purpose, String blobName) throws IOException {
        return new S3RetryingInputStream(purpose, this.blobStore, this.buildKey(blobName));
    }

    public InputStream readBlob(OperationPurpose purpose, String blobName, long position, long length) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("position must be non-negative");
        }
        if (length < 0L) {
            throw new IllegalArgumentException("length must be non-negative");
        }
        if (length == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new S3RetryingInputStream(purpose, this.blobStore, this.buildKey(blobName), position, Math.addExact(position, length - 1L));
    }

    public long readBlobPreferredLength() {
        return ByteSizeValue.of((long)32L, (ByteSizeUnit)ByteSizeUnit.MB).getBytes();
    }

    public void writeBlob(OperationPurpose purpose, String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (BlobContainer.assertPurposeConsistency((OperationPurpose)purpose, (String)blobName));
        assert (inputStream.markSupported()) : "No mark support on inputStream breaks the S3 SDK's ability to retry requests";
        if (blobSize <= this.getLargeBlobThresholdInBytes()) {
            this.executeSingleUpload(purpose, this.blobStore, this.buildKey(blobName), inputStream, blobSize, failIfAlreadyExists);
        } else {
            this.executeMultipartUpload(purpose, this.blobStore, this.buildKey(blobName), inputStream, blobSize, failIfAlreadyExists);
        }
    }

    public void writeMetadataBlob(final OperationPurpose purpose, final String blobName, final boolean failIfAlreadyExists, boolean atomic, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        assert (purpose != OperationPurpose.SNAPSHOT_DATA && BlobContainer.assertPurposeConsistency((OperationPurpose)purpose, (String)blobName)) : purpose;
        final String absoluteBlobKey = this.buildKey(blobName);
        try (ChunkedBlobOutputStream<CompletedPart> out = new ChunkedBlobOutputStream<CompletedPart>(this.blobStore.bigArrays(), this.blobStore.bufferSizeInBytes()){
            private final SetOnce<String> uploadId;
            {
                super(arg0, arg1);
                this.uploadId = new SetOnce();
            }

            protected void flushBuffer() throws IOException {
                this.flushBuffer(false);
            }

            private void flushBuffer(boolean lastPart) throws IOException {
                UploadPartResponse uploadResponse;
                if (this.buffer.size() == 0) {
                    return;
                }
                if (this.flushedBytes == 0L) {
                    assert (!lastPart) : "use single part upload if there's only a single part";
                    try (AmazonS3Reference clientReference = S3BlobContainer.this.blobStore.clientReference();){
                        this.uploadId.set((Object)clientReference.client().createMultipartUpload(S3BlobContainer.this.createMultipartUpload(purpose, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, absoluteBlobKey)).uploadId());
                    }
                    if (Strings.isEmpty((CharSequence)((CharSequence)this.uploadId.get()))) {
                        throw new IOException("Failed to initialize multipart upload " + absoluteBlobKey);
                    }
                }
                assert (!lastPart || this.successful) : "must only write last part if successful";
                UploadPartRequest uploadRequest = S3BlobContainer.this.createPartUploadRequest(purpose, (String)this.uploadId.get(), this.parts.size() + 1, absoluteBlobKey, this.buffer.size(), lastPart);
                StreamInput partContentStream = this.buffer.bytes().streamInput();
                try (AmazonS3Reference clientReference = S3BlobContainer.this.blobStore.clientReference();){
                    uploadResponse = clientReference.client().uploadPart(uploadRequest, RequestBody.fromInputStream((InputStream)partContentStream, (long)this.buffer.size()));
                }
                this.finishPart((CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(this.parts.size() + 1)).eTag(uploadResponse.eTag()).build());
            }

            protected void onCompletion() throws IOException {
                if (this.flushedBytes == 0L) {
                    S3BlobContainer.this.writeBlob(purpose, blobName, this.buffer.bytes(), failIfAlreadyExists);
                } else {
                    this.flushBuffer(true);
                    CompleteMultipartUploadRequest.Builder completeMultipartUploadRequestBuilder = CompleteMultipartUploadRequest.builder().bucket(S3BlobContainer.this.blobStore.bucket()).key(absoluteBlobKey).uploadId((String)this.uploadId.get()).multipartUpload(b -> b.parts((Collection)this.parts));
                    S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)completeMultipartUploadRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, purpose);
                    CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)completeMultipartUploadRequestBuilder.build();
                    try (AmazonS3Reference clientReference = S3BlobContainer.this.blobStore.clientReference();){
                        clientReference.client().completeMultipartUpload(completeMultipartUploadRequest);
                    }
                }
            }

            protected void onFailure() {
                if (Strings.hasText((String)((String)this.uploadId.get()))) {
                    S3BlobContainer.this.abortMultiPartUpload(purpose, (String)this.uploadId.get(), absoluteBlobKey);
                }
            }
        };){
            writer.accept((Object)out);
            out.markSuccess();
        }
    }

    private boolean doesObjectExist(OperationPurpose purpose, AmazonS3Reference clientReference, String bucketName, String objectName) {
        try {
            HeadObjectRequest.Builder headObjectRequestBuilder = HeadObjectRequest.builder().bucket(bucketName).key(objectName);
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)headObjectRequestBuilder, this.blobStore, S3BlobStore.Operation.HEAD_OBJECT, purpose);
            clientReference.client().headObject((HeadObjectRequest)headObjectRequestBuilder.build());
            return true;
        }
        catch (S3Exception e) {
            if (e.statusCode() == 404) {
                return false;
            }
            throw e;
        }
    }

    private UploadPartRequest createPartUploadRequest(OperationPurpose purpose, String uploadId, int number, String blobName, long size, boolean lastPart) {
        UploadPartRequest.Builder uploadPartRequestBuilder = UploadPartRequest.builder();
        uploadPartRequestBuilder.bucket(this.blobStore.bucket());
        uploadPartRequestBuilder.key(blobName);
        uploadPartRequestBuilder.uploadId(uploadId);
        uploadPartRequestBuilder.partNumber(Integer.valueOf(number));
        uploadPartRequestBuilder.contentLength(Long.valueOf(size));
        uploadPartRequestBuilder.sdkPartType(lastPart ? SdkPartType.LAST : SdkPartType.DEFAULT);
        S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)uploadPartRequestBuilder, this.blobStore, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, purpose);
        return (UploadPartRequest)uploadPartRequestBuilder.build();
    }

    private void abortMultiPartUpload(OperationPurpose purpose, String uploadId, String blobName) {
        AbortMultipartUploadRequest.Builder abortMultipartUploadRequestBuilder = AbortMultipartUploadRequest.builder().bucket(this.blobStore.bucket()).key(blobName).uploadId(uploadId);
        S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)abortMultipartUploadRequestBuilder, this.blobStore, S3BlobStore.Operation.ABORT_MULTIPART_OBJECT, purpose);
        AbortMultipartUploadRequest abortMultipartUploadRequest = (AbortMultipartUploadRequest)abortMultipartUploadRequestBuilder.build();
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            clientReference.client().abortMultipartUpload(abortMultipartUploadRequest);
        }
    }

    private CreateMultipartUploadRequest createMultipartUpload(OperationPurpose purpose, S3BlobStore.Operation operation, String blobName) {
        CreateMultipartUploadRequest.Builder createMultipartUploadRequestBuilder = CreateMultipartUploadRequest.builder().bucket(this.blobStore.bucket()).key(blobName).storageClass(this.blobStore.getStorageClass()).acl(this.blobStore.getCannedACL());
        if (this.blobStore.serverSideEncryption()) {
            createMultipartUploadRequestBuilder.serverSideEncryption(ServerSideEncryption.AES256);
        }
        S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)createMultipartUploadRequestBuilder, this.blobStore, operation, purpose);
        return (CreateMultipartUploadRequest)createMultipartUploadRequestBuilder.build();
    }

    long getLargeBlobThresholdInBytes() {
        return this.blobStore.bufferSizeInBytes();
    }

    long getMaxCopySizeBeforeMultipart() {
        return this.blobStore.maxCopySizeBeforeMultipart();
    }

    public void writeBlobAtomic(OperationPurpose purpose, String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        this.writeBlob(purpose, blobName, inputStream, blobSize, failIfAlreadyExists);
    }

    public void writeBlobAtomic(OperationPurpose purpose, String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
        this.writeBlob(purpose, blobName, bytes, failIfAlreadyExists);
    }

    public void copyBlob(OperationPurpose purpose, BlobContainer sourceBlobContainer, String sourceBlobName, String blobName, long blobSize) throws IOException {
        block13: {
            assert (BlobContainer.assertPurposeConsistency((OperationPurpose)purpose, (String)sourceBlobName));
            assert (BlobContainer.assertPurposeConsistency((OperationPurpose)purpose, (String)blobName));
            if (!(sourceBlobContainer instanceof S3BlobContainer)) {
                throw new IllegalArgumentException("source blob container must be a S3BlobContainer");
            }
            S3BlobContainer s3SourceBlobContainer = (S3BlobContainer)sourceBlobContainer;
            try {
                if (blobSize > this.getMaxCopySizeBeforeMultipart()) {
                    this.executeMultipartCopy(purpose, s3SourceBlobContainer, sourceBlobName, blobName, blobSize);
                    break block13;
                }
                String blobKey = this.buildKey(blobName);
                CopyObjectRequest.Builder copyObjectRequestBuilder = CopyObjectRequest.builder().sourceBucket(s3SourceBlobContainer.blobStore.bucket()).sourceKey(s3SourceBlobContainer.buildKey(sourceBlobName)).destinationBucket(this.blobStore.bucket()).destinationKey(blobKey).acl(this.blobStore.getCannedACL()).storageClass(this.blobStore.getStorageClass());
                S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)copyObjectRequestBuilder, this.blobStore, S3BlobStore.Operation.COPY_OBJECT, purpose);
                CopyObjectRequest copyObjectRequest = (CopyObjectRequest)copyObjectRequestBuilder.build();
                try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
                    clientReference.client().copyObject(copyObjectRequest);
                }
            }
            catch (SdkException e) {
                SdkServiceException sse;
                if (e instanceof SdkServiceException && (sse = (SdkServiceException)((Object)e)).statusCode() == RestStatus.NOT_FOUND.getStatus()) {
                    throw new NoSuchFileException("Copy source [" + s3SourceBlobContainer.buildKey(sourceBlobName) + "] not found: " + sse.getMessage());
                }
                throw new IOException("Unable to copy object [" + blobName + "] from [" + String.valueOf(sourceBlobContainer) + "][" + sourceBlobName + "]", e);
            }
        }
    }

    public DeleteResult delete(OperationPurpose purpose) throws IOException {
        AtomicLong deletedBlobs = new AtomicLong();
        AtomicLong deletedBytes = new AtomicLong();
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            Iterator blobNameIterator;
            ListObjectsV2Response prevListing = null;
            while (true) {
                ListObjectsV2Request.Builder listObjectsRequestBuilder = ListObjectsV2Request.builder().bucket(this.blobStore.bucket()).prefix(this.keyPath);
                S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)listObjectsRequestBuilder, this.blobStore, S3BlobStore.Operation.LIST_OBJECTS, purpose);
                if (prevListing != null) {
                    listObjectsRequestBuilder.continuationToken(prevListing.nextContinuationToken());
                }
                ListObjectsV2Request listObjectsRequest = (ListObjectsV2Request)listObjectsRequestBuilder.build();
                ListObjectsV2Response listObjectsResponse = clientReference.client().listObjectsV2(listObjectsRequest);
                blobNameIterator = Iterators.map(listObjectsResponse.contents().iterator(), s3Object -> {
                    deletedBlobs.incrementAndGet();
                    deletedBytes.addAndGet(s3Object.size());
                    return s3Object.key();
                });
                if (!listObjectsResponse.isTruncated().booleanValue()) break;
                this.blobStore.deleteBlobs(purpose, blobNameIterator);
                prevListing = listObjectsResponse;
            }
            this.blobStore.deleteBlobs(purpose, Iterators.concat((Iterator[])new Iterator[]{blobNameIterator, Iterators.single((Object)this.keyPath)}));
        }
        catch (SdkException e) {
            throw new IOException("Exception when deleting blob container [" + this.keyPath + "]", e);
        }
        return new DeleteResult(deletedBlobs.get(), deletedBytes.get());
    }

    public void deleteBlobsIgnoringIfNotExists(OperationPurpose purpose, Iterator<String> blobNames) throws IOException {
        this.blobStore.deleteBlobs(purpose, Iterators.map(blobNames, this::buildKey));
    }

    public Map<String, BlobMetadata> listBlobsByPrefix(OperationPurpose purpose, @Nullable String blobNamePrefix) throws IOException {
        try {
            HashMap<String, BlobMetadata> results = new HashMap<String, BlobMetadata>();
            Iterator<ListObjectsV2Response> iterator = this.executeListing(purpose, blobNamePrefix == null ? this.keyPath : this.buildKey(blobNamePrefix));
            while (iterator.hasNext()) {
                ListObjectsV2Response currentPage = iterator.next();
                for (S3Object s3Object : currentPage.contents()) {
                    String blobName = s3Object.key().substring(this.keyPath.length());
                    if (results.put(blobName, new BlobMetadata(blobName, s3Object.size().longValue())) == null) continue;
                    throw new IllegalStateException("listing objects by prefix [" + blobNamePrefix + "] yielded multiple blobs with key [" + s3Object.key() + "]");
                }
            }
            return results;
        }
        catch (SdkException e) {
            throw new IOException("Exception when listing blobs by prefix [" + blobNamePrefix + "]", e);
        }
    }

    public Map<String, BlobMetadata> listBlobs(OperationPurpose purpose) throws IOException {
        return this.listBlobsByPrefix(purpose, null);
    }

    public Map<String, BlobContainer> children(OperationPurpose purpose) throws IOException {
        try {
            HashMap<String, BlobContainer> results = new HashMap<String, BlobContainer>();
            int relativePrefixStart = this.keyPath.length();
            Iterator<ListObjectsV2Response> iterator = this.executeListing(purpose, this.keyPath);
            while (iterator.hasNext()) {
                ListObjectsV2Response currentPage = iterator.next();
                for (CommonPrefix commonPrefix : currentPage.commonPrefixes()) {
                    String absolutePrefix = commonPrefix.prefix();
                    if (absolutePrefix.length() <= relativePrefixStart + 1) continue;
                    String relativePrefix = absolutePrefix.substring(relativePrefixStart, absolutePrefix.length() - 1);
                    assert (!relativePrefix.isEmpty());
                    assert (currentPage.contents().stream().noneMatch(s3Object -> s3Object.key().startsWith(absolutePrefix))) : "Response contained children for listed common prefix " + absolutePrefix;
                    if (results.put(relativePrefix, this.blobStore.blobContainer(this.path().add(relativePrefix))) == null) continue;
                    throw new IllegalStateException("listing child containers of [" + this.keyPath + "] yielded multiple children with key [" + relativePrefix + "]");
                }
            }
            return results;
        }
        catch (SdkException e) {
            throw new IOException("Exception when listing children of [" + this.path().buildAsString() + "]", e);
        }
    }

    private Iterator<ListObjectsV2Response> executeListing(final OperationPurpose purpose, final String pathPrefix) {
        return new Iterator<ListObjectsV2Response>(){
            @Nullable
            private ListObjectsV2Response nextResponse;
            {
                this.nextResponse = S3BlobContainer.this.listNextObjects(purpose, pathPrefix, null);
            }

            @Override
            public boolean hasNext() {
                return this.nextResponse != null;
            }

            @Override
            public ListObjectsV2Response next() {
                ListObjectsV2Response currentResponse = this.nextResponse;
                this.nextResponse = currentResponse.nextContinuationToken() == null ? null : S3BlobContainer.this.listNextObjects(purpose, pathPrefix, currentResponse);
                return currentResponse;
            }
        };
    }

    private ListObjectsV2Response listNextObjects(OperationPurpose operationPurpose, String pathPrefix, @Nullable ListObjectsV2Response previousResponse) {
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            ListObjectsV2Request.Builder listObjectsRequestBuilder = ListObjectsV2Request.builder().bucket(this.blobStore.bucket()).prefix(pathPrefix).delimiter("/");
            if (previousResponse != null) {
                if (previousResponse.nextContinuationToken() == null) {
                    throw new IllegalStateException("cannot request next page of object listing without a continuation token");
                }
                listObjectsRequestBuilder.continuationToken(previousResponse.nextContinuationToken());
            }
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)listObjectsRequestBuilder, this.blobStore, S3BlobStore.Operation.LIST_OBJECTS, operationPurpose);
            ListObjectsV2Request listObjectsRequest = (ListObjectsV2Request)listObjectsRequestBuilder.build();
            ListObjectsV2Response listObjectsV2Response = clientReference.client().listObjectsV2(listObjectsRequest);
            return listObjectsV2Response;
        }
    }

    String buildKey(String blobName) {
        return this.keyPath + blobName;
    }

    void executeSingleUpload(OperationPurpose purpose, S3BlobStore s3BlobStore, String blobName, InputStream input, long blobSize, boolean failIfAlreadyExists) throws IOException {
        try (AmazonS3Reference clientReference = s3BlobStore.clientReference();){
            if (blobSize > S3Repository.MAX_FILE_SIZE.getBytes()) {
                throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than " + String.valueOf(S3Repository.MAX_FILE_SIZE));
            }
            if (blobSize > s3BlobStore.bufferSizeInBytes()) {
                throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than buffer size");
            }
            PutObjectRequest.Builder putRequestBuilder = PutObjectRequest.builder().bucket(s3BlobStore.bucket()).key(blobName).contentLength(Long.valueOf(blobSize)).storageClass(s3BlobStore.getStorageClass()).acl(s3BlobStore.getCannedACL());
            if (s3BlobStore.serverSideEncryption()) {
                putRequestBuilder.serverSideEncryption(ServerSideEncryption.AES256);
            }
            if (failIfAlreadyExists && s3BlobStore.supportsConditionalWrites(purpose)) {
                putRequestBuilder.ifNoneMatch("*");
            }
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)putRequestBuilder, this.blobStore, S3BlobStore.Operation.PUT_OBJECT, purpose);
            PutObjectRequest putRequest = (PutObjectRequest)putRequestBuilder.build();
            clientReference.client().putObject(putRequest, RequestBody.fromInputStream((InputStream)input, (long)blobSize));
        }
        catch (SdkException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using a single upload", e);
        }
    }

    private void executeMultipart(OperationPurpose purpose, S3BlobStore.Operation operation, S3BlobStore s3BlobStore, String blobName, long partSize, long blobSize, PartOperation partOperation, boolean failIfAlreadyExists) throws IOException {
        this.ensureMultiPartUploadSize(blobSize);
        Tuple<Long, Long> multiparts = S3BlobContainer.numberOfMultiparts(blobSize, partSize);
        if ((Long)multiparts.v1() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too many multipart upload requests, maybe try a larger part size?");
        }
        int nbParts = ((Long)multiparts.v1()).intValue();
        long lastPartSize = (Long)multiparts.v2();
        assert (blobSize == (long)(nbParts - 1) * partSize + lastPartSize) : "blobSize does not match multipart sizes";
        ArrayList<Runnable> cleanupOnFailureActions = new ArrayList<Runnable>(1);
        String bucketName = s3BlobStore.bucket();
        try {
            String uploadId;
            try (AmazonS3Reference clientReference = s3BlobStore.clientReference();){
                uploadId = clientReference.client().createMultipartUpload(this.createMultipartUpload(purpose, operation, blobName)).uploadId();
                cleanupOnFailureActions.add(() -> {
                    try {
                        this.abortMultiPartUpload(purpose, uploadId, blobName);
                    }
                    catch (Exception e) {
                        SdkServiceException sdkServiceException;
                        if (e instanceof SdkServiceException && (sdkServiceException = (SdkServiceException)((Object)((Object)e))).statusCode() == RestStatus.NOT_FOUND.getStatus()) {
                            logger.atDebug().withThrowable((Throwable)e).log("multipart upload of [{}] with ID [{}] not found on abort", (Object)blobName, (Object)uploadId);
                        }
                        logger.atWarn().withThrowable((Throwable)e).log("failed to clean up multipart upload of [{}] with ID [{}] after earlier failure", (Object)blobName, (Object)uploadId);
                    }
                });
            }
            if (Strings.isEmpty((CharSequence)uploadId)) {
                throw new IOException("Failed to initialize multipart operation for " + blobName);
            }
            ArrayList<CompletedPart> parts = new ArrayList<CompletedPart>();
            long bytesCount = 0L;
            for (int i = 1; i <= nbParts; ++i) {
                boolean lastPart = i == nbParts;
                long curPartSize = lastPart ? lastPartSize : partSize;
                CompletedPart partEtag = partOperation.doPart(uploadId, i, curPartSize, lastPart);
                bytesCount += curPartSize;
                parts.add(partEtag);
            }
            if (bytesCount != blobSize) {
                throw new IOException("Failed to execute multipart operation for [" + blobName + "], expected " + blobSize + "bytes sent but got " + bytesCount);
            }
            CompleteMultipartUploadRequest.Builder completeMultipartUploadRequestBuilder = CompleteMultipartUploadRequest.builder().bucket(bucketName).key(blobName).uploadId(uploadId).multipartUpload(b -> b.parts((Collection)parts));
            if (failIfAlreadyExists && s3BlobStore.supportsConditionalWrites(purpose)) {
                completeMultipartUploadRequestBuilder.ifNoneMatch("*");
            }
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)completeMultipartUploadRequestBuilder, this.blobStore, operation, purpose);
            CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)completeMultipartUploadRequestBuilder.build();
            try (AmazonS3Reference clientReference = s3BlobStore.clientReference();){
                clientReference.client().completeMultipartUpload(completeMultipartUploadRequest);
            }
            cleanupOnFailureActions.clear();
        }
        catch (SdkException e) {
            SdkServiceException sse;
            if (e instanceof SdkServiceException && (sse = (SdkServiceException)((Object)e)).statusCode() == RestStatus.NOT_FOUND.getStatus()) {
                throw new NoSuchFileException(blobName, null, e.getMessage());
            }
            throw new IOException("Unable to upload or copy object [" + blobName + "] using multipart upload", e);
        }
        finally {
            cleanupOnFailureActions.forEach(Runnable::run);
        }
    }

    void executeMultipartUpload(OperationPurpose purpose, S3BlobStore s3BlobStore, String blobName, InputStream input, long blobSize, boolean failIfAlreadyExists) throws IOException {
        this.executeMultipart(purpose, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, s3BlobStore, blobName, s3BlobStore.bufferSizeInBytes(), blobSize, (uploadId, partNum, partSize, lastPart) -> {
            UploadPartRequest uploadRequest = this.createPartUploadRequest(purpose, uploadId, partNum, blobName, partSize, lastPart);
            try (AmazonS3Reference clientReference = s3BlobStore.clientReference();){
                UploadPartResponse uploadResponse = clientReference.client().uploadPart(uploadRequest, RequestBody.fromInputStream((InputStream)input, (long)partSize));
                CompletedPart completedPart = (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(partNum)).eTag(uploadResponse.eTag()).build();
                return completedPart;
            }
        }, failIfAlreadyExists);
    }

    void executeMultipartCopy(OperationPurpose purpose, S3BlobContainer sourceContainer, String sourceBlobName, String destinationBlobName, long blobSize) throws IOException {
        long copyPartSize = S3Repository.MAX_FILE_SIZE.getBytes();
        String destinationKey = this.buildKey(destinationBlobName);
        this.executeMultipart(purpose, S3BlobStore.Operation.COPY_MULTIPART_OBJECT, this.blobStore, destinationKey, copyPartSize, blobSize, (uploadId, partNum, partSize, lastPart) -> {
            long startOffset = (long)(partNum - 1) * copyPartSize;
            UploadPartCopyRequest.Builder uploadPartCopyRequestBuilder = UploadPartCopyRequest.builder().sourceBucket(sourceContainer.blobStore.bucket()).sourceKey(sourceContainer.buildKey(sourceBlobName)).destinationBucket(this.blobStore.bucket()).destinationKey(destinationKey).uploadId(uploadId).partNumber(Integer.valueOf(partNum)).copySourceRange("bytes=" + startOffset + "-" + (startOffset + partSize - 1L));
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)uploadPartCopyRequestBuilder, this.blobStore, S3BlobStore.Operation.COPY_MULTIPART_OBJECT, purpose);
            UploadPartCopyRequest uploadPartCopyRequest = (UploadPartCopyRequest)uploadPartCopyRequestBuilder.build();
            try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
                UploadPartCopyResponse uploadPartCopyResponse = clientReference.client().uploadPartCopy(uploadPartCopyRequest);
                CompletedPart completedPart = (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(partNum)).eTag(uploadPartCopyResponse.copyPartResult().eTag()).build();
                return completedPart;
            }
        }, false);
    }

    void ensureMultiPartUploadSize(long blobSize) {
        if (blobSize > S3Repository.MAX_FILE_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be larger than " + String.valueOf(S3Repository.MAX_FILE_SIZE_USING_MULTIPART));
        }
        if (blobSize < S3Repository.MIN_PART_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be smaller than " + String.valueOf(S3Repository.MIN_PART_SIZE_USING_MULTIPART));
        }
    }

    static Tuple<Long, Long> numberOfMultiparts(long totalSize, long partSize) {
        if (partSize <= 0L) {
            throw new IllegalArgumentException("Part size must be greater than zero");
        }
        if (totalSize == 0L || totalSize <= partSize) {
            return Tuple.tuple((Object)1L, (Object)totalSize);
        }
        long parts = totalSize / partSize;
        long remaining = totalSize % partSize;
        if (remaining == 0L) {
            return Tuple.tuple((Object)parts, (Object)partSize);
        }
        return Tuple.tuple((Object)(parts + 1L), (Object)remaining);
    }

    public void compareAndExchangeRegister(OperationPurpose purpose, String key, BytesReference expected, BytesReference updated, ActionListener<OptionalBytesReference> listener) {
        AmazonS3Reference clientReference = this.blobStore.clientReference();
        new MultipartUploadCompareAndExchangeOperation(purpose, clientReference.client(), this.blobStore.bucket(), key, this.blobStore.getThreadPool()).run(expected, updated, (ActionListener<OptionalBytesReference>)ActionListener.releaseBefore((Releasable)clientReference, listener));
    }

    public void getRegister(OperationPurpose purpose, String key, ActionListener<OptionalBytesReference> listener) {
        this.getRegisterAndEtag(purpose, key, (ActionListener<RegisterAndEtag>)listener.map(registerAndEtag -> OptionalBytesReference.of((BytesReference)registerAndEtag.registerContents())));
    }

    void getRegisterAndEtag(OperationPurpose purpose, String key, ActionListener<RegisterAndEtag> listener) {
        ActionListener.completeWith(listener, () -> {
            BackoffPolicy backoffPolicy = purpose == OperationPurpose.REPOSITORY_ANALYSIS ? BackoffPolicy.noBackoff() : BackoffPolicy.constantBackoff((TimeValue)this.blobStore.getGetRegisterRetryDelay(), (int)this.blobStore.getMaxRetries());
            Iterator retryDelayIterator = backoffPolicy.iterator();
            Exception finalException = null;
            while (true) {
                RegisterAndEtag registerAndEtag;
                AmazonS3Reference clientReference;
                block23: {
                    GetObjectRequest.Builder getObjectRequestBuilder = GetObjectRequest.builder().bucket(this.blobStore.bucket()).key(this.buildKey(key));
                    S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)getObjectRequestBuilder, this.blobStore, S3BlobStore.Operation.GET_OBJECT, purpose);
                    GetObjectRequest getObjectRequest = (GetObjectRequest)getObjectRequestBuilder.build();
                    clientReference = this.blobStore.clientReference();
                    ResponseInputStream s3Object2 = clientReference.client().getObject(getObjectRequest);
                    try {
                        registerAndEtag = new RegisterAndEtag(BlobContainerUtils.getRegisterUsingConsistentRead((InputStream)s3Object2, (String)this.keyPath, (String)key), this.getRequiredEtag(purpose, (GetObjectResponse)s3Object2.response()));
                        if (s3Object2 == null) break block23;
                    }
                    catch (Throwable throwable) {
                        try {
                            try {
                                if (s3Object2 != null) {
                                    try {
                                        s3Object2.close();
                                    }
                                    catch (Throwable throwable2) {
                                        throwable.addSuppressed(throwable2);
                                    }
                                }
                                throw throwable;
                            }
                            catch (Throwable s3Object2) {
                                if (clientReference != null) {
                                    try {
                                        clientReference.close();
                                    }
                                    catch (Throwable throwable3) {
                                        s3Object2.addSuppressed(throwable3);
                                    }
                                }
                                throw s3Object2;
                            }
                        }
                        catch (Exception attemptException) {
                            block24: {
                                SdkServiceException sdkException;
                                logger.trace(() -> Strings.format((String)"[%s]: getRegister failed", (Object[])new Object[]{key}), (Throwable)attemptException);
                                if (attemptException instanceof SdkServiceException && (sdkException = (SdkServiceException)((Object)((Object)attemptException))).statusCode() == 404) {
                                    return RegisterAndEtag.ABSENT;
                                }
                                if (finalException == null) {
                                    finalException = attemptException;
                                } else if (finalException != attemptException) {
                                    finalException.addSuppressed(attemptException);
                                }
                                if (retryDelayIterator.hasNext()) {
                                    try {
                                        Thread.sleep(((TimeValue)retryDelayIterator.next()).millis());
                                    }
                                    catch (InterruptedException interruptedException) {
                                        Thread.currentThread().interrupt();
                                        finalException.addSuppressed(interruptedException);
                                        break block24;
                                    }
                                    continue;
                                }
                            }
                            throw finalException;
                        }
                    }
                    s3Object2.close();
                }
                if (clientReference != null) {
                    clientReference.close();
                }
                return registerAndEtag;
                break;
            }
        });
    }

    private String getRequiredEtag(OperationPurpose purpose, GetObjectResponse getObjectResponse) {
        String etag = getObjectResponse.eTag();
        if (Strings.hasText((String)etag)) {
            return etag;
        }
        if (this.blobStore.supportsConditionalWrites(purpose)) {
            throw new UnsupportedOperationException("GetObject response contained no ETag header, cannot perform conditional write");
        }
        return "es-missing-but-ignored-etag";
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    ActionListener<Void> getMultipartUploadCleanupListener(int maxUploads, RefCountingRunnable refs) {
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            String bucket = this.blobStore.bucket();
            ListMultipartUploadsRequest listMultipartUploadsRequest = (ListMultipartUploadsRequest)ListMultipartUploadsRequest.builder().bucket(bucket).prefix(this.keyPath).maxUploads(Integer.valueOf(maxUploads)).overrideConfiguration(b -> this.blobStore.addPurposeQueryParameter(OperationPurpose.SNAPSHOT_DATA, (AwsRequestOverrideConfiguration.Builder)b)).build();
            ListMultipartUploadsResponse multipartUploadListing = clientReference.client().listMultipartUploads(listMultipartUploadsRequest);
            List multipartUploads = multipartUploadListing.uploads();
            if (multipartUploads.isEmpty()) {
                logger.debug("found no multipart uploads to clean up");
                ActionListener actionListener = ActionListener.noop();
                return actionListener;
            }
            if (multipartUploadListing.isTruncated().booleanValue()) {
                logger.info("found at least [{}] possibly-dangling multipart uploads; will clean up the first [{}] after finalizing the current snapshot deletions, and will check for further possibly-dangling multipart uploads in future snapshot deletions", (Object)multipartUploads.size(), (Object)multipartUploads.size());
            } else {
                logger.info("found [{}] possibly-dangling multipart uploads; will clean them up after finalizing the current snapshot deletions", (Object)multipartUploads.size());
            }
            ActionListener<Void> actionListener = this.newMultipartUploadCleanupListener(refs, Iterators.map(multipartUploads.iterator(), u -> (AbortMultipartUploadRequest)AbortMultipartUploadRequest.builder().bucket(bucket).key(u.key()).uploadId(u.uploadId()).overrideConfiguration(b -> this.blobStore.addPurposeQueryParameter(OperationPurpose.SNAPSHOT_DATA, (AwsRequestOverrideConfiguration.Builder)b)).build()));
            return actionListener;
        }
        catch (Exception e) {
            logger.warn("failure while checking for possibly-dangling multipart uploads", (Throwable)e);
            return ActionListener.noop();
        }
    }

    private ActionListener<Void> newMultipartUploadCleanupListener(RefCountingRunnable refs, final Iterator<AbortMultipartUploadRequest> abortMultipartUploadRequestIterator) {
        return new ThreadedActionListener(this.blobStore.getSnapshotExecutor(), ActionListener.releaseAfter((ActionListener)new ActionListener<Void>(){

            public void onResponse(Void unused) {
                block10: {
                    AmazonS3Reference clientReference = S3BlobContainer.this.blobStore.clientReference();
                    block7: while (true) {
                        while (abortMultipartUploadRequestIterator.hasNext()) {
                            AbortMultipartUploadRequest abortMultipartUploadRequest = (AbortMultipartUploadRequest)abortMultipartUploadRequestIterator.next();
                            try {
                                clientReference.client().abortMultipartUpload(abortMultipartUploadRequest);
                                logger.info("cleaned up dangling multipart upload [{}] of blob [{}][{}][{}]", (Object)abortMultipartUploadRequest.uploadId(), (Object)S3BlobContainer.this.blobStore.getRepositoryMetadata().name(), (Object)abortMultipartUploadRequest.bucket(), (Object)abortMultipartUploadRequest.key());
                                continue block7;
                            }
                            catch (Exception e) {
                                logger.warn(Strings.format((String)"failed to clean up multipart upload [%s] of blob [%s][%s][%s]", (Object[])new Object[]{abortMultipartUploadRequest.uploadId(), S3BlobContainer.this.blobStore.getRepositoryMetadata().name(), abortMultipartUploadRequest.bucket(), abortMultipartUploadRequest.key()}), (Throwable)e);
                            }
                        }
                        break block10;
                        {
                            continue block7;
                            break;
                        }
                        break;
                    }
                    finally {
                        if (clientReference != null) {
                            clientReference.close();
                        }
                    }
                }
            }

            public void onFailure(Exception e) {
                Exception cause;
                RepositoryException repositoryException;
                Throwable throwable;
                logger.log(MasterService.isPublishFailureException((Exception)e) || e instanceof RepositoryException && (throwable = (repositoryException = (RepositoryException)e).getCause()) instanceof Exception && MasterService.isPublishFailureException((Exception)(cause = (Exception)throwable)) ? Level.DEBUG : Level.WARN, "failed to start cleanup of dangling multipart uploads", (Throwable)e);
            }
        }, (Releasable)refs.acquire()));
    }

    private static interface PartOperation {
        public CompletedPart doPart(String var1, int var2, long var3, boolean var5);
    }

    private class MultipartUploadCompareAndExchangeOperation {
        private final OperationPurpose purpose;
        private final S3Client client;
        private final String bucket;
        private final String rawKey;
        private final String blobKey;
        private final ThreadPool threadPool;

        MultipartUploadCompareAndExchangeOperation(OperationPurpose purpose, S3Client client, String bucket, String key, ThreadPool threadPool) {
            this.purpose = purpose;
            this.client = client;
            this.bucket = bucket;
            this.rawKey = key;
            this.blobKey = S3BlobContainer.this.buildKey(key);
            this.threadPool = threadPool;
        }

        void run(BytesReference expected, BytesReference updated, ActionListener<OptionalBytesReference> listener) {
            ActionListener.run((ActionListener)listener.delegateResponse((delegate, e) -> {
                AwsServiceException awsServiceException;
                logger.trace(() -> Strings.format((String)"[%s]: compareAndExchangeRegister failed", (Object[])new Object[]{this.rawKey}), (Throwable)e);
                if (e instanceof AwsServiceException && ((awsServiceException = (AwsServiceException)e).statusCode() == RestStatus.NOT_FOUND.getStatus() || awsServiceException.statusCode() == RestStatus.CONFLICT.getStatus() || awsServiceException.statusCode() == RestStatus.PRECONDITION_FAILED.getStatus() || awsServiceException.statusCode() == RestStatus.OK.getStatus() && "NoSuchUpload".equals(awsServiceException.awsErrorDetails().errorCode()))) {
                    delegate.onResponse((Object)OptionalBytesReference.MISSING);
                } else {
                    delegate.onFailure(e);
                }
            }), l -> this.innerRun(expected, updated, (ActionListener<OptionalBytesReference>)l));
        }

        void innerRun(BytesReference expected, BytesReference updated, ActionListener<OptionalBytesReference> listener) throws Exception {
            BlobContainerUtils.ensureValidRegisterContent((BytesReference)updated);
            if (this.hasPreexistingUploads()) {
                listener.onResponse((Object)OptionalBytesReference.MISSING);
                return;
            }
            String uploadId = this.createMultipartUpload();
            logger.trace("[{}] initiated upload [{}]", (Object)this.blobKey, (Object)uploadId);
            String partETag = this.uploadPartAndGetEtag(updated, uploadId);
            logger.trace("[{}] uploaded update to [{}]", (Object)this.blobKey, (Object)uploadId);
            List<MultipartUpload> currentUploads = this.listMultipartUploads();
            this.logUploads("uploads before current", currentUploads);
            int uploadIndex = this.getUploadIndex(uploadId, currentUploads);
            logger.trace("[{}] upload [{}] has index [{}]", (Object)this.blobKey, (Object)uploadId, (Object)uploadIndex);
            if (uploadIndex < 0) {
                listener.onResponse((Object)OptionalBytesReference.MISSING);
                return;
            }
            SubscribableListener.newForked(l -> this.ensureOtherUploadsComplete(uploadId, uploadIndex, currentUploads, (ActionListener<Void>)l)).andThen(l -> S3BlobContainer.this.getRegisterAndEtag(this.purpose, this.rawKey, (ActionListener<RegisterAndEtag>)l)).andThenApply(currentValueAndEtag -> {
                if (currentValueAndEtag.registerContents().equals((Object)expected)) {
                    logger.trace("[{}] completing upload [{}]", (Object)this.blobKey, (Object)uploadId);
                    this.completeMultipartUpload(uploadId, partETag, currentValueAndEtag.eTag());
                } else {
                    logger.trace("[{}] aborting upload [{}]", (Object)this.blobKey, (Object)uploadId);
                    this.safeAbortMultipartUpload(uploadId);
                }
                return OptionalBytesReference.of((BytesReference)currentValueAndEtag.registerContents());
            }).addListener(listener.delegateResponse((l, e) -> {
                logger.trace(() -> Strings.format((String)"[%s] aborting upload [%s] on exception", (Object[])new Object[]{this.blobKey, uploadId}), (Throwable)e);
                this.safeAbortMultipartUpload(uploadId);
                l.onFailure(e);
            }));
        }

        private boolean hasPreexistingUploads() {
            long timeToLiveMillis = S3BlobContainer.this.blobStore.getCompareAndExchangeTimeToLive().millis();
            if (timeToLiveMillis < 0L) {
                return false;
            }
            List<MultipartUpload> uploads = this.listMultipartUploads();
            this.logUploads("preexisting uploads", uploads);
            if (uploads.isEmpty()) {
                logger.trace("[{}] no preexisting uploads", (Object)this.blobKey);
                return false;
            }
            Instant expiryDate = Instant.ofEpochMilli(S3BlobContainer.this.blobStore.getThreadPool().absoluteTimeInMillis() - timeToLiveMillis);
            if (uploads.stream().anyMatch(upload -> upload.initiated().compareTo(expiryDate) > 0)) {
                logger.trace("[{}] fresh preexisting uploads vs {}", (Object)this.blobKey, (Object)expiryDate);
                return true;
            }
            for (MultipartUpload upload2 : uploads) {
                logger.warn("cleaning up stale compare-and-swap upload [{}] initiated at [{}]", (Object)upload2.uploadId(), (Object)upload2.initiated());
                this.safeAbortMultipartUpload(upload2.uploadId());
            }
            logger.trace("[{}] stale preexisting uploads vs {}", (Object)this.blobKey, (Object)expiryDate);
            return false;
        }

        private void logUploads(String description, List<MultipartUpload> uploads) {
            if (logger.isTraceEnabled()) {
                logger.trace("[{}] {}: [{}]", (Object)this.blobKey, (Object)description, (Object)uploads.stream().map(multipartUpload -> multipartUpload.uploadId() + ": " + String.valueOf(multipartUpload.initiated())).collect(Collectors.joining(",")));
            }
        }

        private List<MultipartUpload> listMultipartUploads() {
            ListMultipartUploadsRequest.Builder listRequestBuilder = ListMultipartUploadsRequest.builder().bucket(this.bucket).prefix(this.blobKey);
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)listRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.LIST_OBJECTS, this.purpose);
            ListMultipartUploadsRequest listRequest = (ListMultipartUploadsRequest)listRequestBuilder.build();
            try {
                return this.client.listMultipartUploads(listRequest).uploads();
            }
            catch (SdkServiceException e) {
                if (e.statusCode() == 404) {
                    return List.of();
                }
                throw e;
            }
        }

        private String createMultipartUpload() {
            CreateMultipartUploadRequest.Builder createMultipartUploadRequestBuilder = CreateMultipartUploadRequest.builder().bucket(this.bucket).key(this.blobKey);
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)createMultipartUploadRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, this.purpose);
            CreateMultipartUploadRequest createMultipartUploadRequest = (CreateMultipartUploadRequest)createMultipartUploadRequestBuilder.build();
            return this.client.createMultipartUpload(createMultipartUploadRequest).uploadId();
        }

        private String uploadPartAndGetEtag(BytesReference updated, String uploadId) throws IOException {
            UploadPartRequest.Builder uploadPartRequestBuilder = UploadPartRequest.builder();
            uploadPartRequestBuilder.bucket(this.bucket);
            uploadPartRequestBuilder.key(this.blobKey);
            uploadPartRequestBuilder.uploadId(uploadId);
            uploadPartRequestBuilder.partNumber(Integer.valueOf(1));
            uploadPartRequestBuilder.sdkPartType(SdkPartType.LAST);
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)uploadPartRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, this.purpose);
            return this.client.uploadPart((UploadPartRequest)uploadPartRequestBuilder.build(), RequestBody.fromInputStream((InputStream)updated.streamInput(), (long)updated.length())).eTag();
        }

        private int getUploadIndex(String targetUploadId, List<MultipartUpload> multipartUploads) {
            int uploadIndex = 0;
            boolean found = false;
            for (MultipartUpload multipartUpload : multipartUploads) {
                String observedUploadId = multipartUpload.uploadId();
                if (observedUploadId.equals(targetUploadId)) {
                    long currentTimeMillis = S3BlobContainer.this.blobStore.getThreadPool().absoluteTimeInMillis();
                    long ageMillis = currentTimeMillis - multipartUpload.initiated().toEpochMilli();
                    long expectedAgeRangeMillis = S3BlobContainer.this.blobStore.getCompareAndExchangeTimeToLive().millis();
                    if (0L <= expectedAgeRangeMillis && (ageMillis < -expectedAgeRangeMillis || ageMillis > expectedAgeRangeMillis)) {
                        logger.warn("compare-and-exchange of blob [{}:{}] was initiated at [{}={}] which deviates from local node epoch time [{}] by more than the warn threshold of [{}ms]", (Object)this.bucket, (Object)this.blobKey, (Object)multipartUpload.initiated(), (Object)multipartUpload.initiated().toEpochMilli(), (Object)currentTimeMillis, (Object)expectedAgeRangeMillis);
                    }
                    found = true;
                    continue;
                }
                if (observedUploadId.compareTo(targetUploadId) >= 0) continue;
                ++uploadIndex;
            }
            return found ? uploadIndex : -1;
        }

        private void ensureOtherUploadsComplete(String uploadId, int uploadIndex, List<MultipartUpload> currentUploads, ActionListener<Void> listener) {
            if (uploadIndex > 0) {
                this.threadPool.scheduleUnlessShuttingDown(TimeValue.timeValueMillis((long)((long)uploadIndex * S3BlobContainer.this.blobStore.getCompareAndExchangeAntiContentionDelay().millis() + (long)Randomness.get().nextInt(50))), S3BlobContainer.this.blobStore.getSnapshotExecutor(), (Runnable)ActionRunnable.wrap(listener, l -> this.cancelOtherUploads(uploadId, currentUploads, (ActionListener<Void>)l)));
            } else {
                this.cancelOtherUploads(uploadId, currentUploads, listener);
            }
        }

        private void cancelOtherUploads(String uploadId, List<MultipartUpload> currentUploads, ActionListener<Void> listener) {
            logger.trace("[{}] upload [{}] cancelling other uploads", (Object)this.blobKey, (Object)uploadId);
            Executor executor = S3BlobContainer.this.blobStore.getSnapshotExecutor();
            try (RefCountingListener listeners = new RefCountingListener(listener);){
                for (MultipartUpload currentUpload : currentUploads) {
                    String currentUploadId = currentUpload.uploadId();
                    if (uploadId.equals(currentUploadId)) continue;
                    executor.execute((Runnable)ActionRunnable.run((ActionListener)listeners.acquire(), () -> this.abortMultipartUploadIfExists(currentUploadId)));
                }
            }
        }

        private void safeAbortMultipartUpload(String uploadId) {
            try {
                this.abortMultipartUploadIfExists(uploadId);
            }
            catch (Exception e) {
                logger.error("unexpected error cleaning up upload [" + uploadId + "] of [" + this.blobKey + "]", (Throwable)e);
            }
        }

        private void abortMultipartUploadIfExists(String uploadId) {
            block2: {
                try {
                    AbortMultipartUploadRequest.Builder abortMultipartUploadRequestBuilder = AbortMultipartUploadRequest.builder().bucket(this.bucket).key(this.blobKey).uploadId(uploadId);
                    S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)abortMultipartUploadRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.ABORT_MULTIPART_OBJECT, this.purpose);
                    AbortMultipartUploadRequest abortMultipartUploadRequest = (AbortMultipartUploadRequest)abortMultipartUploadRequestBuilder.build();
                    this.client.abortMultipartUpload(abortMultipartUploadRequest);
                }
                catch (SdkServiceException e) {
                    if (e.statusCode() == 404) break block2;
                    throw e;
                }
            }
        }

        private void completeMultipartUpload(String uploadId, String partETag, String existingEtag) {
            CompleteMultipartUploadRequest.Builder completeMultipartUploadRequestBuilder = CompleteMultipartUploadRequest.builder().bucket(this.bucket).key(this.blobKey).uploadId(uploadId).multipartUpload(b -> b.parts(new CompletedPart[]{(CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(1)).eTag(partETag).build()}));
            S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)completeMultipartUploadRequestBuilder, S3BlobContainer.this.blobStore, S3BlobStore.Operation.PUT_MULTIPART_OBJECT, this.purpose);
            if (S3BlobContainer.this.blobStore.supportsConditionalWrites(this.purpose)) {
                if (existingEtag == null) {
                    completeMultipartUploadRequestBuilder.ifNoneMatch("*");
                } else {
                    completeMultipartUploadRequestBuilder.ifMatch(existingEtag);
                }
            }
            CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)completeMultipartUploadRequestBuilder.build();
            this.client.completeMultipartUpload(completeMultipartUploadRequest);
        }
    }

    private record RegisterAndEtag(BytesReference registerContents, String eTag) {
        static RegisterAndEtag ABSENT = new RegisterAndEtag((BytesReference)BytesArray.EMPTY, null);
    }
}

