/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.ingest;

import java.io.IOException;
import java.lang.ref.SoftReference;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.IndicesRequest;
import org.elasticsearch.action.admin.indices.sampling.SamplingConfiguration;
import org.elasticsearch.action.admin.indices.sampling.SamplingMetadata;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.cluster.AckedBatchedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateAckListener;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.Diffable;
import org.elasticsearch.cluster.SimpleBatchedAckListenerTaskExecutor;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.project.ProjectResolver;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.cluster.service.MasterServiceTaskQueue;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.scheduler.SchedulerEngine;
import org.elasticsearch.common.scheduler.TimeValueSchedule;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.FeatureFlag;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.common.xcontent.LoggingDeprecationHandler;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.ingest.IngestDocument;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;
import org.elasticsearch.script.IngestConditionalScript;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptService;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentParseException;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xcontent.json.JsonXContent;

public class SamplingService
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    public static final boolean RANDOM_SAMPLING_FEATURE_FLAG = new FeatureFlag("random_sampling").isEnabled();
    public static final Setting<TimeValue> TTL_POLL_INTERVAL_SETTING = Setting.timeSetting("random_sampling.ttl_poll_interval", TimeValue.timeValueMinutes((long)30L), TimeValue.timeValueSeconds((long)1L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    private static final Logger logger = LogManager.getLogger(SamplingService.class);
    private static final String TTL_JOB_ID = "sampling_ttl";
    private final ScriptService scriptService;
    private final ClusterService clusterService;
    private final LongSupplier statsTimeSupplier = System::nanoTime;
    private final MasterServiceTaskQueue<UpdateSamplingConfigurationTask> updateSamplingConfigurationTaskQueue;
    private final MasterServiceTaskQueue<DeleteSampleConfigurationTask> deleteSamplingConfigurationTaskQueue;
    private static final Setting<Integer> MAX_CONFIGURATIONS_SETTING = Setting.intSetting("sampling.max_configurations", 100, 1, Setting.Property.NodeScope, Setting.Property.Dynamic);
    private final SetOnce<SchedulerEngine> scheduler = new SetOnce();
    private SchedulerEngine.Job scheduledJob;
    private volatile TimeValue pollInterval;
    private final Settings settings;
    private final Clock clock = Clock.systemUTC();
    private final Map<ProjectIndex, SoftReference<SampleInfo>> samples = new ConcurrentHashMap<ProjectIndex, SoftReference<SampleInfo>>();

    public static SamplingService create(ScriptService scriptService, ClusterService clusterService, Settings settings) {
        SamplingService samplingService = new SamplingService(scriptService, clusterService, settings);
        samplingService.configureListeners();
        return samplingService;
    }

    private SamplingService(ScriptService scriptService, ClusterService clusterService, Settings settings) {
        this.scriptService = scriptService;
        this.clusterService = clusterService;
        this.updateSamplingConfigurationTaskQueue = clusterService.createTaskQueue("update-sampling-configuration", Priority.NORMAL, new UpdateSamplingConfigurationExecutor());
        this.deleteSamplingConfigurationTaskQueue = clusterService.createTaskQueue("delete-sampling-configuration", Priority.NORMAL, new DeleteSampleConfigurationExecutor());
        this.settings = settings;
        this.pollInterval = TTL_POLL_INTERVAL_SETTING.get(settings);
    }

    private void configureListeners() {
        ClusterSettings clusterSettings = this.clusterService.getClusterSettings();
        clusterSettings.addSettingsUpdateConsumer(TTL_POLL_INTERVAL_SETTING, v -> {
            this.pollInterval = v;
            if (this.clusterService.state().nodes().isLocalNodeElectedMaster()) {
                this.maybeScheduleJob();
            }
        });
        this.addLifecycleListener(new LifecycleListener(){

            @Override
            public void afterStop() {
                SamplingService.this.cancelJob();
            }
        });
    }

    public void maybeSample(ProjectMetadata projectMetadata, IndexRequest indexRequest) {
        this.maybeSample(projectMetadata, indexRequest.index(), indexRequest, () -> {
            Map<String, Object> sourceAsMap;
            try {
                sourceAsMap = indexRequest.sourceAsMap();
            }
            catch (XContentParseException e) {
                sourceAsMap = Map.of();
                logger.trace("Invalid index request source, attempting to sample anyway");
            }
            return new IngestDocument(indexRequest.index(), indexRequest.id(), indexRequest.version(), indexRequest.routing(), indexRequest.versionType(), sourceAsMap);
        });
    }

    public void maybeSample(ProjectMetadata projectMetadata, IndexRequest indexRequest, IngestDocument ingestDocument) {
        for (String index : ingestDocument.getIndexHistory()) {
            this.maybeSample(projectMetadata, index, indexRequest, () -> ingestDocument);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeSample(ProjectMetadata projectMetadata, String indexName, IndexRequest indexRequest, Supplier<IngestDocument> ingestDocumentSupplier) {
        if (!RANDOM_SAMPLING_FEATURE_FLAG) {
            return;
        }
        long startTime = this.statsTimeSupplier.getAsLong();
        SoftReference sampleInfoReference = this.samples.compute(new ProjectIndex(projectMetadata.id(), indexName), (k, v) -> {
            if (v == null || v.get() == null) {
                SamplingConfiguration samplingConfig = this.getSamplingConfiguration(projectMetadata, indexName);
                if (samplingConfig == null) {
                    return new SoftReference<SampleInfo>(SampleInfo.NONE);
                }
                return new SoftReference<SampleInfo>(new SampleInfo(samplingConfig.maxSamples()));
            }
            return v;
        });
        SampleInfo sampleInfo = (SampleInfo)sampleInfoReference.get();
        if (sampleInfo == null || sampleInfo == SampleInfo.NONE) {
            return;
        }
        SampleStats stats = sampleInfo.stats;
        stats.potentialSamples.increment();
        try {
            if (sampleInfo.isFull) {
                stats.samplesRejectedForMaxSamplesExceeded.increment();
                return;
            }
            SamplingConfiguration samplingConfig = this.getSamplingConfiguration(projectMetadata, indexName);
            if (samplingConfig == null) {
                return;
            }
            if (sampleInfo.getSizeInBytes() + (long)indexRequest.source().length() > samplingConfig.maxSize().getBytes()) {
                stats.samplesRejectedForSize.increment();
                return;
            }
            if (Math.random() >= samplingConfig.rate()) {
                stats.samplesRejectedForRate.increment();
                return;
            }
            String condition = samplingConfig.condition();
            if (condition != null && (sampleInfo.script == null || sampleInfo.factory == null)) {
                long compileScriptStartTime = this.statsTimeSupplier.getAsLong();
                try {
                    if (sampleInfo.compilationFailed) {
                        stats.samplesRejectedForException.increment();
                        return;
                    }
                    Script script = SamplingService.getScript(condition);
                    sampleInfo.setScript(script, this.scriptService.compile(script, IngestConditionalScript.CONTEXT));
                }
                catch (Exception e) {
                    sampleInfo.compilationFailed = true;
                    throw e;
                }
                finally {
                    stats.timeCompilingConditionInNanos.add(this.statsTimeSupplier.getAsLong() - compileScriptStartTime);
                }
            }
            if (condition != null && !this.evaluateCondition(ingestDocumentSupplier, sampleInfo.script, sampleInfo.factory, sampleInfo.stats)) {
                stats.samplesRejectedForCondition.increment();
                return;
            }
            RawDocument sample = this.getRawDocumentForIndexRequest(indexName, indexRequest);
            if (sampleInfo.offer(sample)) {
                stats.samples.increment();
                logger.trace("Sampling " + String.valueOf(indexRequest));
            } else {
                stats.samplesRejectedForMaxSamplesExceeded.increment();
            }
        }
        catch (Exception e) {
            stats.samplesRejectedForException.increment();
            stats.lastException = e;
            logger.debug("Error performing sampling for " + indexName, (Throwable)e);
        }
        finally {
            stats.timeSamplingInNanos.add(this.statsTimeSupplier.getAsLong() - startTime);
        }
    }

    public SamplingConfiguration getSamplingConfiguration(ProjectMetadata projectMetadata, String indexName) {
        SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
        if (samplingMetadata == null) {
            return null;
        }
        return samplingMetadata.getIndexToSamplingConfigMap().get(indexName);
    }

    public List<RawDocument> getLocalSample(ProjectId projectId, String index) {
        SoftReference<SampleInfo> sampleInfoReference = this.samples.get(new ProjectIndex(projectId, index));
        SampleInfo sampleInfo = sampleInfoReference == null ? null : sampleInfoReference.get();
        return sampleInfo == null ? List.of() : Arrays.stream(sampleInfo.getRawDocuments()).filter(Objects::nonNull).toList();
    }

    public SampleStats getLocalSampleStats(ProjectId projectId, String index) {
        SoftReference<SampleInfo> sampleInfoReference = this.samples.get(new ProjectIndex(projectId, index));
        if (sampleInfoReference == null) {
            return new SampleStats();
        }
        SampleInfo sampleInfo = sampleInfoReference.get();
        return sampleInfo == null ? new SampleStats() : sampleInfo.stats;
    }

    public static void throwIndexNotFoundExceptionIfNotDataStreamOrIndex(IndexNameExpressionResolver indexNameExpressionResolver, ProjectResolver projectResolver, ClusterState state, IndicesRequest request) {
        assert (request.indices().length == 1) : "Expected IndicesRequest to have a single index but found " + request.indices().length;
        assert (request.includeDataStreams()) : "Expected IndicesRequest to include data streams but it did not";
        boolean isDataStream = projectResolver.getProjectMetadata(state).dataStreams().containsKey(request.indices()[0]);
        if (!isDataStream) {
            indexNameExpressionResolver.concreteIndexNames(state, request);
        }
    }

    public boolean atLeastOneSampleConfigured(ProjectMetadata projectMetadata) {
        if (RANDOM_SAMPLING_FEATURE_FLAG) {
            SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
            return samplingMetadata != null && !samplingMetadata.getIndexToSamplingConfigMap().isEmpty();
        }
        return false;
    }

    public void updateSampleConfiguration(ProjectId projectId, String index, SamplingConfiguration samplingConfiguration, TimeValue masterNodeTimeout, TimeValue ackTimeout, ActionListener<AcknowledgedResponse> listener) {
        ClusterState clusterState = this.clusterService.state();
        boolean maxConfigLimitBreached = SamplingService.checkMaxConfigLimitBreached(projectId, index, clusterState);
        if (maxConfigLimitBreached) {
            Integer maxConfigurations = MAX_CONFIGURATIONS_SETTING.get(clusterState.getMetadata().settings());
            listener.onFailure(new IllegalStateException("Cannot add sampling configuration for index [" + index + "]. Maximum number of sampling configurations (" + maxConfigurations + ") already reached."));
            return;
        }
        this.updateSamplingConfigurationTaskQueue.submitTask("Updating Sampling Configuration", new UpdateSamplingConfigurationTask(projectId, index, samplingConfiguration, ackTimeout, listener), masterNodeTimeout);
    }

    public void deleteSampleConfiguration(ProjectId projectId, String index, TimeValue masterNodeTimeout, TimeValue ackTimeout, ActionListener<AcknowledgedResponse> listener) {
        this.deleteSamplingConfigurationTaskQueue.submitTask("deleting sampling configuration for index " + index, new DeleteSampleConfigurationTask(projectId, index, ackTimeout, listener), masterNodeTimeout);
    }

    private void deleteSampleConfiguration(ProjectId projectId, String index) {
        this.deleteSampleConfiguration(projectId, index, TimeValue.MAX_VALUE, TimeValue.MAX_VALUE, ActionListener.wrap(response -> {
            if (response.isAcknowledged()) {
                logger.debug("Deleted sampling configuration for {}", new Object[]{index});
            } else {
                logger.warn("Deletion of sampling configuration for {} not acknowledged", new Object[]{index});
            }
        }, e -> logger.warn("Failed to delete sample configuration for " + index, (Throwable)e)));
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!RANDOM_SAMPLING_FEATURE_FLAG) {
            return;
        }
        boolean isMaster = event.localNodeMaster();
        boolean wasMaster = event.previousState().nodes().isLocalNodeElectedMaster();
        if (wasMaster != isMaster) {
            if (isMaster) {
                this.maybeScheduleJob();
            } else {
                this.cancelJob();
            }
        }
        if (!isMaster && this.samples.isEmpty()) {
            return;
        }
        if (event.metadataChanged()) {
            Set<ProjectId> allProjectIds = Sets.union(event.state().metadata().projects().keySet(), event.previousState().metadata().projects().keySet());
            for (ProjectId projectId : allProjectIds) {
                this.maybeRemoveStaleSamples(event, projectId);
                if (!isMaster) continue;
                this.maybeDeleteSamplingConfigurations(event, projectId);
            }
        }
    }

    private void maybeRemoveStaleSamples(ClusterChangedEvent event, ProjectId projectId) {
        if (!this.samples.isEmpty() && event.customMetadataChanged(projectId, "sampling")) {
            Map oldSampleConfigsMap = Optional.ofNullable(event.previousState().metadata().projects().get(projectId)).map(p -> (SamplingMetadata)p.custom("sampling")).map(SamplingMetadata::getIndexToSamplingConfigMap).orElse(Map.of());
            Map newSampleConfigsMap = Optional.ofNullable(event.state().metadata().projects().get(projectId)).map(p -> (SamplingMetadata)p.custom("sampling")).map(SamplingMetadata::getIndexToSamplingConfigMap).orElse(Map.of());
            HashSet indicesWithRemovedConfigs = new HashSet(oldSampleConfigsMap.keySet());
            indicesWithRemovedConfigs.removeAll(newSampleConfigsMap.keySet());
            for (String string : indicesWithRemovedConfigs) {
                logger.debug("Removing sample info for {} because its configuration has been removed", new Object[]{string});
                this.samples.remove(new ProjectIndex(projectId, string));
            }
            for (Map.Entry entry : newSampleConfigsMap.entrySet()) {
                String indexName = (String)entry.getKey();
                if (oldSampleConfigsMap.containsKey(indexName) && !((SamplingConfiguration)entry.getValue()).equals(oldSampleConfigsMap.get(indexName))) {
                    logger.debug("Removing sample info for {} because its configuration has changed", new Object[]{indexName});
                    this.samples.remove(new ProjectIndex(projectId, indexName));
                    continue;
                }
                if (oldSampleConfigsMap.containsKey(indexName) || !this.samples.containsKey(new ProjectIndex(projectId, indexName))) continue;
                logger.debug("Removing sample info for {} because its configuration has been created", new Object[]{indexName});
                this.samples.remove(new ProjectIndex(projectId, indexName));
            }
        }
    }

    private void maybeDeleteSamplingConfigurations(ClusterChangedEvent event, ProjectId projectId) {
        SamplingConfiguration samplingConfiguration;
        Diffable<IndexMetadata> current;
        ProjectMetadata currentProject = event.state().metadata().projects().get(projectId);
        ProjectMetadata previousProject = event.previousState().metadata().projects().get(projectId);
        if (currentProject == null || previousProject == null) {
            return;
        }
        if (currentProject.indices() != previousProject.indices()) {
            for (IndexMetadata index : previousProject.indices().values()) {
                current = currentProject.index(index.getIndex());
                if (current != null) continue;
                String indexName = index.getIndex().getName();
                samplingConfiguration = this.getSamplingConfiguration(event.state().projectState(projectId).metadata(), indexName);
                if (samplingConfiguration == null) continue;
                logger.debug("Deleting sample configuration for {} because the index has been deleted", new Object[]{indexName});
                this.deleteSampleConfiguration(projectId, indexName);
            }
        }
        if (currentProject.dataStreams() != previousProject.dataStreams()) {
            for (DataStream dataStream : previousProject.dataStreams().values()) {
                current = currentProject.dataStreams().get(dataStream.getName());
                if (current != null) continue;
                String dataStreamName = dataStream.getName();
                samplingConfiguration = this.getSamplingConfiguration(event.state().projectState(projectId).metadata(), dataStreamName);
                if (samplingConfiguration == null) continue;
                logger.debug("Deleting sample configuration for {} because the data stream has been deleted", new Object[]{dataStreamName});
                this.deleteSampleConfiguration(projectId, dataStreamName);
            }
        }
    }

    private void maybeScheduleJob() {
        if (this.isClusterServiceStoppedOrClosed()) {
            return;
        }
        if (this.scheduler.get() == null) {
            this.scheduler.set((Object)new SchedulerEngine(this.settings, this.clock));
            ((SchedulerEngine)this.scheduler.get()).register(this);
        }
        this.scheduledJob = new SchedulerEngine.Job(TTL_JOB_ID, new TimeValueSchedule(this.pollInterval));
        ((SchedulerEngine)this.scheduler.get()).add(this.scheduledJob);
    }

    private void cancelJob() {
        if (this.scheduler.get() != null) {
            ((SchedulerEngine)this.scheduler.get()).remove(TTL_JOB_ID);
            this.scheduledJob = null;
        }
    }

    private boolean isClusterServiceStoppedOrClosed() {
        Lifecycle.State state = this.clusterService.lifecycleState();
        return state == Lifecycle.State.STOPPED || state == Lifecycle.State.CLOSED;
    }

    private boolean evaluateCondition(Supplier<IngestDocument> ingestDocumentSupplier, Script script, IngestConditionalScript.Factory factory, SampleStats stats) {
        long conditionStartTime = this.statsTimeSupplier.getAsLong();
        boolean passedCondition = factory.newInstance(script.getParams(), ingestDocumentSupplier.get().getUnmodifiableSourceAndMetadata()).execute();
        stats.timeEvaluatingConditionInNanos.add(this.statsTimeSupplier.getAsLong() - conditionStartTime);
        return passedCondition;
    }

    private static Script getScript(String conditional) throws IOException {
        logger.debug("Parsing script for conditional " + conditional);
        try (XContentBuilder builder = XContentBuilder.builder((XContent)JsonXContent.jsonXContent).map(Map.of("source", conditional));){
            Script script;
            block12: {
                XContentParser parser = XContentHelper.createParserNotCompressed(LoggingDeprecationHandler.XCONTENT_PARSER_CONFIG, BytesReference.bytes(builder), XContentType.JSON);
                try {
                    script = Script.parse(parser);
                    if (parser == null) break block12;
                }
                catch (Throwable throwable) {
                    if (parser != null) {
                        try {
                            parser.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                parser.close();
            }
            return script;
        }
    }

    private static boolean checkMaxConfigLimitBreached(ProjectId projectId, String index, ClusterState currentState) {
        Metadata currentMetadata = currentState.metadata();
        ProjectMetadata projectMetadata = currentMetadata.getProject(projectId);
        if (projectMetadata != null) {
            SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
            Map<Object, Object> existingConfigs = samplingMetadata != null ? samplingMetadata.getIndexToSamplingConfigMap() : Map.of();
            boolean isUpdate = existingConfigs.containsKey(index);
            Integer maxConfigurations = MAX_CONFIGURATIONS_SETTING.get(currentMetadata.settings());
            if (!isUpdate && existingConfigs.size() >= maxConfigurations) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        logger.debug("job triggered: {}, {}, {}", new Object[]{event.jobName(), event.scheduledTime(), event.triggeredTime()});
        this.checkTTLs();
    }

    @Override
    protected void doStart() {
    }

    @Override
    protected void doStop() {
        this.clusterService.removeListener(this);
        logger.debug("Sampling service is stopping.");
    }

    @Override
    protected void doClose() throws IOException {
        logger.debug("Sampling service is closing.");
        SchedulerEngine engine = (SchedulerEngine)this.scheduler.get();
        if (engine != null) {
            engine.stop();
        }
    }

    private void checkTTLs() {
        long now = this.clock.instant().toEpochMilli();
        for (ProjectMetadata projectMetadata : this.clusterService.state().metadata().projects().values()) {
            SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
            if (samplingMetadata == null) continue;
            for (Map.Entry<String, SamplingConfiguration> entry : samplingMetadata.getIndexToSamplingConfigMap().entrySet()) {
                SamplingConfiguration samplingConfiguration = entry.getValue();
                if (samplingConfiguration.creationTime() + samplingConfiguration.timeToLive().millis() >= now) continue;
                String indexName = entry.getKey();
                logger.debug("Deleting configuration for {} created at {} because it is older than {} now at {}", new Object[]{indexName, ZonedDateTime.ofInstant(Instant.ofEpochMilli(samplingConfiguration.creationTime()), ZoneOffset.UTC), samplingConfiguration.timeToLive(), ZonedDateTime.ofInstant(Instant.ofEpochMilli(now), ZoneOffset.UTC)});
                this.deleteSampleConfiguration(projectMetadata.id(), indexName);
            }
        }
    }

    private RawDocument getRawDocumentForIndexRequest(String indexName, IndexRequest indexRequest) {
        BytesReference sourceReference = indexRequest.source();
        assert (sourceReference != null) : "Cannot sample an IndexRequest with no source";
        byte[] source = sourceReference.array();
        byte[] sourceCopy = new byte[sourceReference.length()];
        System.arraycopy(source, sourceReference.arrayOffset(), sourceCopy, 0, sourceReference.length());
        return new RawDocument(indexName, sourceCopy, indexRequest.getContentType());
    }

    static class UpdateSamplingConfigurationExecutor
    extends SimpleBatchedAckListenerTaskExecutor<UpdateSamplingConfigurationTask> {
        private static final Logger logger = LogManager.getLogger(UpdateSamplingConfigurationExecutor.class);

        UpdateSamplingConfigurationExecutor() {
        }

        @Override
        public Tuple<ClusterState, ClusterStateAckListener> executeTask(UpdateSamplingConfigurationTask updateSamplingConfigurationTask, ClusterState clusterState) {
            logger.debug("Updating sampling configuration for index [{}] with rate [{}], maxSamples [{}], maxSize [{}], timeToLive [{}], condition [{}], creationTime [{}]", new Object[]{updateSamplingConfigurationTask.indexName, updateSamplingConfigurationTask.samplingConfiguration.rate(), updateSamplingConfigurationTask.samplingConfiguration.maxSamples(), updateSamplingConfigurationTask.samplingConfiguration.maxSize(), updateSamplingConfigurationTask.samplingConfiguration.timeToLive(), updateSamplingConfigurationTask.samplingConfiguration.condition(), updateSamplingConfigurationTask.samplingConfiguration.creationTime()});
            Metadata metadata = clusterState.getMetadata();
            ProjectMetadata projectMetadata = metadata.getProject(updateSamplingConfigurationTask.projectId);
            SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
            boolean isNewConfiguration = samplingMetadata == null;
            int existingConfigCount = isNewConfiguration ? 0 : samplingMetadata.getIndexToSamplingConfigMap().size();
            logger.trace("Current sampling metadata state: {} (number of existing configurations: {})", new Object[]{isNewConfiguration ? "null" : "exists", existingConfigCount});
            HashMap<String, SamplingConfiguration> updatedConfigMap = new HashMap<String, SamplingConfiguration>();
            if (samplingMetadata != null) {
                updatedConfigMap.putAll(samplingMetadata.getIndexToSamplingConfigMap());
            }
            boolean isUpdate = updatedConfigMap.containsKey(updateSamplingConfigurationTask.indexName);
            Integer maxConfigurations = MAX_CONFIGURATIONS_SETTING.get(metadata.settings());
            boolean maxConfigLimitBreached = SamplingService.checkMaxConfigLimitBreached(updateSamplingConfigurationTask.projectId, updateSamplingConfigurationTask.indexName, clusterState);
            if (maxConfigLimitBreached) {
                throw new IllegalStateException("Cannot add sampling configuration for index [" + updateSamplingConfigurationTask.indexName + "]. Maximum number of sampling configurations (" + maxConfigurations + ") already reached.");
            }
            updatedConfigMap.put(updateSamplingConfigurationTask.indexName, updateSamplingConfigurationTask.samplingConfiguration);
            logger.trace("{} sampling configuration for index [{}], total configurations after update: {}", new Object[]{isUpdate ? "Updated" : "Added", updateSamplingConfigurationTask.indexName, updatedConfigMap.size()});
            SamplingMetadata newSamplingMetadata = new SamplingMetadata(updatedConfigMap);
            ProjectMetadata.Builder projectMetadataBuilder = ProjectMetadata.builder(projectMetadata);
            projectMetadataBuilder.putCustom("sampling", newSamplingMetadata);
            ClusterState updatedClusterState = ClusterState.builder(clusterState).putProjectMetadata(projectMetadataBuilder).build();
            logger.debug("Successfully {} sampling configuration for index [{}]", new Object[]{isUpdate ? "updated" : "created", updateSamplingConfigurationTask.indexName});
            return new Tuple((Object)updatedClusterState, (Object)updateSamplingConfigurationTask);
        }
    }

    static final class DeleteSampleConfigurationExecutor
    extends SimpleBatchedAckListenerTaskExecutor<DeleteSampleConfigurationTask> {
        private static final Logger logger = LogManager.getLogger(DeleteSampleConfigurationExecutor.class);

        DeleteSampleConfigurationExecutor() {
        }

        @Override
        public Tuple<ClusterState, ClusterStateAckListener> executeTask(DeleteSampleConfigurationTask deleteSampleConfigurationTask, ClusterState clusterState) {
            Metadata metadata = clusterState.getMetadata();
            ProjectMetadata projectMetadata = metadata.getProject(deleteSampleConfigurationTask.projectId);
            SamplingMetadata samplingMetadata = (SamplingMetadata)projectMetadata.custom("sampling");
            Map<String, SamplingConfiguration> oldConfigMap = DeleteSampleConfigurationExecutor.validateConfigExists(deleteSampleConfigurationTask.indexName, samplingMetadata);
            logger.debug("Deleting sampling configuration for index [{}]", new Object[]{deleteSampleConfigurationTask.indexName});
            HashMap<String, SamplingConfiguration> updatedConfigMap = new HashMap<String, SamplingConfiguration>(oldConfigMap);
            updatedConfigMap.remove(deleteSampleConfigurationTask.indexName);
            SamplingMetadata newSamplingMetadata = new SamplingMetadata(updatedConfigMap);
            ProjectMetadata.Builder projectMetadataBuilder = ProjectMetadata.builder(projectMetadata);
            projectMetadataBuilder.putCustom("sampling", newSamplingMetadata);
            ClusterState updatedClusterState = ClusterState.builder(clusterState).putProjectMetadata(projectMetadataBuilder).build();
            logger.debug("Successfully deleted sampling configuration for index [{}], total configurations after deletion: [{}]", new Object[]{deleteSampleConfigurationTask.indexName, updatedConfigMap.size()});
            return new Tuple((Object)updatedClusterState, (Object)deleteSampleConfigurationTask);
        }

        private static Map<String, SamplingConfiguration> validateConfigExists(String indexName, SamplingMetadata samplingMetadata) {
            String exceptionMessage = "provided index [" + indexName + "] has no sampling configuration";
            if (samplingMetadata == null) {
                throw new ResourceNotFoundException(exceptionMessage, new Object[0]);
            }
            Map<String, SamplingConfiguration> configMap = samplingMetadata.getIndexToSamplingConfigMap();
            if (configMap == null || !configMap.containsKey(indexName)) {
                throw new ResourceNotFoundException(exceptionMessage, new Object[0]);
            }
            return configMap;
        }
    }

    private record ProjectIndex(ProjectId projectId, String indexName) {
    }

    private static final class SampleInfo {
        public static final SampleInfo NONE = new SampleInfo(0);
        private final RawDocument[] rawDocuments;
        private final AtomicInteger rawDocumentsIndex = new AtomicInteger(-1);
        private volatile Tuple<Integer, Long> sizeInBytesAtIndex = Tuple.tuple((Object)-1, (Object)0L);
        private final SampleStats stats;
        private volatile Script script;
        private volatile IngestConditionalScript.Factory factory;
        private volatile boolean compilationFailed = false;
        private volatile boolean isFull = false;

        SampleInfo(int maxSamples) {
            this.rawDocuments = new RawDocument[maxSamples];
            this.stats = new SampleStats();
        }

        public RawDocument[] getRawDocuments() {
            return this.rawDocuments;
        }

        public long getSizeInBytes() {
            Tuple<Integer, Long> knownIndexAndSize = this.sizeInBytesAtIndex;
            int knownSizeIndex = (Integer)knownIndexAndSize.v1();
            long knownSize = (Long)knownIndexAndSize.v2();
            int currentRawDocumentsIndex = Math.min(this.rawDocumentsIndex.get(), this.rawDocuments.length - 1);
            if (currentRawDocumentsIndex == knownSizeIndex) {
                return knownSize;
            }
            long size = knownSize;
            boolean anyNulls = false;
            for (int i = knownSizeIndex + 1; i <= currentRawDocumentsIndex; ++i) {
                RawDocument rawDocument = this.rawDocuments[i];
                if (rawDocument == null) {
                    anyNulls = true;
                    continue;
                }
                size += this.rawDocuments[i].getSizeInBytes();
            }
            if (!anyNulls) {
                this.sizeInBytesAtIndex = Tuple.tuple((Object)currentRawDocumentsIndex, (Object)size);
            }
            return size;
        }

        public boolean offer(RawDocument rawDocument) {
            int index = this.rawDocumentsIndex.incrementAndGet();
            if (index < this.rawDocuments.length) {
                this.rawDocuments[index] = rawDocument;
                if (index == this.rawDocuments.length - 1) {
                    this.isFull = true;
                }
                return true;
            }
            return false;
        }

        void setScript(Script script, IngestConditionalScript.Factory factory) {
            this.script = script;
            this.factory = factory;
        }
    }

    public static final class SampleStats
    implements Writeable,
    ToXContent {
        final LongAdder samples = new LongAdder();
        final LongAdder potentialSamples = new LongAdder();
        final LongAdder samplesRejectedForMaxSamplesExceeded = new LongAdder();
        final LongAdder samplesRejectedForCondition = new LongAdder();
        final LongAdder samplesRejectedForRate = new LongAdder();
        final LongAdder samplesRejectedForException = new LongAdder();
        final LongAdder samplesRejectedForSize = new LongAdder();
        final LongAdder timeSamplingInNanos = new LongAdder();
        final LongAdder timeEvaluatingConditionInNanos = new LongAdder();
        final LongAdder timeCompilingConditionInNanos = new LongAdder();
        Exception lastException = null;

        public SampleStats() {
        }

        public SampleStats(SampleStats other) {
            SampleStats.addAllFields(other, this);
        }

        public SampleStats(long samples, long potentialSamples, long samplesRejectedForMaxSamplesExceeded, long samplesRejectedForCondition, long samplesRejectedForRate, long samplesRejectedForException, long samplesRejectedForSize, TimeValue timeSampling, TimeValue timeEvaluatingCondition, TimeValue timeCompilingCondition, Exception lastException) {
            this.samples.add(samples);
            this.potentialSamples.add(potentialSamples);
            this.samplesRejectedForMaxSamplesExceeded.add(samplesRejectedForMaxSamplesExceeded);
            this.samplesRejectedForCondition.add(samplesRejectedForCondition);
            this.samplesRejectedForRate.add(samplesRejectedForRate);
            this.samplesRejectedForException.add(samplesRejectedForException);
            this.samplesRejectedForSize.add(samplesRejectedForSize);
            this.timeSamplingInNanos.add(timeSampling.nanos());
            this.timeEvaluatingConditionInNanos.add(timeEvaluatingCondition.nanos());
            this.timeCompilingConditionInNanos.add(timeCompilingCondition.nanos());
            this.lastException = lastException;
        }

        public SampleStats(StreamInput in) throws IOException {
            this.potentialSamples.add(in.readLong());
            this.samplesRejectedForMaxSamplesExceeded.add(in.readLong());
            this.samplesRejectedForCondition.add(in.readLong());
            this.samplesRejectedForRate.add(in.readLong());
            this.samplesRejectedForException.add(in.readLong());
            this.samplesRejectedForSize.add(in.readLong());
            this.samples.add(in.readLong());
            this.timeSamplingInNanos.add(in.readLong());
            this.timeEvaluatingConditionInNanos.add(in.readLong());
            this.timeCompilingConditionInNanos.add(in.readLong());
            this.lastException = in.readBoolean() ? in.readException() : null;
        }

        public long getSamples() {
            return this.samples.longValue();
        }

        public long getPotentialSamples() {
            return this.potentialSamples.longValue();
        }

        public long getSamplesRejectedForMaxSamplesExceeded() {
            return this.samplesRejectedForMaxSamplesExceeded.longValue();
        }

        public long getSamplesRejectedForCondition() {
            return this.samplesRejectedForCondition.longValue();
        }

        public long getSamplesRejectedForRate() {
            return this.samplesRejectedForRate.longValue();
        }

        public long getSamplesRejectedForException() {
            return this.samplesRejectedForException.longValue();
        }

        public long getSamplesRejectedForSize() {
            return this.samplesRejectedForSize.longValue();
        }

        public TimeValue getTimeSampling() {
            return TimeValue.timeValueNanos((long)this.timeSamplingInNanos.longValue());
        }

        public TimeValue getTimeEvaluatingCondition() {
            return TimeValue.timeValueNanos((long)this.timeEvaluatingConditionInNanos.longValue());
        }

        public TimeValue getTimeCompilingCondition() {
            return TimeValue.timeValueNanos((long)this.timeCompilingConditionInNanos.longValue());
        }

        public Exception getLastException() {
            return this.lastException;
        }

        public String toString() {
            return "potential_samples: " + String.valueOf(this.potentialSamples) + ", samples_rejected_for_max_samples_exceeded: " + String.valueOf(this.samplesRejectedForMaxSamplesExceeded) + ", samples_rejected_for_condition: " + String.valueOf(this.samplesRejectedForCondition) + ", samples_rejected_for_rate: " + String.valueOf(this.samplesRejectedForRate) + ", samples_rejected_for_exception: " + String.valueOf(this.samplesRejectedForException) + ", samples_accepted: " + String.valueOf(this.samples) + ", time_sampling: " + String.valueOf(TimeValue.timeValueNanos((long)this.timeSamplingInNanos.longValue())) + ", time_evaluating_condition: " + String.valueOf(TimeValue.timeValueNanos((long)this.timeEvaluatingConditionInNanos.longValue())) + ", time_compiling_condition: " + String.valueOf(TimeValue.timeValueNanos((long)this.timeCompilingConditionInNanos.longValue()));
        }

        public SampleStats combine(SampleStats other) {
            SampleStats result = new SampleStats(this);
            SampleStats.addAllFields(other, result);
            return result;
        }

        private static void addAllFields(SampleStats source, SampleStats dest) {
            dest.potentialSamples.add(source.potentialSamples.longValue());
            dest.samplesRejectedForMaxSamplesExceeded.add(source.samplesRejectedForMaxSamplesExceeded.longValue());
            dest.samplesRejectedForCondition.add(source.samplesRejectedForCondition.longValue());
            dest.samplesRejectedForRate.add(source.samplesRejectedForRate.longValue());
            dest.samplesRejectedForException.add(source.samplesRejectedForException.longValue());
            dest.samplesRejectedForSize.add(source.samplesRejectedForSize.longValue());
            dest.samples.add(source.samples.longValue());
            dest.timeSamplingInNanos.add(source.timeSamplingInNanos.longValue());
            dest.timeEvaluatingConditionInNanos.add(source.timeEvaluatingConditionInNanos.longValue());
            dest.timeCompilingConditionInNanos.add(source.timeCompilingConditionInNanos.longValue());
            if (dest.lastException == null) {
                dest.lastException = source.lastException;
            }
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject();
            builder.field("potential_samples", this.potentialSamples.longValue());
            builder.field("samples_rejected_for_max_samples_exceeded", this.samplesRejectedForMaxSamplesExceeded.longValue());
            builder.field("samples_rejected_for_condition", this.samplesRejectedForCondition.longValue());
            builder.field("samples_rejected_for_rate", this.samplesRejectedForRate.longValue());
            builder.field("samples_rejected_for_exception", this.samplesRejectedForException.longValue());
            builder.field("samples_rejected_for_size", this.samplesRejectedForSize.longValue());
            builder.field("samples_accepted", this.samples.longValue());
            builder.humanReadableField("time_sampling_millis", "time_sampling", (Object)TimeValue.timeValueNanos((long)this.timeSamplingInNanos.longValue()));
            builder.humanReadableField("time_evaluating_condition_millis", "time_evaluating_condition", (Object)TimeValue.timeValueNanos((long)this.timeEvaluatingConditionInNanos.longValue()));
            builder.humanReadableField("time_compiling_condition_millis", "time_compiling_condition", (Object)TimeValue.timeValueNanos((long)this.timeCompilingConditionInNanos.longValue()));
            if (this.lastException != null) {
                Throwable unwrapped = ExceptionsHelper.unwrapCause(this.lastException);
                builder.startObject("last_exception");
                builder.field("type", ElasticsearchException.getExceptionName(unwrapped));
                builder.field("message", unwrapped.getMessage());
                builder.field("stack_trace", ExceptionsHelper.limitedStackTrace(unwrapped, 5));
                builder.endObject();
            }
            builder.endObject();
            return builder;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeLong(this.potentialSamples.longValue());
            out.writeLong(this.samplesRejectedForMaxSamplesExceeded.longValue());
            out.writeLong(this.samplesRejectedForCondition.longValue());
            out.writeLong(this.samplesRejectedForRate.longValue());
            out.writeLong(this.samplesRejectedForException.longValue());
            out.writeLong(this.samplesRejectedForSize.longValue());
            out.writeLong(this.samples.longValue());
            out.writeLong(this.timeSamplingInNanos.longValue());
            out.writeLong(this.timeEvaluatingConditionInNanos.longValue());
            out.writeLong(this.timeCompilingConditionInNanos.longValue());
            if (this.lastException == null) {
                out.writeBoolean(false);
            } else {
                out.writeBoolean(true);
                out.writeException(this.lastException);
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SampleStats that = (SampleStats)o;
            if (this.samples.longValue() != that.samples.longValue()) {
                return false;
            }
            if (this.potentialSamples.longValue() != that.potentialSamples.longValue()) {
                return false;
            }
            if (this.samplesRejectedForMaxSamplesExceeded.longValue() != that.samplesRejectedForMaxSamplesExceeded.longValue()) {
                return false;
            }
            if (this.samplesRejectedForCondition.longValue() != that.samplesRejectedForCondition.longValue()) {
                return false;
            }
            if (this.samplesRejectedForRate.longValue() != that.samplesRejectedForRate.longValue()) {
                return false;
            }
            if (this.samplesRejectedForException.longValue() != that.samplesRejectedForException.longValue()) {
                return false;
            }
            if (this.samplesRejectedForSize.longValue() != that.samplesRejectedForSize.longValue()) {
                return false;
            }
            if (this.timeSamplingInNanos.longValue() != that.timeSamplingInNanos.longValue()) {
                return false;
            }
            if (this.timeEvaluatingConditionInNanos.longValue() != that.timeEvaluatingConditionInNanos.longValue()) {
                return false;
            }
            if (this.timeCompilingConditionInNanos.longValue() != that.timeCompilingConditionInNanos.longValue()) {
                return false;
            }
            return this.exceptionsAreEqual(this.lastException, that.lastException);
        }

        private boolean exceptionsAreEqual(Exception e1, Exception e2) {
            if (e1 == null && e2 == null) {
                return true;
            }
            if (e1 == null || e2 == null) {
                return false;
            }
            return e1.getClass().equals(e2.getClass()) && e1.getMessage().equals(e2.getMessage());
        }

        public int hashCode() {
            return Objects.hash(this.samples.longValue(), this.potentialSamples.longValue(), this.samplesRejectedForMaxSamplesExceeded.longValue(), this.samplesRejectedForCondition.longValue(), this.samplesRejectedForRate.longValue(), this.samplesRejectedForException.longValue(), this.samplesRejectedForSize.longValue(), this.timeSamplingInNanos.longValue(), this.timeEvaluatingConditionInNanos.longValue(), this.timeCompilingConditionInNanos.longValue()) + this.hashException(this.lastException);
        }

        private int hashException(Exception e) {
            if (e == null) {
                return 0;
            }
            return Objects.hash(e.getClass(), e.getMessage());
        }

        public SampleStats adjustForMaxSize(int maxSize) {
            long actualSamples = this.samples.longValue();
            if (actualSamples > (long)maxSize) {
                SampleStats adjusted = new SampleStats().combine(this);
                adjusted.samples.add((long)maxSize - actualSamples);
                adjusted.samplesRejectedForMaxSamplesExceeded.add(actualSamples - (long)maxSize);
                return adjusted;
            }
            return this;
        }
    }

    public record RawDocument(String indexName, byte[] source, XContentType contentType) implements Writeable
    {
        public RawDocument(StreamInput in) throws IOException {
            this(in.readString(), in.readByteArray(), in.readEnum(XContentType.class));
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.indexName);
            out.writeByteArray(this.source);
            XContentHelper.writeTo(out, this.contentType);
        }

        public long getSizeInBytes() {
            return this.indexName.length() + this.source.length;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RawDocument rawDocument = (RawDocument)o;
            return Objects.equals(this.indexName, rawDocument.indexName) && Arrays.equals(this.source, rawDocument.source) && this.contentType == rawDocument.contentType;
        }

        @Override
        public int hashCode() {
            int result = Objects.hash(this.indexName, this.contentType);
            result = 31 * result + Arrays.hashCode(this.source);
            return result;
        }
    }

    static class UpdateSamplingConfigurationTask
    extends AckedBatchedClusterStateUpdateTask {
        private final ProjectId projectId;
        private final String indexName;
        private final SamplingConfiguration samplingConfiguration;

        UpdateSamplingConfigurationTask(ProjectId projectId, String indexName, SamplingConfiguration samplingConfiguration, TimeValue ackTimeout, ActionListener<AcknowledgedResponse> listener) {
            super(ackTimeout, listener);
            this.projectId = projectId;
            this.indexName = indexName;
            this.samplingConfiguration = samplingConfiguration;
        }
    }

    static final class DeleteSampleConfigurationTask
    extends AckedBatchedClusterStateUpdateTask {
        private final ProjectId projectId;
        private final String indexName;

        DeleteSampleConfigurationTask(ProjectId projectId, String indexName, TimeValue ackTimeout, ActionListener<AcknowledgedResponse> listener) {
            super(ackTimeout, listener);
            this.projectId = projectId;
            this.indexName = indexName;
        }
    }
}

