/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.api.client.http.HttpResponse;
import com.google.api.core.ApiClock;
import com.google.api.gax.retrying.ResultRetryAlgorithm;
import com.google.api.gax.retrying.RetrySettings;
import com.google.cloud.BaseService;
import com.google.cloud.RetryHelper;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.StorageException;
import java.io.Closeable;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.repositories.blobstore.RequestedRangeNotSatisfiedException;
import org.elasticsearch.repositories.gcs.MeteredStorage;
import org.elasticsearch.rest.RestStatus;

class GoogleCloudStorageRetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(GoogleCloudStorageRetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final OperationPurpose purpose;
    private final MeteredStorage client;
    private final BlobId blobId;
    private final long start;
    private final long end;
    private final int maxAttempts;
    private InputStream currentStream;
    private int attempt = 1;
    private List<StorageException> failures = new ArrayList<StorageException>(10);
    private long currentOffset;
    private boolean closed;
    private Long lastGeneration;

    GoogleCloudStorageRetryingInputStream(OperationPurpose purpose, MeteredStorage client, BlobId blobId) throws IOException {
        this(purpose, client, blobId, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    GoogleCloudStorageRetryingInputStream(OperationPurpose purpose, MeteredStorage client, BlobId blobId, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.purpose = purpose;
        this.client = client;
        this.blobId = blobId;
        this.start = start;
        this.end = end;
        this.maxAttempts = client.getOptions().getRetrySettings().getMaxAttempts();
        this.currentStream = this.openStream();
    }

    private InputStream openStream() throws IOException {
        try {
            try {
                return (InputStream)RetryHelper.runWithRetries(() -> {
                    try {
                        MeteredStorage.MeteredObjectsGetRequest meteredGet = this.client.meteredObjectsGet(this.purpose, this.blobId.getBucket(), this.blobId.getName());
                        meteredGet.setReturnRawInputStream(true);
                        if (this.lastGeneration != null) {
                            meteredGet.setGeneration(this.lastGeneration);
                        }
                        if ((this.currentOffset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) && meteredGet.getRequestHeaders() != null) {
                            meteredGet.getRequestHeaders().setRange("bytes=" + Math.addExact(this.start, this.currentOffset) + "-" + this.end);
                        }
                        HttpResponse resp = meteredGet.executeMedia();
                        if (this.lastGeneration == null) {
                            this.lastGeneration = this.parseGenerationHeader(resp);
                        }
                        Long contentLength = resp.getHeaders().getContentLength();
                        InputStream content = resp.getContent();
                        if (contentLength != null) {
                            content = new ContentLengthValidatingInputStream(content, contentLength);
                        }
                        return content;
                    }
                    catch (IOException e) {
                        throw StorageException.translate((IOException)e);
                    }
                }, (RetrySettings)this.client.getOptions().getRetrySettings(), (ResultRetryAlgorithm)BaseService.EXCEPTION_HANDLER, (ApiClock)this.client.getOptions().getClock());
            }
            catch (RetryHelper.RetryHelperException e) {
                throw StorageException.translateAndThrow((RetryHelper.RetryHelperException)e);
            }
        }
        catch (StorageException storageException) {
            if (storageException.getCode() == RestStatus.NOT_FOUND.getStatus()) {
                if (this.lastGeneration != null) {
                    throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobId.getName() + "] generation [" + this.lastGeneration + "] unavailable on resume (contents changed, or object deleted): " + storageException.getMessage()));
                }
                throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobId.getName() + "] not found: " + storageException.getMessage()));
            }
            if (storageException.getCode() == RestStatus.REQUESTED_RANGE_NOT_SATISFIED.getStatus()) {
                long currentPosition = Math.addExact(this.start, this.currentOffset);
                throw this.addSuppressedExceptions(new RequestedRangeNotSatisfiedException(this.blobId.getName(), currentPosition, this.end < 0x7FFFFFFFFFFFFFFEL ? this.end - currentPosition + 1L : this.end, (Throwable)storageException));
            }
            throw this.addSuppressedExceptions(storageException);
        }
    }

    private Long parseGenerationHeader(HttpResponse response) {
        String generationHeader = response.getHeaders().getFirstHeaderStringValue("x-goog-generation");
        if (generationHeader != null) {
            try {
                return Long.parseLong(generationHeader);
            }
            catch (NumberFormatException e) {
                String message = "Unexpected value for x-goog-generation header: " + generationHeader;
                logger.warn(message);
                assert (false) : message;
            }
        } else {
            String message = "Missing x-goog-generation header";
            logger.warn(message);
            assert (false) : message;
        }
        return null;
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int result = this.currentStream.read();
                ++this.currentOffset;
                return result;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(StorageException.translate((IOException)e));
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    return -1;
                }
                this.currentOffset += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(StorageException.translate((IOException)e));
                continue;
            }
            break;
        }
    }

    void closeCurrentStream() throws IOException {
        this.currentStream.close();
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using GoogleCloudStorageRetryingInputStream after close";
            throw new IllegalStateException("using GoogleCloudStorageRetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(StorageException e) throws IOException {
        if (this.attempt >= this.maxAttempts) {
            throw this.addSuppressedExceptions(e);
        }
        logger.debug(() -> Strings.format((String)"failed reading [%s] at offset [%s], attempt [%s] of [%s], retrying", (Object[])new Object[]{this.blobId, this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
        ++this.attempt;
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        IOUtils.closeWhileHandlingException((Closeable)this.currentStream);
        this.currentStream = this.openStream();
    }

    @Override
    public void close() throws IOException {
        this.currentStream.close();
        this.closed = true;
    }

    @Override
    public long skip(long n) throws IOException {
        return super.skip(n);
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("GoogleCloudStorageRetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (StorageException failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    static final class ContentLengthValidatingInputStream
    extends FilterInputStream {
        private final long contentLength;
        private long read = 0L;

        ContentLengthValidatingInputStream(InputStream in, long contentLength) {
            super(in);
            this.contentLength = contentLength;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int n = this.in.read(b, off, len);
            if (n == -1) {
                this.checkContentLengthOnEOF();
            } else {
                this.read += (long)n;
            }
            return n;
        }

        @Override
        public int read() throws IOException {
            int n = this.in.read();
            if (n == -1) {
                this.checkContentLengthOnEOF();
            } else {
                ++this.read;
            }
            return n;
        }

        @Override
        public long skip(long len) throws IOException {
            long n = this.in.skip(len);
            this.read += n;
            return n;
        }

        private void checkContentLengthOnEOF() throws IOException {
            if (this.read < this.contentLength) {
                throw new IOException("Connection closed prematurely: read = " + this.read + ", Content-Length = " + this.contentLength);
            }
        }
    }
}

