"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIngestionRatePerTier = exports.useIngestionRate = exports.getAggregations = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _react = require("react");
var _common = require("@kbn/data-plugin/common");
var _rxjs = require("rxjs");
var _use_kibana = require("../../../../hooks/use_kibana");
var _use_streams_app_fetch = require("../../../../hooks/use_streams_app_fetch");
var _use_ilm_phases_color_and_description = require("./use_ilm_phases_color_and_description");
var _failure_store_index_name = require("../helpers/failure_store_index_name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TIMESTAMP_FIELD = '@timestamp';
const DEFAULT_SAMPLER_PROBABILITY = 0.1;
const STATISTICAL_ERROR_THRESHOLD = 0.01;
// Calculate sampling probability based on total docs to keep statistical error below threshold
const getSamplingProbability = totalDocs => {
  if (!totalDocs) {
    // If it turns out there aren't many docs, we just restart without sampling - it will be fast anyway
    // If it turns out there are lots of docs then we were right, yay!
    return DEFAULT_SAMPLER_PROBABILITY;
  }

  // Calculate required sample size for 1% statistical error (simplified formula)
  // For large populations, sample size ≈ 1 / (error^2) for 95% confidence
  const requiredSampleSize = 1 / STATISTICAL_ERROR_THRESHOLD ** 2;
  if (totalDocs <= requiredSampleSize) {
    return 1; // No sampling needed
  }
  return DEFAULT_SAMPLER_PROBABILITY;
};

// some units are not supported for the fixed_interval of the date histogram
// this function uses the calculateAutoTimeExpression function to determine
// if the interval should be a calendar_interval or a fixed_interval
const getIntervalAndType = (timeState, core) => {
  const interval = (0, _common.getCalculateAutoTimeExpression)(key => core.uiSettings.get(key))(timeState.asAbsoluteTimeRange);
  if (!interval) {
    return undefined;
  }
  const calendarIntervalUnits = new Set(['w', 'M', 'q', 'y']);
  const intervalType = calendarIntervalUnits.has(interval.replace(/^\d+/, '')) ? 'calendar_interval' : 'fixed_interval';
  return {
    interval,
    intervalType
  };
};
const useIngestionRate = ({
  calculatedStats,
  timeState,
  isLoading,
  aggregations,
  error
}) => {
  const ingestionRate = (0, _react.useMemo)(() => {
    if (!aggregations) {
      return undefined;
    }
    const start = (0, _moment.default)(timeState.start);
    const end = (0, _moment.default)(timeState.end);
    if (!aggregations || !aggregations.buckets || aggregations.buckets.length === 0) {
      return {
        start,
        end,
        interval: aggregations.interval,
        buckets: []
      };
    }
    const {
      buckets,
      interval
    } = aggregations;
    return {
      start,
      end,
      interval,
      buckets: buckets.map(({
        key,
        doc_count: docCount
      }) => ({
        key,
        value: docCount * ((calculatedStats === null || calculatedStats === void 0 ? void 0 : calculatedStats.bytesPerDoc) || 1)
      }))
    };
  }, [aggregations, timeState.start, timeState.end, calculatedStats === null || calculatedStats === void 0 ? void 0 : calculatedStats.bytesPerDoc]);
  return {
    ingestionRate,
    isLoading,
    error
  };
};
exports.useIngestionRate = useIngestionRate;
const useIngestionRatePerTier = ({
  definition,
  calculatedStats,
  timeState,
  isFailureStore = false
}) => {
  const {
    core,
    dependencies: {
      start: {
        data,
        streams: {
          streamsRepositoryClient
        }
      }
    }
  } = (0, _use_kibana.useKibana)();
  const {
    ilmPhases
  } = (0, _use_ilm_phases_color_and_description.useIlmPhasesColorAndDescription)();
  const ingestionRateFetch = (0, _use_streams_app_fetch.useStreamsAppFetch)(async ({
    signal
  }) => {
    const intervalData = getIntervalAndType(timeState, core);
    if (!timeState.start || !timeState.end || !intervalData) {
      return;
    }
    const start = (0, _moment.default)(timeState.start);
    const end = (0, _moment.default)(timeState.end);
    const {
      interval,
      intervalType
    } = intervalData;
    const indexName = isFailureStore ? (0, _failure_store_index_name.getFailureStoreIndexName)(definition) : definition.stream.name;
    const {
      rawResponse: {
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: {
        index: indexName,
        track_total_hits: false,
        body: {
          size: 0,
          query: {
            bool: {
              filter: [{
                range: {
                  [TIMESTAMP_FIELD]: {
                    gte: start,
                    lte: end
                  }
                }
              }]
            }
          },
          aggs: {
            sampler: {
              random_sampler: {
                probability: DEFAULT_SAMPLER_PROBABILITY,
                seed: 42
              },
              aggs: {
                docs_count: {
                  date_histogram: {
                    field: TIMESTAMP_FIELD,
                    [intervalType]: interval,
                    min_doc_count: 0,
                    extended_bounds: {
                      min: start,
                      max: end
                    }
                  },
                  aggs: {
                    indices: {
                      terms: {
                        field: '_index'
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
    }, {
      abortSignal: signal
    }));
    if (!aggregations || aggregations.sampler.docs_count.buckets.length === 0) {
      return {
        start,
        end,
        interval,
        buckets: {}
      };
    }

    // For failure store, use simplified ILM logic since we can't use streams API with failure store selector
    const ilmExplain = isFailureStore ? {
      indices: {}
    } // Simplified for failure store
    : await streamsRepositoryClient.fetch('GET /internal/streams/{name}/lifecycle/_explain', {
      params: {
        path: {
          name: definition.stream.name
        }
      },
      signal
    });
    const fallbackTier = 'hot';
    const buckets = aggregations.sampler.docs_count.buckets.reduce((acc, {
      key,
      doc_count: docCount,
      indices
    }) => {
      if (docCount === 0) {
        var _acc$fallbackTier;
        // there's no data for this bucket. push an empty data point
        // so we still graph the timestamp.
        (acc[fallbackTier] = (_acc$fallbackTier = acc[fallbackTier]) !== null && _acc$fallbackTier !== void 0 ? _acc$fallbackTier : []).push({
          key,
          value: 0
        });
        return acc;
      }
      const countByTier = indices.buckets.reduce((tiers, index) => {
        var _tiers$tier;
        const explain = ilmExplain.indices[index.key];
        // For failure store, use fallback tier since ILM explain won't exist
        const tier = explain !== null && explain !== void 0 && explain.managed && explain !== null && explain !== void 0 && explain.phase && explain.phase in ilmPhases ? explain.phase : fallbackTier;
        tiers[tier] = ((_tiers$tier = tiers[tier]) !== null && _tiers$tier !== void 0 ? _tiers$tier : 0) + index.doc_count;
        return tiers;
      }, {});
      for (const entry of Object.entries(countByTier)) {
        var _acc$tier;
        const tier = entry[0];
        (acc[tier] = (_acc$tier = acc[tier]) !== null && _acc$tier !== void 0 ? _acc$tier : []).push({
          key,
          value: entry[1] * ((calculatedStats === null || calculatedStats === void 0 ? void 0 : calculatedStats.bytesPerDoc) || 1)
        });
      }
      return acc;
    }, {});
    return {
      start,
      end,
      interval,
      buckets
    };
  }, [definition, calculatedStats, timeState, core, data.search, streamsRepositoryClient, ilmPhases, isFailureStore]);
  return {
    ingestionRate: ingestionRateFetch.value,
    isLoading: ingestionRateFetch.loading,
    error: ingestionRateFetch.error
  };
};
exports.useIngestionRatePerTier = useIngestionRatePerTier;
const getAggregations = async ({
  definition,
  timeState,
  totalDocs,
  isFailureStore = false,
  core,
  search,
  signal
}) => {
  const samplingProbability = getSamplingProbability(totalDocs);
  const intervalData = getIntervalAndType(timeState, core);
  if (!intervalData) {
    return;
  }
  const {
    interval,
    intervalType
  } = intervalData;
  const indexName = isFailureStore ? (0, _failure_store_index_name.getFailureStoreIndexName)(definition) : definition.stream.name;
  const {
    rawResponse: {
      aggregations
    }
  } = await (0, _rxjs.lastValueFrom)(search.search({
    params: {
      index: indexName,
      track_total_hits: false,
      body: {
        size: 0,
        query: {
          bool: {
            filter: [{
              range: {
                [TIMESTAMP_FIELD]: {
                  gte: timeState.start,
                  lte: timeState.end
                }
              }
            }]
          }
        },
        aggs: {
          sampler: {
            random_sampler: {
              probability: samplingProbability,
              seed: 42
            },
            aggs: {
              docs_count: {
                date_histogram: {
                  field: TIMESTAMP_FIELD,
                  [intervalType]: interval,
                  min_doc_count: 0,
                  extended_bounds: {
                    min: timeState.start,
                    max: timeState.end
                  }
                }
              }
            }
          }
        }
      }
    }
  }, {
    abortSignal: signal
  }));
  return {
    buckets: (aggregations === null || aggregations === void 0 ? void 0 : aggregations.sampler.docs_count.buckets) || [],
    interval
  };
};
exports.getAggregations = getAggregations;