"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.merge = merge;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Deeply merges two objects, omitting undefined values, and not deeply merging Arrays.
 *
 * @remarks
 * Should behave identically to lodash.merge, however it will not merge Array values like lodash does.
 * Any properties with `undefined` values on both objects will be ommitted from the returned object.
 */

function merge(baseObj, ...sources) {
  const firstSource = sources[0];
  if (firstSource === undefined) {
    return baseObj;
  }
  return sources.slice(1).reduce((merged, nextSource) => mergeObjects(merged, nextSource), mergeObjects(baseObj, firstSource));
}
const isMergable = obj => (0, _lodash.isPlainObject)(obj);
const mergeObjects = (baseObj, overrideObj) => [...new Set([...Object.keys(baseObj), ...Object.keys(overrideObj)])].reduce((merged, key) => {
  const baseVal = baseObj[key];
  const overrideVal = overrideObj[key];
  if (isMergable(baseVal) && isMergable(overrideVal)) {
    merged[key] = mergeObjects(baseVal, overrideVal);
  } else if (overrideVal !== undefined) {
    merged[key] = overrideVal;
  } else if (baseVal !== undefined) {
    merged[key] = baseVal;
  }
  return merged;
}, {});