"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExceptionsPreExportHandler = void 0;
var _errors = require("../validators/errors");
var _stringify = require("../../../endpoint/utils/stringify");
var _utils = require("../utils");
var _validators = require("../validators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getExceptionsPreExportHandler = endpointAppContextService => {
  const logger = endpointAppContextService.createLogger('listsExceptionsPreExportHandler');
  return async function ({
    data,
    context: {
      request,
      exceptionListClient
    }
  }) {
    if (data.namespaceType !== 'agnostic') {
      return data;
    }
    const {
      listId: maybeListId,
      id
    } = data;
    let listId = maybeListId;
    let isEndpointArtifact = false;
    if (!listId && id) {
      var _await$exceptionListC, _await$exceptionListC2;
      listId = (_await$exceptionListC = (_await$exceptionListC2 = await exceptionListClient.getExceptionList(data)) === null || _await$exceptionListC2 === void 0 ? void 0 : _await$exceptionListC2.list_id) !== null && _await$exceptionListC !== void 0 ? _await$exceptionListC : null;
    }
    if (!listId) {
      return data;
    }

    // Validate Trusted Applications
    if (_validators.TrustedAppValidator.isTrustedApp({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.TrustedAppValidator(endpointAppContextService, request).validatePreExport();
    }

    // Validate Trusted Devices
    if (_validators.TrustedDeviceValidator.isTrustedDevice({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.TrustedDeviceValidator(endpointAppContextService, request).validatePreExport();
    }

    // Host Isolation Exceptions validations
    if (_validators.HostIsolationExceptionsValidator.isHostIsolationException({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.HostIsolationExceptionsValidator(endpointAppContextService, request).validatePreExport();
    }

    // Event Filter validations
    if (_validators.EventFilterValidator.isEventFilter({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.EventFilterValidator(endpointAppContextService, request).validatePreExport();
    }

    // Validate Blocklists
    if (_validators.BlocklistValidator.isBlocklist({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.BlocklistValidator(endpointAppContextService, request).validatePreExport();
    }

    // Validate Endpoint Exceptions
    if (_validators.EndpointExceptionsValidator.isEndpointException({
      listId
    })) {
      isEndpointArtifact = true;
      await new _validators.EndpointExceptionsValidator(endpointAppContextService, request).validatePreExport();
    }

    // If space awareness is enabled, add space filter to export options
    if (isEndpointArtifact && endpointAppContextService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
      if (!request) {
        throw new _errors.EndpointArtifactExceptionValidationError(`Missing HTTP Request object`);
      }
      const spaceDataFilter = (await (0, _utils.buildSpaceDataFilter)(endpointAppContextService, request)).filter;
      data.filter = spaceDataFilter + (data.filter ? ` AND (${data.filter})` : '');
      logger.debug(`Export request after adding space filter:\n${(0, _stringify.stringify)(data)}`);
    }
    return data;
  };
};
exports.getExceptionsPreExportHandler = getExceptionsPreExportHandler;