"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MonitoringEntitySourceDescriptorClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _monitoring_entity_source_type = require("./monitoring_entity_source_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MonitoringEntitySourceDescriptorClient {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "getQueryFilters", query => {
      return Object.entries(query !== null && query !== void 0 ? query : {}).map(([key, value]) => `${_monitoring_entity_source_type.monitoringEntitySourceTypeName}.attributes.${key}: ${value}`).join(' and ');
    });
    this.dependencies = dependencies;
  }
  async create(attributes) {
    var _attributes$managed;
    await this.assertNameUniqueness(attributes);
    const {
      id,
      attributes: created
    } = await this.dependencies.soClient.create(_monitoring_entity_source_type.monitoringEntitySourceTypeName, {
      ...attributes,
      managed: (_attributes$managed = attributes.managed) !== null && _attributes$managed !== void 0 ? _attributes$managed : false
    },
    // Ensure managed is set to true on creation
    {
      refresh: 'wait_for'
    });
    return {
      ...created,
      id
    };
  }
  async bulkCreate(sources) {
    const createdSources = await this.dependencies.soClient.bulkCreate(sources.map(source => ({
      type: _monitoring_entity_source_type.monitoringEntitySourceTypeName,
      attributes: {
        ...source
      }
    })), {
      refresh: 'wait_for'
    });
    return createdSources;
  }
  async upsert(source) {
    const foundResult = await this.find({
      name: source.name
    });
    const found = foundResult.saved_objects[0];
    if (found) {
      await this.update({
        ...source,
        id: found.id
      });
      return {
        action: 'updated',
        source: {
          ...source,
          id: found.id
        }
      };
    } else {
      const createdSource = await this.create(source);
      return {
        action: 'created',
        source: createdSource
      };
    }
  }
  async bulkUpsert(sources) {
    if (!sources.length) return {
      created: 0,
      updated: 0,
      results: []
    };
    const existing = await this.findAll({});
    const byName = new Map(existing.map(s => [s.name, s]));
    let created = 0;
    let updated = 0;
    const results = [];
    for (const attrs of sources) {
      const found = byName.get(attrs.name);
      if (!found) {
        const createdSo = await this.create(attrs);
        created++;
        byName.set(createdSo.name, createdSo);
        results.push({
          action: 'created',
          source: createdSo
        });
      } else {
        const updatedSo = await this.update({
          id: found.id,
          ...attrs
        });
        updated++;
        byName.set(updatedSo.name, updatedSo);
        results.push({
          action: 'updated',
          source: updatedSo
        });
      }
    }
    return {
      created,
      updated,
      results
    };
  }
  async update(monitoringEntitySource) {
    await this.assertNameUniqueness(monitoringEntitySource);
    const {
      attributes
    } = await this.dependencies.soClient.update(_monitoring_entity_source_type.monitoringEntitySourceTypeName, monitoringEntitySource.id, monitoringEntitySource, {
      refresh: 'wait_for'
    });
    return {
      ...attributes,
      id: monitoringEntitySource.id
    };
  }
  async find(query) {
    const scopedSoClient = this.dependencies.soClient;
    return scopedSoClient.find({
      type: _monitoring_entity_source_type.monitoringEntitySourceTypeName,
      filter: this.getQueryFilters(query)
    });
  }
  async get(id) {
    const {
      attributes
    } = await this.dependencies.soClient.get(_monitoring_entity_source_type.monitoringEntitySourceTypeName, id);
    return attributes;
  }
  async delete(id) {
    await this.dependencies.soClient.delete(_monitoring_entity_source_type.monitoringEntitySourceTypeName, id);
  }

  /**
   * entity_analytics_integration or index type
   */
  async findSourcesByType(type) {
    const result = await this.find();
    return result.saved_objects.filter(so => so.attributes.type === type).map(so => ({
      ...so.attributes,
      id: so.id
    }));
  }
  async findAll(query) {
    const result = await this.find(query);
    return result.saved_objects.filter(so => so.attributes.type !== 'csv') // from the spec we are not using CSV on monitoring
    .map(so => ({
      ...so.attributes,
      id: so.id
    }));
  }
  async findByQuery(query) {
    const scopedSoClient = this.dependencies.soClient;
    const results = await scopedSoClient.find({
      type: _monitoring_entity_source_type.monitoringEntitySourceTypeName,
      filter: query,
      namespaces: [this.dependencies.namespace]
    });
    return results.saved_objects.map(so => ({
      ...so.attributes,
      id: so.id
    }));
  }
  async assertNameUniqueness(attributes) {
    if (attributes.name) {
      const {
        saved_objects: savedObjects
      } = await this.find({
        name: attributes.name
      });

      // Exclude the current entity source if updating
      const filteredSavedObjects = attributes.id ? savedObjects.filter(so => so.id !== attributes.id) : savedObjects;
      if (filteredSavedObjects.length > 0) {
        throw new Error(`A monitoring entity source with the name "${attributes.name}" already exists.`);
      }
    }
  }

  /**
   * Integrations Specific Methods
   */
  async updateLastProcessedMarker(source, lastProcessedMarker) {
    await this.update({
      ...source,
      integrations: {
        syncData: {
          lastUpdateProcessed: lastProcessedMarker
        }
      }
    });
  }
  async getLastProcessedMarker(source) {
    var _source$integrations, _source$integrations$;
    return (_source$integrations = source.integrations) === null || _source$integrations === void 0 ? void 0 : (_source$integrations$ = _source$integrations.syncData) === null || _source$integrations$ === void 0 ? void 0 : _source$integrations$.lastUpdateProcessed;
  }
  async getLastFullSyncMarker(source) {
    var _source$integrations2, _source$integrations3;
    return (_source$integrations2 = source.integrations) === null || _source$integrations2 === void 0 ? void 0 : (_source$integrations3 = _source$integrations2.syncData) === null || _source$integrations3 === void 0 ? void 0 : _source$integrations3.lastFullSync;
  }
  async updateLastFullSyncMarker(source, lastFullSyncMarker) {
    await this.update({
      ...source,
      integrations: {
        syncData: {
          lastFullSync: lastFullSyncMarker
        }
      }
    });
  }
}
exports.MonitoringEntitySourceDescriptorClient = MonitoringEntitySourceDescriptorClient;