"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.traverseAndMutateDoc = exports.getIsEcsField = void 0;
var _alertsAsDataUtils = require("@kbn/alerts-as-data-utils");
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _is_valid_ip_type = require("./ecs_types_validators/is_valid_ip_type");
var _is_valid_date_type = require("./ecs_types_validators/is_valid_date_type");
var _is_valid_numeric_type = require("./ecs_types_validators/is_valid_numeric_type");
var _is_valid_boolean_type = require("./ecs_types_validators/is_valid_boolean_type");
var _is_valid_long_type = require("./ecs_types_validators/is_valid_long_type");
var _field_names = require("../../../../../../common/field_maps/field_names");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * type guard for object of SearchTypes
 */
const isSearchTypesRecord = value => {
  return (0, _lodash.isPlainObject)(value);
};

/**
 * retrieve all nested object fields from ecsFieldMap
 * field `agent.build.original` will be converted into
 * { agent: true, agent.build: true }
 */
const getEcsObjectFields = () => {
  const result = {};
  Object.entries(_alertsAsDataUtils.ecsFieldMap).forEach(([key, value]) => {
    const objects = key.split('.');
    // last item can be any of type, as precedent are objects
    objects.pop();
    objects.reduce((parentPath, itemKey) => {
      const fullPath = parentPath ? `${parentPath}.${itemKey}` : itemKey;
      if (!result[fullPath]) {
        result[fullPath] = true;
      }
      return fullPath;
    }, '');
  });
  return result;
};
const ecsObjectFields = getEcsObjectFields();

/**
 * checks if path is a valid Ecs object type (object or flattened)
 * geo_point also can be object
 */
const getIsEcsFieldObject = path => {
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  return ['object', 'flattened', 'geo_point'].includes(ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) || ecsObjectFields[path];
};

/**
 * checks if path is in Ecs mapping
 */
const getIsEcsField = path => {
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  const isEcsField = Boolean(!!ecsField || ecsObjectFields[path]);
  return isEcsField;
};

/**
 * if any of partial path in dotted notation is not an object in ECS mapping
 * it means the field itself is not valid as well
 * For example, 'agent.name.conflict' - if agent.name is keyword, so the whole path is invalid
 */
exports.getIsEcsField = getIsEcsField;
const validateDottedPathInEcsMappings = path => {
  let isValid = true;
  path.split('.').slice(0, -1) // exclude last path item, as we check only if all parent are objects
  .reduce((acc, key) => {
    const pathToValidate = [acc, key].filter(Boolean).join('.');
    const isEcsField = getIsEcsField(pathToValidate);
    const isEcsFieldObject = getIsEcsFieldObject(pathToValidate);

    // if field is in Ecs mapping and not object, the whole path is invalid
    if (isEcsField && !isEcsFieldObject) {
      isValid = false;
    }
    return pathToValidate;
  }, '');
  return isValid;
};

/**
 * check whether source field value is ECS compliant
 */
const computeIsEcsCompliant = (value, path) => {
  // if path consists of dot-notation, ensure each path within it is ECS compliant (object or flattened)
  if (path.includes('.') && !validateDottedPathInEcsMappings(path)) {
    return false;
  }
  const isEcsField = getIsEcsField(path);

  // if field is not present is ECS mapping, it's valid as doesn't have any conflicts with existing mapping
  if (!isEcsField) {
    return true;
  }
  const ecsField = _alertsAsDataUtils.ecsFieldMap[path];
  const isEcsFieldObject = getIsEcsFieldObject(path);

  // do not validate geo_point, since it's very complex type that can be string/array/object
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'geo_point') {
    return true;
  }

  // validate if value is a long type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'long') {
    return (0, _is_valid_long_type.isValidLongType)(value);
  }

  // validate if value is a numeric type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'float' || (ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'scaled_float') {
    return (0, _is_valid_numeric_type.isValidNumericType)(value);
  }

  // validate if value is a valid ip type
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'ip') {
    return (0, _is_valid_ip_type.isValidIpType)(value);
  }

  // validate if value is a valid date
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'date') {
    return (0, _is_valid_date_type.isValidDateType)(value);
  }

  // validate if value is a valid boolean
  if ((ecsField === null || ecsField === void 0 ? void 0 : ecsField.type) === 'boolean') {
    return (0, _is_valid_boolean_type.isValidBooleanType)(value);
  }

  // if ECS mapping is JS object and source value also JS object then they are compliant
  // otherwise not
  return isEcsFieldObject ? (0, _lodash.isPlainObject)(value) : !(0, _lodash.isPlainObject)(value);
};
const bannedFields = ['kibana', 'signal', 'threshold_result', _field_names.ALERT_THRESHOLD_RESULT];

/**
 * Traverse an entire source document and mutate it to prepare for indexing into the alerts index. Traversing the document
 * is computationally expensive so we only want to traverse it once, therefore a few distinct cases are handled in this function:
 * 1. Fields that we must explicitly remove, like `kibana` and `signal`, fields, are removed from the document.
 * 2. Fields that are incompatible with ECS are removed.
 * 3. All `event.*` fields are collected and copied to `kibana.alert.original_event.*` using `fieldsToAdd`
 * @param document The document to traverse
 * @returns The mutated document, a list of removed fields
 */
const traverseAndMutateDoc = document => {
  const {
    result,
    removed,
    fieldsToAdd
  } = internalTraverseAndMutateDoc({
    document,
    path: [],
    topLevel: true,
    removed: [],
    fieldsToAdd: []
  });
  fieldsToAdd.forEach(({
    key,
    value
  }) => {
    result[key] = value;
  });
  return {
    result,
    removed
  };
};
exports.traverseAndMutateDoc = traverseAndMutateDoc;
const internalTraverseAndMutateDoc = ({
  document,
  path,
  topLevel,
  removed,
  fieldsToAdd
}) => {
  Object.keys(document).forEach(key => {
    // Using Object.keys and fetching the value for each key separately performs better in profiling than using Object.entries
    const value = document[key];
    const fullPathArray = [...path, key];
    const fullPath = fullPathArray.join('.');
    // Insert checks that don't care about the value - only depend on the key - up here
    let deleted = false;
    if (topLevel) {
      const firstKeyString = key.split('.')[0];
      bannedFields.forEach(bannedField => {
        if (firstKeyString === bannedField) {
          delete document[key];
          deleted = true;
          removed.push({
            key: fullPath,
            value
          });
        }
      });
    }

    // If we passed the key check, additional checks based on key and value are done below. Items in arrays are treated independently from each other.
    if (!deleted) {
      if ((0, _lodash.isArray)(value)) {
        const newValue = traverseArray({
          array: value,
          path: fullPathArray,
          removed,
          fieldsToAdd
        });
        if (newValue.length > 0) {
          (0, _saferLodashSet.set)(document, key, newValue);
        } else {
          delete document[key];
          deleted = true;
        }
      } else if (!computeIsEcsCompliant(value, fullPath)) {
        delete document[key];
        deleted = true;
        removed.push({
          key: fullPath,
          value
        });
      } else if (isSearchTypesRecord(value)) {
        internalTraverseAndMutateDoc({
          document: value,
          path: fullPathArray,
          topLevel: false,
          removed,
          fieldsToAdd
        });
        if (Object.keys(value).length === 0) {
          delete document[key];
          deleted = true;
        }
      }
    }

    // We're keeping the field, but maybe we want to copy it to a different field as well
    if (!deleted && topLevel) {
      const topLevelPath = getTopLevelPath(fullPath);
      if (pathNeedsCopying(topLevelPath)) {
        // The value might have changed above when we `set` after traversing an array
        const valueRefetch = document[key];
        const newKey = getCopyDestinationPath(fullPath, topLevelPath);
        if ((0, _lodash.isPlainObject)(valueRefetch)) {
          const flattenedObject = (0, _securitysolutionRules.flattenWithPrefix)(newKey, valueRefetch);
          for (const [k, v] of Object.entries(flattenedObject)) {
            fieldsToAdd.push({
              key: k,
              value: v
            });
          }
        } else {
          fieldsToAdd.push({
            key: newKey,
            value: valueRefetch
          });
        }
      }
    }
  });
  return {
    result: document,
    removed,
    fieldsToAdd
  };
};
const traverseArray = ({
  array,
  path,
  removed,
  fieldsToAdd
}) => {
  const pathString = path.join('.');
  for (let i = 0; i < array.length; i++) {
    const value = array[i];
    if ((0, _lodash.isArray)(value)) {
      array[i] = traverseArray({
        array: value,
        path,
        removed,
        fieldsToAdd
      });
    }
  }
  return array.filter(value => {
    if ((0, _lodash.isArray)(value)) {
      return value.length > 0;
    } else if (!computeIsEcsCompliant(value, pathString)) {
      removed.push({
        key: pathString,
        value
      });
      return false;
    } else if (isSearchTypesRecord(value)) {
      internalTraverseAndMutateDoc({
        document: value,
        path,
        topLevel: false,
        removed,
        fieldsToAdd
      });
      return Object.keys(value).length > 0;
    } else {
      return true;
    }
  });
};
const getTopLevelPath = fullPath => fullPath.split('.')[0];

/**
 *
 * A map of ECS namespaces to their additional alerting namespaces. In cases
 * where the alert metadata may overwrite this source data, or where there is
 * not an appropriate mapping in ECS, we copy those fields to these additional
 * locations so as to preserve them and (with mappings) make them search/filterable.
 */
const alertingNamespaceCopyMap = {
  event: _field_names.ALERT_ORIGINAL_EVENT,
  data_stream: _field_names.ALERT_ORIGINAL_DATA_STREAM
};

/**
 *
 * @param topLevelPath The top-level path to the field in the document
 * @returns whether the path needs to be copied to an additional location
 */
const pathNeedsCopying = topLevelPath => topLevelPath in alertingNamespaceCopyMap;

/**
 *
 * @param fullPath The full path to the field in the document
 * @param topLevelPath The initial path/namespace of `fullPath`, i.e. `fullPath.startsWith(topLevelPath)`
 * @returns the full destination path to copy the field into
 */
const getCopyDestinationPath = (fullPath, topLevelPath) => {
  const copyPathRoot = alertingNamespaceCopyMap[topLevelPath];
  return `${copyPathRoot}${fullPath.replace(topLevelPath, '')}`;
};