"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateBulkScheduleBackfill = exports.validateBulkRuleGapFilling = exports.validateBulkEnableRule = exports.validateBulkEditRule = exports.validateBulkDuplicateRule = exports.validateBulkDisableRule = exports.dryRunValidateBulkEditRule = void 0;
var _rule_management = require("../../../../../../common/api/detection_engine/rule_management");
var _utils = require("../../../../../../common/detection_engine/utils");
var _helpers = require("../../../../../../common/machine_learning/helpers");
var _invariant = require("../../../../../../common/utils/invariant");
var _validation = require("../../../../machine_learning/validation");
var _dry_run = require("./dry_run");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * throws ML authorization error wrapped with MACHINE_LEARNING_AUTH error code
 * @param mlAuthz - {@link MlAuthz}
 * @param ruleType - {@link RuleType}
 */
const throwMlAuthError = (mlAuthz, ruleType) => (0, _dry_run.throwDryRunError)(async () => (0, _validation.throwAuthzError)(await mlAuthz.validateRuleType(ruleType)), _rule_management.BulkActionsDryRunErrCodeEnum.MACHINE_LEARNING_AUTH);

/**
 * runs validation for bulk enable for a single rule
 * @param params - {@link BulkActionsValidationArgs}
 */
const validateBulkEnableRule = async ({
  rule,
  mlAuthz
}) => {
  await throwMlAuthError(mlAuthz, rule.params.type);
};

/**
 * runs validation for bulk disable for a single rule
 * @param params - {@link BulkActionsValidationArgs}
 */
exports.validateBulkEnableRule = validateBulkEnableRule;
const validateBulkDisableRule = async ({
  rule,
  mlAuthz
}) => {
  await throwMlAuthError(mlAuthz, rule.params.type);
};

/**
 * runs validation for bulk duplicate for a single rule
 * @param params - {@link BulkActionsValidationArgs}
 */
exports.validateBulkDisableRule = validateBulkDisableRule;
const validateBulkDuplicateRule = async ({
  rule,
  mlAuthz
}) => {
  await throwMlAuthError(mlAuthz, rule.params.type);
};

/**
 * runs validation for bulk schedule backfill for a single rule
 * @param params - {@link DryRunManualRuleRunBulkActionsValidationArgs}
 */
exports.validateBulkDuplicateRule = validateBulkDuplicateRule;
const validateBulkScheduleBackfill = async ({
  rule
}) => {
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(rule.enabled, 'Cannot schedule manual rule run for a disabled rule'), _rule_management.BulkActionsDryRunErrCodeEnum.MANUAL_RULE_RUN_DISABLED_RULE);
};

/**
 * runs validation for bulk gap filling for a single rule
 * @param params - {@link DryRunRuleFillGapsBulkActionsValidationArgs}
 */
exports.validateBulkScheduleBackfill = validateBulkScheduleBackfill;
const validateBulkRuleGapFilling = async ({
  rule
}) => {
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(rule.enabled, 'Cannot bulk fill gaps for a disabled rule'), _rule_management.BulkActionsDryRunErrCodeEnum.RULE_FILL_GAPS_DISABLED_RULE);
};
exports.validateBulkRuleGapFilling = validateBulkRuleGapFilling;
/**
 * runs validation for bulk edit for a single rule
 */
const validateBulkEditRule = async ({
  ruleType,
  mlAuthz,
  edit,
  immutable,
  ruleCustomizationStatus
}) => {
  await throwMlAuthError(mlAuthz, ruleType);

  // Prebuilt rule customization checks
  if (immutable) {
    if (ruleCustomizationStatus.isRulesCustomizationEnabled) {
      // Rule customization is enabled; prebuilt rules can be edited
      return undefined;
    }

    // Rule customization is disabled; only certain actions can be applied to immutable rules
    const canRuleBeEdited = isEditApplicableToImmutableRule(edit);
    if (!canRuleBeEdited) {
      await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(canRuleBeEdited, "Elastic rule can't be edited"), _rule_management.BulkActionsDryRunErrCodeEnum.PREBUILT_CUSTOMIZATION_LICENSE);
    }
  }
};

/**
 * add_rule_actions, set_rule_actions can be applied to prebuilt/immutable rules
 */
exports.validateBulkEditRule = validateBulkEditRule;
const isEditApplicableToImmutableRule = edit => {
  const applicableActions = [_rule_management.BulkActionEditTypeEnum.set_rule_actions, _rule_management.BulkActionEditTypeEnum.add_rule_actions];
  return edit.every(({
    type
  }) => applicableActions.includes(type));
};
/**
 * executes dry run validations for bulk edit of a single rule
 */
const dryRunValidateBulkEditRule = async ({
  rule,
  edit,
  mlAuthz,
  ruleCustomizationStatus
}) => {
  await validateBulkEditRule({
    ruleType: rule.params.type,
    mlAuthz,
    edit,
    immutable: rule.params.immutable,
    ruleCustomizationStatus
  });

  // if rule is machine_learning, index pattern action can't be applied to it
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(!(0, _helpers.isMlRule)(rule.params.type) || !edit.some(action => (0, _utils2.isIndexPatternsBulkEditAction)(action.type)), "Machine learning rule doesn't have index patterns"), _rule_management.BulkActionsDryRunErrCodeEnum.MACHINE_LEARNING_INDEX_PATTERN);

  // if rule is es|ql, index pattern action can't be applied to it
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(!(0, _utils.isEsqlRule)(rule.params.type) || !edit.some(action => (0, _utils2.isIndexPatternsBulkEditAction)(action.type)), "ES|QL rule doesn't have index patterns"), _rule_management.BulkActionsDryRunErrCodeEnum.ESQL_INDEX_PATTERN);

  // if rule is threshold, set_alert_suppression action can't be applied to it
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)(!(0, _utils.isThresholdRule)(rule.params.type) || !edit.some(action => action.type === _rule_management.BulkActionEditTypeEnum.set_alert_suppression), "Threshold rule doesn't support this action. Use 'set_alert_suppression_for_threshold' action instead"), _rule_management.BulkActionsDryRunErrCodeEnum.THRESHOLD_RULE_TYPE_IN_SUPPRESSION);

  // if rule noy threshold, set_alert_suppression_for_threshold action can't be applied to it
  await (0, _dry_run.throwDryRunError)(() => (0, _invariant.invariant)((0, _utils.isThresholdRule)(rule.params.type) || !edit.some(action => action.type === _rule_management.BulkActionEditTypeEnum.set_alert_suppression_for_threshold), "Rule type doesn't support this action. Use 'set_alert_suppression' action instead."), _rule_management.BulkActionsDryRunErrCodeEnum.UNSUPPORTED_RULE_IN_SUPPRESSION_FOR_THRESHOLD);
};
exports.dryRunValidateBulkEditRule = dryRunValidateBulkEditRule;