"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mapToNormalizedActionRequest = exports.mapResponsesByActionId = exports.isLogsEndpointActionResponse = exports.isLogsEndpointAction = exports.getUniqueLogData = exports.getDateFilters = exports.getAgentIdFromActionResponse = exports.getAgentHostNamesWithIds = exports.getActionStatus = exports.getActionRequestExpiration = exports.getActionIdFromActionResponse = exports.getActionCompletionInfo = exports.formatEndpointActionResults = exports.createActionDetailsRecord = exports.categorizeResponseResults = exports.categorizeActionResults = void 0;
var _moment = _interopRequireDefault(require("moment/moment"));
var _lodash = require("lodash");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _utils = require("../../../routes/actions/utils");
var _utils2 = require("../../../utils");
var _constants = require("../../../../../common/endpoint/constants");
var _types = require("../../../../../common/endpoint/types");
var _get_file_download_id = require("../../../../../common/endpoint/service/response_actions/get_file_download_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Type guard to check if a given Action is in the shape of the Endpoint Action.
 * @param item
 */
const isLogsEndpointAction = item => {
  return 'EndpointActions' in item && 'user' in item && 'agent' in item && '@timestamp' in item;
};

/**
 * Type guard to track if a given action response is in the shape of the Endpoint Action Response (from the endpoint index)
 * @param item
 */
exports.isLogsEndpointAction = isLogsEndpointAction;
const isLogsEndpointActionResponse = item => {
  return 'EndpointActions' in item && 'agent' in item;
};
exports.isLogsEndpointActionResponse = isLogsEndpointActionResponse;
/**
 * Given an Action record - either a fleet action or an endpoint action - this utility
 * will return a normalized data structure based on those two types, which
 * will avoid us having to either cast or do type guards against the two different
 * types of action request.
 */
const mapToNormalizedActionRequest = actionRequest => {
  const type = 'ACTION_REQUEST';
  if (isLogsEndpointAction(actionRequest)) {
    var _actionRequest$rule, _actionRequest$rule2, _actionRequest$Endpoi;
    return {
      agents: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id : [actionRequest.agent.id],
      agentType: actionRequest.EndpointActions.input_type,
      command: actionRequest.EndpointActions.data.command,
      comment: actionRequest.EndpointActions.data.comment,
      createdBy: actionRequest.user.id,
      createdAt: actionRequest['@timestamp'],
      expiration: actionRequest.EndpointActions.expiration,
      id: actionRequest.EndpointActions.action_id,
      type,
      parameters: actionRequest.EndpointActions.data.parameters,
      alertIds: actionRequest.EndpointActions.data.alert_id,
      ruleId: (_actionRequest$rule = actionRequest.rule) === null || _actionRequest$rule === void 0 ? void 0 : _actionRequest$rule.id,
      ruleName: (_actionRequest$rule2 = actionRequest.rule) === null || _actionRequest$rule2 === void 0 ? void 0 : _actionRequest$rule2.name,
      error: actionRequest.error,
      hosts: (_actionRequest$Endpoi = actionRequest.EndpointActions.data.hosts) !== null && _actionRequest$Endpoi !== void 0 ? _actionRequest$Endpoi : {}
    };
  }

  // Else, it's a Fleet Endpoint Action record
  return {
    agents: actionRequest.agents,
    agentType: actionRequest.input_type,
    command: actionRequest.data.command,
    comment: actionRequest.data.comment,
    createdBy: actionRequest.user_id,
    createdAt: actionRequest['@timestamp'],
    expiration: actionRequest.expiration,
    id: actionRequest.action_id,
    type,
    parameters: actionRequest.data.parameters,
    hosts: {}
  };
};

/**
 * Maps the list of fetch action responses (from both Endpoint and Fleet indexes) to a Map
 * whose keys are the action ID and value is the set of responses for that action id
 * @param actionResponses
 */
exports.mapToNormalizedActionRequest = mapToNormalizedActionRequest;
const mapResponsesByActionId = actionResponses => {
  return [...actionResponses.endpointResponses, ...actionResponses.fleetResponses].reduce((acc, response) => {
    const actionId = getActionIdFromActionResponse(response);
    if (!acc[actionId]) {
      acc[actionId] = {
        endpointResponses: [],
        fleetResponses: []
      };
    }
    if (isLogsEndpointActionResponse(response)) {
      acc[actionId].endpointResponses.push(response);
    } else {
      acc[actionId].fleetResponses.push(response);
    }
    return acc;
  }, {});
};
exports.mapResponsesByActionId = mapResponsesByActionId;
const getActionCompletionInfo = (action, actionResponses) => {
  var _action$error;
  const agentIds = action.agents;
  const completedInfo = {
    completedAt: undefined,
    errors: undefined,
    outputs: {},
    agentState: {},
    isCompleted: Boolean(agentIds.length),
    wasSuccessful: Boolean(agentIds.length)
  };
  const responsesByAgentId = mapActionResponsesByAgentId(actionResponses);
  for (const agentId of agentIds) {
    const agentResponse = responsesByAgentId[agentId];

    // Set the overall Action to not completed if at least
    // one of the agent responses is not complete yet.
    if (!agentResponse || !agentResponse.isCompleted) {
      completedInfo.isCompleted = false;
      completedInfo.wasSuccessful = false;
    }

    // individual agent state
    completedInfo.agentState[agentId] = {
      isCompleted: false,
      wasSuccessful: false,
      errors: undefined,
      completedAt: undefined
    };

    // Store the outputs and agent state for any agent that sent a response
    if (agentResponse) {
      completedInfo.agentState[agentId].isCompleted = agentResponse.isCompleted;
      completedInfo.agentState[agentId].wasSuccessful = agentResponse.wasSuccessful;
      completedInfo.agentState[agentId].completedAt = agentResponse.completedAt;
      completedInfo.agentState[agentId].errors = agentResponse.errors;
      if (agentResponse.endpointResponse && agentResponse.endpointResponse.EndpointActions.data.output) {
        completedInfo.outputs[agentId] = agentResponse.endpointResponse.EndpointActions.data.output;
        if ((0, _utils.doesActionHaveFileAccess)(action.agentType, action.command) && completedInfo.agentState[agentId].isCompleted && completedInfo.agentState[agentId].wasSuccessful) {
          (0, _saferLodashSet.set)(completedInfo.outputs[agentId], 'content.downloadUri', _constants.ACTION_AGENT_FILE_DOWNLOAD_ROUTE.replace(`{action_id}`, action.id).replace(`{file_id}`, (0, _get_file_download_id.getFileDownloadId)(action, agentId)));
        }
      }
    }
  }

  // If completed, then get the completed at date and determine if action as a whole was successful or not
  if (completedInfo.isCompleted) {
    const responseErrors = [];
    for (const normalizedAgentResponse of Object.values(responsesByAgentId)) {
      var _normalizedAgentRespo;
      if (!completedInfo.completedAt || completedInfo.completedAt < ((_normalizedAgentRespo = normalizedAgentResponse.completedAt) !== null && _normalizedAgentRespo !== void 0 ? _normalizedAgentRespo : '')) {
        completedInfo.completedAt = normalizedAgentResponse.completedAt;
      }
      if (!normalizedAgentResponse.wasSuccessful) {
        completedInfo.wasSuccessful = false;
        responseErrors.push(...(normalizedAgentResponse.errors ? normalizedAgentResponse.errors : []));
      }
    }
    if (responseErrors.length) {
      completedInfo.errors = responseErrors;
    }
  }

  // If the action request has an Error, then we'll never get actual response from all of the agents
  // to which this action sent. In this case, we adjust the completion information to all be "complete"
  // and un-successful
  if ((_action$error = action.error) !== null && _action$error !== void 0 && _action$error.message) {
    const errorMessage = action.error.message;
    completedInfo.completedAt = action.createdAt;
    completedInfo.isCompleted = true;
    completedInfo.wasSuccessful = false;
    completedInfo.errors = [errorMessage];
    Object.values(completedInfo.agentState).forEach(agentState => {
      agentState.completedAt = action.createdAt;
      agentState.isCompleted = true;
      agentState.wasSuccessful = false;
      agentState.errors = [errorMessage];
    });
  }
  return completedInfo;
};
exports.getActionCompletionInfo = getActionCompletionInfo;
const getActionStatus = ({
  expirationDate,
  isCompleted,
  wasSuccessful
}) => {
  const isExpired = !isCompleted && expirationDate < new Date().toISOString();
  const status = isExpired ? 'failed' : isCompleted ? wasSuccessful ? 'successful' : 'failed' : 'pending';
  return {
    isExpired,
    status
  };
};
exports.getActionStatus = getActionStatus;
/**
 * Given a list of Action Responses, it will return a Map where keys are the Agent ID and
 * value is a object having information about the action responses associated with that agent id
 * @param actionResponses
 */
const mapActionResponsesByAgentId = actionResponses => {
  const response = [...actionResponses.endpointResponses, ...actionResponses.fleetResponses].reduce((acc, actionResponseRecord) => {
    const agentId = getAgentIdFromActionResponse(actionResponseRecord);
    if (!acc[agentId]) {
      acc[agentId] = {
        isCompleted: false,
        completedAt: undefined,
        wasSuccessful: false,
        errors: undefined,
        fleetResponse: undefined,
        endpointResponse: undefined
      };
    }
    if (isLogsEndpointActionResponse(actionResponseRecord)) {
      acc[agentId].endpointResponse = actionResponseRecord;
    } else {
      acc[agentId].fleetResponse = actionResponseRecord;
    }
    return acc;
  }, {});
  for (const agentNormalizedResponse of Object.values(response)) {
    var _agentNormalizedRespo;
    agentNormalizedResponse.isCompleted =
    // Action is complete if an Endpoint Action Response was received
    Boolean(agentNormalizedResponse.endpointResponse) ||
    // OR:
    // If we did not have an endpoint response and the Fleet response has `error`, then
    // action is complete. Elastic Agent was unable to deliver the action request to the
    // endpoint, so we are unlikely to ever receive an Endpoint Response.
    Boolean((_agentNormalizedRespo = agentNormalizedResponse.fleetResponse) === null || _agentNormalizedRespo === void 0 ? void 0 : _agentNormalizedRespo.error);

    // When completed, calculate additional properties about the action
    if (agentNormalizedResponse.isCompleted) {
      var _agentNormalizedRespo3, _agentNormalizedRespo4, _agentNormalizedRespo5;
      if (agentNormalizedResponse.endpointResponse) {
        var _agentNormalizedRespo2;
        agentNormalizedResponse.completedAt = (_agentNormalizedRespo2 = agentNormalizedResponse.endpointResponse) === null || _agentNormalizedRespo2 === void 0 ? void 0 : _agentNormalizedRespo2['@timestamp'];
        agentNormalizedResponse.wasSuccessful = true;
      } else if (
      // Check if perhaps the Fleet action response returned an error, in which case, the Fleet Agent
      // failed to deliver the Action to the Endpoint. If that's the case, we are not going to get
      // a Response from endpoint, thus mark the Action as completed and use the Fleet Message's
      // timestamp for the complete data/time.
      agentNormalizedResponse.fleetResponse && agentNormalizedResponse.fleetResponse.error) {
        agentNormalizedResponse.isCompleted = true;
        agentNormalizedResponse.completedAt = agentNormalizedResponse.fleetResponse['@timestamp'];
      }
      const errors = [];

      // only one of the errors should be in there
      if ((_agentNormalizedRespo3 = agentNormalizedResponse.endpointResponse) !== null && _agentNormalizedRespo3 !== void 0 && (_agentNormalizedRespo4 = _agentNormalizedRespo3.error) !== null && _agentNormalizedRespo4 !== void 0 && _agentNormalizedRespo4.message) {
        errors.push(`Endpoint action response error: ${agentNormalizedResponse.endpointResponse.error.message}`);
      }
      if ((_agentNormalizedRespo5 = agentNormalizedResponse.fleetResponse) !== null && _agentNormalizedRespo5 !== void 0 && _agentNormalizedRespo5.error) {
        errors.push(`Fleet action response error: ${agentNormalizedResponse.fleetResponse.error}`);
      }
      if (errors.length) {
        agentNormalizedResponse.wasSuccessful = false;
        agentNormalizedResponse.errors = errors;
      }
    }
  }
  return response;
};

/**
 * Given an Action response, this will return the Agent ID for that action response.
 * @param actionResponse
 */
const getAgentIdFromActionResponse = actionResponse => {
  if (isLogsEndpointActionResponse(actionResponse)) {
    return Array.isArray(actionResponse.agent.id) ? actionResponse.agent.id[0] : actionResponse.agent.id;
  }
  return actionResponse.agent_id;
};

/**
 * Given an Action response from either Endpoint or Fleet, utility will return its action id
 * @param actionResponse
 */
exports.getAgentIdFromActionResponse = getAgentIdFromActionResponse;
const getActionIdFromActionResponse = actionResponse => {
  if (isLogsEndpointActionResponse(actionResponse)) {
    return actionResponse.EndpointActions.action_id;
  }
  return actionResponse.action_id;
};

// common helpers used by old and new log API
exports.getActionIdFromActionResponse = getActionIdFromActionResponse;
const getDateFilters = ({
  startDate,
  endDate
}) => {
  const dateFilters = [];
  if (startDate) {
    dateFilters.push({
      range: {
        '@timestamp': {
          gte: startDate
        }
      }
    });
  }
  if (endDate) {
    dateFilters.push({
      range: {
        '@timestamp': {
          lte: endDate
        }
      }
    });
  }
  return dateFilters;
};
exports.getDateFilters = getDateFilters;
const getUniqueLogData = activityLogEntries => {
  // find the error responses for actions that didn't make it to fleet index
  const onlyResponsesForFleetErrors = activityLogEntries.reduce((acc, curr) => {
    var _curr$item$data$error;
    if (curr.type === _types.ActivityLogItemTypes.RESPONSE && ((_curr$item$data$error = curr.item.data.error) === null || _curr$item$data$error === void 0 ? void 0 : _curr$item$data$error.code) === _constants.failedFleetActionErrorCode) {
      acc.push(curr.item.data.EndpointActions.action_id);
    }
    return acc;
  }, []);

  // all actions and responses minus endpoint actions.
  const nonEndpointActionsDocs = activityLogEntries.filter(e => e.type !== _types.ActivityLogItemTypes.ACTION);

  // only endpoint actions that match the error responses
  const onlyEndpointActionsDocWithoutFleetActions = activityLogEntries.filter(e => e.type === _types.ActivityLogItemTypes.ACTION && onlyResponsesForFleetErrors.includes(e.item.data.EndpointActions.action_id));

  // join the error actions and the rest
  return [...nonEndpointActionsDocs, ...onlyEndpointActionsDocWithoutFleetActions];
};
exports.getUniqueLogData = getUniqueLogData;
const matchesDsNamePattern = ({
  dataStreamName,
  index
}) => index.includes(dataStreamName);
const categorizeResponseResults = ({
  results
}) => {
  return results !== null && results !== void 0 && results.length ? results === null || results === void 0 ? void 0 : results.map(e => {
    const isResponseDoc = matchesDsNamePattern({
      dataStreamName: _constants.ENDPOINT_ACTION_RESPONSES_DS,
      index: e._index
    });
    return isResponseDoc ? {
      type: _types.ActivityLogItemTypes.RESPONSE,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      item: {
        id: e._id,
        data: e._source
      }
    } : {
      type: _types.ActivityLogItemTypes.FLEET_RESPONSE,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      item: {
        id: e._id,
        data: e._source
      }
    };
  }) : [];
};
exports.categorizeResponseResults = categorizeResponseResults;
const categorizeActionResults = ({
  results
}) => {
  return results !== null && results !== void 0 && results.length ? results === null || results === void 0 ? void 0 : results.map(e => {
    const isActionDoc = matchesDsNamePattern({
      dataStreamName: _constants.ENDPOINT_ACTIONS_DS,
      index: e._index
    });
    return isActionDoc ? {
      type: _types.ActivityLogItemTypes.ACTION,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      item: {
        id: e._id,
        data: e._source
      }
    } : {
      type: _types.ActivityLogItemTypes.FLEET_ACTION,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      item: {
        id: e._id,
        data: e._source
      }
    };
  }) : [];
};

// for 8.4+ we only search on endpoint actions index
// and thus there are only endpoint actions in the results
exports.categorizeActionResults = categorizeActionResults;
const formatEndpointActionResults = results => {
  return results !== null && results !== void 0 && results.length ? results === null || results === void 0 ? void 0 : results.map(e => {
    return {
      type: _types.ActivityLogItemTypes.ACTION,
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      item: {
        id: e._id,
        data: e._source
      }
    };
  }) : [];
};

/**
 * Retrieves the hosts name for each agent ID provided on input.
 * Note that if any ID provided is not a fleet agent ID (ex. 3rd party EDR agent id),
 * then no host name will be returned for agent.
 * @param agentIds
 * @param metadataService
 */
exports.formatEndpointActionResults = formatEndpointActionResults;
const getAgentHostNamesWithIds = async ({
  agentIds,
  endpointService,
  spaceId
}) => {
  if (agentIds.length === 0) {
    return {};
  }
  const fleetServices = endpointService.getInternalFleetServices(spaceId);
  const agentFound = await fleetServices.agent.getByIds(agentIds, {
    ignoreMissing: true
  }).catch(_utils2.catchAndWrapError);
  const agentDocById = (0, _lodash.keyBy)(agentFound, 'id');
  return agentIds.reduce((acc, id) => {
    var _agentDocById$id, _agentDocById$id$loca;
    const agentHostInfo = (_agentDocById$id = agentDocById[id]) === null || _agentDocById$id === void 0 ? void 0 : (_agentDocById$id$loca = _agentDocById$id.local_metadata) === null || _agentDocById$id$loca === void 0 ? void 0 : _agentDocById$id$loca.host;
    acc[id] = (agentHostInfo === null || agentHostInfo === void 0 ? void 0 : agentHostInfo.name) || (agentHostInfo === null || agentHostInfo === void 0 ? void 0 : agentHostInfo.hostname) || '';
    return acc;
  }, {});
};
exports.getAgentHostNamesWithIds = getAgentHostNamesWithIds;
const createActionDetailsRecord = (actionRequest, actionResponses, agentHostInfo) => {
  const {
    isCompleted,
    completedAt,
    wasSuccessful,
    errors,
    outputs,
    agentState
  } = getActionCompletionInfo(actionRequest, actionResponses);
  const {
    isExpired,
    status
  } = getActionStatus({
    expirationDate: actionRequest.expiration,
    isCompleted,
    wasSuccessful
  });
  const actionDetails = {
    action: actionRequest.id,
    id: actionRequest.id,
    agentType: actionRequest.agentType,
    agents: actionRequest.agents,
    hosts: actionRequest.agents.reduce((acc, id) => {
      var _actionRequest$hosts$;
      acc[id] = {
        name: agentHostInfo[id] || ((_actionRequest$hosts$ = actionRequest.hosts[id]) === null || _actionRequest$hosts$ === void 0 ? void 0 : _actionRequest$hosts$.name) || ''
      };
      return acc;
    }, {}),
    command: actionRequest.command,
    startedAt: actionRequest.createdAt,
    isCompleted,
    completedAt,
    wasSuccessful,
    errors,
    isExpired,
    status,
    outputs,
    agentState,
    createdBy: actionRequest.createdBy,
    comment: actionRequest.comment,
    parameters: actionRequest.parameters,
    alertIds: actionRequest.alertIds,
    ruleId: actionRequest.ruleId,
    ruleName: actionRequest.ruleName
  };
  return actionDetails;
};
exports.createActionDetailsRecord = createActionDetailsRecord;
const getActionRequestExpiration = () => {
  return (0, _moment.default)().add(2, 'weeks').toISOString();
};
exports.getActionRequestExpiration = getActionRequestExpiration;