"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentStatusRoute = exports.getAgentStatusRouteHandler = void 0;
var _common = require("@kbn/spaces-plugin/common");
var _error_handler = require("../error_handler");
var _get_agent_status_route = require("../../../../common/api/endpoint/agent/get_agent_status_route");
var _constants = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _services = require("../../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerAgentStatusRoute = (router, endpointContext) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.AGENT_STATUS_ROUTE,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    options: {
      authRequired: true
    }
  }).addVersion({
    version: '1',
    validate: {
      request: _get_agent_status_route.EndpointAgentStatusRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canReadSecuritySolution']
  }, endpointContext.logFactory.get('agentStatusRoute'), getAgentStatusRouteHandler(endpointContext)));
};
exports.registerAgentStatusRoute = registerAgentStatusRoute;
const getAgentStatusRouteHandler = endpointContext => {
  const logger = endpointContext.logFactory.get('agentStatusRoute');
  return async (context, request, response) => {
    const {
      agentType = 'endpoint',
      agentIds: _agentIds
    } = request.query;
    const agentIds = Array.isArray(_agentIds) ? _agentIds : [_agentIds];
    logger.debug(`Retrieving status for: agentType [${agentType}], agentIds: [${agentIds.join(', ')}]`);
    try {
      const [securitySolutionPlugin, corePlugin, actionsPlugin] = await Promise.all([context.securitySolution, context.core, context.actions]);
      const esClient = corePlugin.elasticsearch.client.asInternalUser;
      const spaceId = endpointContext.service.experimentalFeatures.endpointManagementSpaceAwarenessEnabled ? securitySolutionPlugin.getSpaceId() : _common.DEFAULT_SPACE_ID;
      const soClient = endpointContext.service.savedObjects.createInternalScopedSoClient({
        spaceId
      });
      const connectorActionsClient = actionsPlugin.getActionsClient();
      const agentStatusClient = (0, _services.getAgentStatusClient)(agentType, {
        esClient,
        soClient,
        spaceId,
        connectorActionsClient,
        endpointService: endpointContext.service
      });
      const data = await agentStatusClient.getAgentStatuses(agentIds);
      return response.ok({
        body: {
          data
        }
      });
    } catch (e) {
      return (0, _error_handler.errorHandler)(logger, response, e);
    }
  };
};
exports.getAgentStatusRouteHandler = getAgentStatusRouteHandler;