"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeResponseAction = exports.ensureUserHasAuthzToFilesForAction = exports.doesActionHaveFileAccess = exports.createCancelActionAdditionalChecks = exports.buildResponseActionResult = void 0;
var _std = require("@kbn/std");
var _lodash = require("lodash");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _is_response_action_supported = require("../../../../common/endpoint/service/response_actions/is_response_action_supported");
var _errors = require("../../errors");
var _fetch_action_request_by_id = require("../../services/actions/utils/fetch_action_request_by_id");
var _cancel_authz_utils = require("../../../../common/endpoint/service/authz/cancel_authz_utils");
var _constants = require("../../services/actions/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// FYI: this object here should help to quickly catch instances where we might forget to update the
//      authz on the file info/download apis when a response action needs to support file downloads.
const COMMANDS_WITH_ACCESS_TO_FILES = (0, _std.deepFreeze)({
  'get-file': {
    endpoint: true,
    sentinel_one: true,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  execute: {
    endpoint: true,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'running-processes': {
    endpoint: false,
    sentinel_one: true,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  upload: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  scan: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  isolate: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  unisolate: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'kill-process': {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  'suspend-process': {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  },
  runscript: {
    endpoint: false,
    sentinel_one: true,
    crowdstrike: false,
    microsoft_defender_endpoint: true
  },
  cancel: {
    endpoint: false,
    sentinel_one: false,
    crowdstrike: false,
    microsoft_defender_endpoint: false
  }
});

/**
 * Returns boolean indicating if the response action has access to files.
 * NOTE that this utility DOES NOT check privileges.
 * @param agentType
 * @param action
 */
const doesActionHaveFileAccess = (agentType, action) => {
  return (0, _lodash.get)(COMMANDS_WITH_ACCESS_TO_FILES, `${action}.${agentType}`, false);
};

/**
 * Checks to ensure that the user has the correct authz for the response action associated with the action id.
 *
 * FYI: Additional check is needed because the File info and download APIs are used by multiple response actions,
 *      thus we want to ensure that we don't allow access to file associated with response actions the user does
 *      not have authz to.
 *
 * @param context
 * @param request
 */
exports.doesActionHaveFileAccess = doesActionHaveFileAccess;
const ensureUserHasAuthzToFilesForAction = async (context, request) => {
  const securitySolution = await context.securitySolution;
  const spaceId = securitySolution.getSpaceId();
  const endpointService = securitySolution.getEndpointService();
  const userAuthz = await securitySolution.getEndpointAuthz();
  const logger = endpointService.createLogger('ensureUserHasAuthzToFilesForAction');
  const {
    action_id: actionId
  } = request.params;
  logger.debug(`Validating action id [${actionId}] has access to files in space [${spaceId}]`);
  const {
    EndpointActions: {
      data: {
        command
      },
      input_type: agentType
    }
  } = await (0, _fetch_action_request_by_id.fetchActionRequestById)(endpointService, spaceId, actionId);
  logger.debug(`Action [${actionId}] is for command [${command}] with agentType [${agentType}]`);

  // Check if command is supported by the agent type
  if (!(0, _is_response_action_supported.isActionSupportedByAgentType)(agentType, command, 'manual')) {
    throw new _custom_http_request_error.CustomHttpRequestError(`Response action [${command}] not supported for agent type [${agentType}]`, 400);
  }

  // Check if the command is marked as having access to files
  if (!(0, _lodash.get)(COMMANDS_WITH_ACCESS_TO_FILES, `${command}.${agentType}`, false)) {
    throw new _custom_http_request_error.CustomHttpRequestError(`Response action [${command}] for agent type [${agentType}] does not support file downloads`, 400);
  }
  let hasAuthzToCommand = false;
  switch (command) {
    case 'get-file':
      hasAuthzToCommand = userAuthz.canWriteFileOperations;
      break;
    case 'execute':
      hasAuthzToCommand = userAuthz.canWriteExecuteOperations;
      break;
    case 'running-processes':
      hasAuthzToCommand = userAuthz.canGetRunningProcesses;
      break;
    case 'runscript':
      hasAuthzToCommand = userAuthz.canWriteExecuteOperations;
      break;
    case 'cancel':
      hasAuthzToCommand = userAuthz.canCancelAction;
      break;
  }
  if (!hasAuthzToCommand) {
    throw new _errors.EndpointAuthorizationError();
  }
};

/**
 * Executes a response action using the appropriate client method based on the command type
 */
exports.ensureUserHasAuthzToFilesForAction = ensureUserHasAuthzToFilesForAction;
const executeResponseAction = async (command, requestBody, responseActionsClient) => {
  switch (command) {
    case 'isolate':
      return responseActionsClient.isolate(requestBody);
    case 'unisolate':
      return responseActionsClient.release(requestBody);
    case 'running-processes':
      return responseActionsClient.runningProcesses(requestBody);
    case 'execute':
      return responseActionsClient.execute(requestBody);
    case 'suspend-process':
      return responseActionsClient.suspendProcess(requestBody);
    case 'kill-process':
      return responseActionsClient.killProcess(requestBody);
    case 'get-file':
      return responseActionsClient.getFile(requestBody);
    case 'upload':
      return responseActionsClient.upload(requestBody);
    case 'scan':
      return responseActionsClient.scan(requestBody);
    case 'runscript':
      return responseActionsClient.runscript(requestBody);
    case 'cancel':
      return responseActionsClient.cancel(requestBody);
    default:
      throw new _custom_http_request_error.CustomHttpRequestError(`No handler found for response action command: [${command}]`, 501);
  }
};

/**
 * Builds the standardized response object for response actions
 */
exports.executeResponseAction = executeResponseAction;
const buildResponseActionResult = (command, action) => {
  var _ref;
  const {
    action: actionId,
    ...data
  } = action;
  const legacyResponseData = _constants.responseActionsWithLegacyActionProperty.includes(command) ? {
    action: (_ref = actionId !== null && actionId !== void 0 ? actionId : data.id) !== null && _ref !== void 0 ? _ref : ''
  } : {};
  return {
    body: {
      ...legacyResponseData,
      data
    }
  };
};

/**
 * Creates additional authorization checks function for cancel action.
 * Business logic validation has been moved to the service layer (validateRequest).
 * This function only handles HTTP-specific authorization checks.
 */
exports.buildResponseActionResult = buildResponseActionResult;
const createCancelActionAdditionalChecks = endpointContext => {
  return async (context, request) => {
    var _originalAction$Endpo, _originalAction$Endpo2, _originalAction$Endpo3;
    const {
      parameters
    } = request.body;
    const actionId = parameters.id;
    const logger = endpointContext.logFactory.get('cancelActionAdditionalChecks');
    // Get space ID from context
    const spaceId = (await context.securitySolution).getSpaceId();

    // Fetch original action to determine command type and agent type
    const originalAction = await (0, _fetch_action_request_by_id.fetchActionRequestById)(endpointContext.service, spaceId, actionId);
    if (!originalAction) {
      throw new _custom_http_request_error.CustomHttpRequestError(`Action with id '${actionId}' not found.`, 404);
    }

    // Extract command and agent type from original action
    const command = (_originalAction$Endpo = originalAction.EndpointActions) === null || _originalAction$Endpo === void 0 ? void 0 : (_originalAction$Endpo2 = _originalAction$Endpo.data) === null || _originalAction$Endpo2 === void 0 ? void 0 : _originalAction$Endpo2.command;
    const agentType = (_originalAction$Endpo3 = originalAction.EndpointActions) === null || _originalAction$Endpo3 === void 0 ? void 0 : _originalAction$Endpo3.input_type;
    if (!command) {
      logger.error(`Action ${actionId} missing command information for ${agentType}`);
      throw new _custom_http_request_error.CustomHttpRequestError(`Unable to determine command type for action '${actionId}'`, 500);
    }
    if (!agentType) {
      logger.error(`Action ${actionId} missing agent type information for ${command}`);
      throw new _custom_http_request_error.CustomHttpRequestError(`Unable to determine agent type for action '${actionId}'`, 500);
    }

    // Use utility to check if cancellation is allowed
    const endpointAuthz = await (await context.securitySolution).getEndpointAuthz();
    const canCancel = (0, _cancel_authz_utils.checkCancelPermission)(endpointAuthz, endpointContext.experimentalFeatures, agentType, command);
    if (!canCancel) {
      throw new _errors.EndpointAuthorizationError();
    }
  };
};
exports.createCancelActionAdditionalChecks = createCancelActionAdditionalChecks;