"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useParseFileInput = void 0;
var _react = require("react");
var _file_upload_error = require("../translations/file_upload_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useParseFileInput = onFileParsed => {
  const [isParsing, setIsParsing] = (0, _react.useState)(false);
  const [error, setError] = (0, _react.useState)();
  const parseFile = (0, _react.useCallback)(files => {
    setError(undefined);
    if (!files || files.length === 0) {
      return;
    }
    const file = files[0];
    const reader = new FileReader();
    reader.onloadstart = () => setIsParsing(true);
    reader.onloadend = () => setIsParsing(false);
    reader.onload = function (e) {
      var _e$target;
      // We can safely cast to string since we call `readAsText` to load the file.
      const fileContent = (_e$target = e.target) === null || _e$target === void 0 ? void 0 : _e$target.result;
      if (fileContent == null) {
        setError(_file_upload_error.FILE_UPLOAD_ERROR.CAN_NOT_READ);
        return;
      }
      if (fileContent === '' && e.loaded > 100000) {
        // V8-based browsers can't handle large files and return an empty string
        // instead of an error; see https://stackoverflow.com/a/61316641
        setError(_file_upload_error.FILE_UPLOAD_ERROR.TOO_LARGE_TO_PARSE);
        return;
      }
      try {
        const parsedData = parseContent(fileContent);
        onFileParsed(parsedData);
      } catch (err) {
        setError(err.message);
      }
    };
    const handleReaderError = function () {
      var _reader$error;
      const message = (_reader$error = reader.error) === null || _reader$error === void 0 ? void 0 : _reader$error.message;
      if (message) {
        setError(_file_upload_error.FILE_UPLOAD_ERROR.CAN_NOT_READ_WITH_REASON(message));
      } else {
        setError(_file_upload_error.FILE_UPLOAD_ERROR.CAN_NOT_READ);
      }
    };
    reader.onerror = handleReaderError;
    reader.onabort = handleReaderError;
    reader.readAsText(file);
  }, [onFileParsed]);
  return {
    parseFile,
    isParsing,
    error
  };
};
exports.useParseFileInput = useParseFileInput;
const parseContent = fileContent => {
  const trimmedContent = fileContent.trim();
  let arrayContent;
  if (trimmedContent.startsWith('[')) {
    arrayContent = parseJSONArray(trimmedContent);
  } else {
    arrayContent = parseNDJSON(trimmedContent);
  }
  if (arrayContent.length === 0) {
    throw new Error(_file_upload_error.FILE_UPLOAD_ERROR.EMPTY);
  }
  return arrayContent;
};
const parseNDJSON = fileContent => {
  return fileContent.split(/\n(?=\{)/) // split at newline followed by '{'.
  .filter(entry => entry.trim() !== '') // Remove empty entries.
  .map(parseJSON); // Parse each entry as JSON.
};
const parseJSONArray = fileContent => {
  const parsedContent = parseJSON(fileContent);
  if (!Array.isArray(parsedContent)) {
    throw new Error(_file_upload_error.FILE_UPLOAD_ERROR.NOT_ARRAY);
  }
  return parsedContent;
};
const parseJSON = fileContent => {
  try {
    return JSON.parse(fileContent);
  } catch (error) {
    if (error instanceof RangeError) {
      throw new Error(_file_upload_error.FILE_UPLOAD_ERROR.TOO_LARGE_TO_PARSE);
    }
    throw new Error(_file_upload_error.FILE_UPLOAD_ERROR.CAN_NOT_PARSE);
  }
};