"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateEaseAlertsDataView = exports.RUNTIME_FIELD_MAP = exports.DATA_VIEW_ID = void 0;
var _react = require("react");
var _constants = require("../../constants");
var _use_space_id = require("../../../common/hooks/use_space_id");
var _constants2 = require("../../../../common/constants");
var _use_create_data_view = require("../../../common/hooks/use_create_data_view");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Unique id for the ad-hoc dataView created for the alert summary page (used so we can retrieve the DataView from the flyout)
const DATA_VIEW_ID = exports.DATA_VIEW_ID = 'alert-summary-data-view-id';

// Runtime field to extract the related integration package name from the alert rule parameters field
const RUNTIME_FIELD_MAP = exports.RUNTIME_FIELD_MAP = {
  [_constants.RELATED_INTEGRATION]: {
    type: 'keyword',
    script: {
      source: `if (params._source.containsKey('kibana.alert.rule.parameters') && params._source['kibana.alert.rule.parameters'].containsKey('related_integrations')) { def integrations = params._source['kibana.alert.rule.parameters']['related_integrations']; if (integrations != null && integrations.size() > 0 && integrations[0].containsKey('package')) { emit(integrations[0]['package']); } }`
    }
  }
};
/**
 * Hook that creates a DataView for the EASE pages (alert summary, attack discovery and case pages as well as for the alert details flyout.
 * - it takes into account the current space.
 * - it uses the DEFAULT_ALERTS_INDEX as the index pattern.
 * - it adds a runtime field to extract the related integration package name from the alert rule parameters field.
 * - we pass a constant id, to prevent recreating the same dataView on other pages or in the flyout. For this we rely on the cache built in the DataViewService.
 *
 * It returns a null dataView if the space is undefined, to prevent unnecessary renders of the components.
 */
const useCreateEaseAlertsDataView = () => {
  const spaceId = (0, _use_space_id.useSpaceId)();
  const signalIndexName = (0, _react.useMemo)(() => spaceId ? `${_constants2.DEFAULT_ALERTS_INDEX}-${spaceId}` : '', [spaceId]);
  const dataViewId = (0, _react.useMemo)(() => spaceId ? `${DATA_VIEW_ID}-${spaceId}` : '', [spaceId]);
  const dataViewSpec = (0, _react.useMemo)(() => ({
    id: dataViewId,
    runtimeFieldMap: RUNTIME_FIELD_MAP,
    title: signalIndexName
  }), [dataViewId, signalIndexName]);

  // If the spaceId is undefined, we skip creating the dataView
  const {
    dataView,
    loading
  } = (0, _use_create_data_view.useCreateDataView)({
    dataViewSpec,
    skip: !spaceId
  });
  return {
    dataView,
    loading
  };
};
exports.useCreateEaseAlertsDataView = useCreateEaseAlertsDataView;