"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.singleEntryThreat = exports.getUpdatedEntriesOnDelete = exports.getFormattedEntry = exports.getFormattedEntries = exports.getEntryOnThreatFieldChange = exports.getEntryOnMatchChange = exports.getEntryOnFieldChange = exports.createOrNewEntryItem = exports.createAndNewEntryItem = exports.containsInvalidItems = void 0;
var _uuid = require("uuid");
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Formats the entry into one that is easily usable for the UI.
 */
const getFormattedEntry = (dataView, threatDataView, item, itemIndex, uuidGen = _uuid.v4) => {
  var _maybeId$id, _item$negate;
  const {
    fields
  } = dataView;
  const {
    fields: threatFields
  } = threatDataView;
  const field = item.field;
  const threatField = item.value;
  const [foundField] = fields.filter(({
    name
  }) => field != null && field === name);
  const [threatFoundField] = threatFields.filter(({
    name
  }) => threatField != null && threatField === name);
  const maybeId = item;

  // Fallback to a string field when field isn't found in known fields.
  // It's required for showing field's value when appropriate data is missing in ES.
  return {
    id: (_maybeId$id = maybeId.id) !== null && _maybeId$id !== void 0 ? _maybeId$id : uuidGen(),
    field: foundField !== null && foundField !== void 0 ? foundField : {
      name: field,
      type: 'string'
    },
    type: 'mapping',
    value: threatFoundField !== null && threatFoundField !== void 0 ? threatFoundField : {
      name: threatField,
      type: 'string'
    },
    entryIndex: itemIndex,
    negate: (_item$negate = item.negate) !== null && _item$negate !== void 0 ? _item$negate : false // Default to false if negate is not provided
  };
};

/**
 * Formats the entries to be easily usable for the UI
 *
 * @param patterns DataViewBase containing available fields on rule index
 * @param entries item entries
 */
exports.getFormattedEntry = getFormattedEntry;
const getFormattedEntries = (indexPattern, threatIndexPatterns, entries) => {
  return entries.reduce((acc, item, index) => {
    const newItemEntry = getFormattedEntry(indexPattern, threatIndexPatterns, item, index);
    return [...acc, newItemEntry];
  }, []);
};

/**
 * Determines whether an entire entry or item need to be removed
 *
 * @param item
 * @param entryIndex index of given entry
 *
 */
exports.getFormattedEntries = getFormattedEntries;
const getUpdatedEntriesOnDelete = (item, entryIndex) => {
  return {
    ...item,
    entries: [...item.entries.slice(0, entryIndex), ...item.entries.slice(entryIndex + 1)]
  };
};

/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current item entry values
 * @param newField - newly selected field
 *
 */
exports.getUpdatedEntriesOnDelete = getUpdatedEntriesOnDelete;
const getEntryOnFieldChange = (item, newField) => {
  var _item$negate2;
  const {
    entryIndex
  } = item;
  return {
    updatedEntry: {
      id: item.id,
      field: newField != null ? newField.name : '',
      type: 'mapping',
      value: item.value != null ? item.value.name : '',
      negate: (_item$negate2 = item.negate) !== null && _item$negate2 !== void 0 ? _item$negate2 : false
    },
    // Cast to ThreatMappingEntry since id is only used as a react key prop and can be ignored elsewhere
    index: entryIndex
  };
};

/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current item entry values
 * @param newField - newly selected field
 *
 */
exports.getEntryOnFieldChange = getEntryOnFieldChange;
const getEntryOnThreatFieldChange = (item, newField) => {
  var _item$negate3;
  const {
    entryIndex
  } = item;
  return {
    updatedEntry: {
      id: item.id,
      field: item.field != null ? item.field.name : '',
      type: 'mapping',
      value: newField != null ? newField.name : '',
      negate: (_item$negate3 = item.negate) !== null && _item$negate3 !== void 0 ? _item$negate3 : false
    },
    // Cast to ThreatMappingEntry since id is only used as a react key prop and can be ignored elsewhere
    index: entryIndex
  };
};
exports.getEntryOnThreatFieldChange = getEntryOnThreatFieldChange;
const getEntryOnMatchChange = (item, newNegate) => {
  var _field$name, _value$name;
  const {
    entryIndex,
    field,
    value,
    id
  } = item;
  return {
    updatedEntry: {
      id,
      field: (_field$name = field === null || field === void 0 ? void 0 : field.name) !== null && _field$name !== void 0 ? _field$name : '',
      type: 'mapping',
      value: (_value$name = value === null || value === void 0 ? void 0 : value.name) !== null && _value$name !== void 0 ? _value$name : '',
      negate: newNegate !== null && newNegate !== void 0 ? newNegate : false
    },
    index: entryIndex
  };
};
exports.getEntryOnMatchChange = getEntryOnMatchChange;
const createAndNewEntryItem = () => {
  return (0, _securitysolutionUtils.addIdToItem)({
    field: '',
    type: 'mapping',
    value: '',
    negate: false
  });
};
exports.createAndNewEntryItem = createAndNewEntryItem;
const createOrNewEntryItem = () => {
  return (0, _securitysolutionUtils.addIdToItem)({
    entries: [(0, _securitysolutionUtils.addIdToItem)({
      field: '',
      type: 'mapping',
      value: '',
      negate: false
    })]
  });
};

/**
 * Given a list of items checks each one to see if any of them have an empty field
 * or an empty value.
 * @param items The items to check if we have an empty entries.
 */
exports.createOrNewEntryItem = createOrNewEntryItem;
const containsInvalidItems = items => {
  return items.some(item => item.entries.some(subEntry => subEntry.field === '' || subEntry.value === ''));
};

/**
 * Given a list of items checks if we have a single empty entry and if we do returns true.
 * @param items The items to check if we have a single empty entry.
 */
exports.containsInvalidItems = containsInvalidItems;
const singleEntryThreat = items => {
  return items.length === 1 && items[0].entries.length === 1 && items[0].entries[0].field === '' && items[0].entries[0].value === '';
};
exports.singleEntryThreat = singleEntryThreat;