"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInternalListRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _reportingCommon = require("@kbn/reporting-common");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _scheduled_reports = require("../../../../services/scheduled_reports");
var _common = require("../../../common");
var _request_handler = require("../../../common/request_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  SCHEDULED
} = _reportingCommon.INTERNAL_ROUTES;
const registerInternalListRoute = ({
  logger,
  router,
  reporting
}) => {
  // list scheduled jobs in the queue, paginated
  const path = SCHEDULED.LIST;
  router.get({
    path,
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization because reporting uses its own authorization model.'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        page: _configSchema.schema.string({
          defaultValue: '1'
        }),
        size: _configSchema.schema.string({
          defaultValue: `${_scheduled_reports.DEFAULT_SCHEDULED_REPORT_LIST_SIZE}`,
          validate: value => {
            try {
              const size = parseInt(value, 10);
              if (size < 1 || size > _scheduled_reports.MAX_SCHEDULED_REPORT_LIST_SIZE) {
                return `size must be between 1 and ${_scheduled_reports.MAX_SCHEDULED_REPORT_LIST_SIZE}: size: ${value}`;
              }
            } catch (e) {
              return `size must be an integer: size: ${value}`;
            }
          }
        })
      })
    },
    options: {
      access: 'internal'
    }
  }, (0, _common.authorizedUserPreRouting)(reporting, async (user, context, req, res) => {
    try {
      const counters = (0, _common.getCounters)(req.route.method, path, reporting.getUsageCounter());

      // ensure the async dependencies are loaded
      if (!context.reporting) {
        return (0, _request_handler.handleUnavailable)(res);
      }

      // check license
      const licenseInfo = await reporting.getLicenseInfo();
      const licenseResults = licenseInfo.scheduledReports;
      if (!licenseResults.enableLinks) {
        return res.forbidden({
          body: licenseResults.message
        });
      }
      const {
        page: queryPage = '1',
        size: querySize = `${_scheduled_reports.DEFAULT_SCHEDULED_REPORT_LIST_SIZE}`
      } = req.query;
      const page = parseInt(queryPage, 10) || 1;
      const size = Math.min(_scheduled_reports.MAX_SCHEDULED_REPORT_LIST_SIZE, parseInt(querySize, 10) || _scheduled_reports.DEFAULT_SCHEDULED_REPORT_LIST_SIZE);
      const scheduledReportsService = await _scheduled_reports.ScheduledReportsService.build({
        logger,
        reportingCore: reporting,
        request: req,
        responseFactory: res
      });
      const results = await scheduledReportsService.list({
        user,
        page,
        size
      });
      counters.usageCounter();
      return res.ok({
        body: results,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (err) {
      if (err instanceof _coreHttpRouterServerInternal.KibanaResponse) {
        return err;
      }
      throw err;
    }
  }));
};
exports.registerInternalListRoute = registerInternalListRoute;