"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ListControl = void 0;
exports.listControlFactory = listControlFactory;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _i18n = require("@kbn/i18n");
var _control = require("./control");
var _phrase_filter_manager = require("./filter_manager/phrase_filter_manager");
var _create_search_source = require("./create_search_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getEscapedQuery(query = '') {
  // https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-regexp-query.html#_standard_operators
  return query.replace(/[.?+*|{}[\]()"\\#@&<>~]/g, match => `\\${match}`);
}
const termsAgg = ({
  field,
  size,
  direction,
  query
}) => {
  const terms = {
    order: {
      _count: direction
    }
  };
  if (size) {
    terms.size = size < 1 ? 1 : size;
  }
  if (field !== null && field !== void 0 && field.scripted) {
    terms.script = {
      source: field.script,
      lang: field.lang
    };
    terms.value_type = field.type === 'number' ? 'float' : field.type;
  } else {
    terms.field = field === null || field === void 0 ? void 0 : field.name;
  }
  if (query) {
    terms.include = `.*${getEscapedQuery(query)}.*`;
  }
  return {
    termsAgg: {
      terms
    }
  };
};
class ListControl extends _control.Control {
  constructor(controlParams, filterManager, useTimeFilter, _searchSource, deps) {
    super(controlParams, filterManager, useTimeFilter);
    (0, _defineProperty2.default)(this, "getSettings", void 0);
    (0, _defineProperty2.default)(this, "timefilter", void 0);
    (0, _defineProperty2.default)(this, "searchSource", void 0);
    (0, _defineProperty2.default)(this, "abortController", void 0);
    (0, _defineProperty2.default)(this, "lastAncestorValues", void 0);
    (0, _defineProperty2.default)(this, "lastQuery", void 0);
    (0, _defineProperty2.default)(this, "partialResults", void 0);
    (0, _defineProperty2.default)(this, "selectOptions", void 0);
    (0, _defineProperty2.default)(this, "fetch", async query => {
      // Abort any in-progress fetch
      if (this.abortController) {
        this.abortController.abort();
      }
      this.abortController = new AbortController();
      const indexPattern = this.filterManager.getIndexPattern();
      if (!indexPattern) {
        this.disable((0, _control.noIndexPatternMsg)(this.controlParams.indexPattern));
        return;
      }
      let ancestorFilters;
      if (this.hasAncestors()) {
        if (this.hasUnsetAncestor()) {
          this.disable(_i18n.i18n.translate('inputControl.listControl.disableTooltip', {
            defaultMessage: "Disabled until ''{label}'' is set.",
            values: {
              label: this.ancestors[0].label
            }
          }));
          this.lastAncestorValues = undefined;
          return;
        }
        const ancestorValues = this.getAncestorValues();
        if (_lodash.default.isEqual(ancestorValues, this.lastAncestorValues) && _lodash.default.isEqual(query, this.lastQuery)) {
          // short circuit to avoid fetching options list for same ancestor values
          return;
        }
        this.lastAncestorValues = ancestorValues;
        this.lastQuery = query;
        ancestorFilters = this.getAncestorFilters();
      }
      const fieldName = this.filterManager.fieldName;
      const settings = await this.getSettings();
      const initialSearchSourceState = {
        timeout: `${settings.autocompleteTimeout}ms`,
        terminate_after: Number(settings.autocompleteTerminateAfter)
      };

      // dynamic options are only allowed on String fields but the setting defaults to true so it could
      // be enabled for non-string fields (since UI input is hidden for non-string fields).
      // If field is not string, then disable dynamic options.
      const field = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.fields.getAll().find(({
        name
      }) => name === this.controlParams.fieldName);
      if (field && field.type !== 'string') {
        this.options.dynamicOptions = false;
      }
      const aggs = termsAgg({
        field: indexPattern.fields.getByName(fieldName),
        size: this.options.dynamicOptions ? null : _lodash.default.get(this.options, 'size', 5),
        direction: 'desc',
        query
      });
      const searchSource = await (0, _create_search_source.createSearchSource)(this.searchSource, initialSearchSourceState, indexPattern, aggs, this.useTimeFilter, ancestorFilters, this.timefilter);
      const abortSignal = this.abortController.signal;
      this.lastQuery = query;
      let resp;
      try {
        resp = await searchSource.fetch({
          abortSignal
        });
      } catch (error) {
        // If the fetch was aborted then no need to surface this error in the UI
        if (error.name === 'AbortError') return;
        this.disable(_i18n.i18n.translate('inputControl.listControl.unableToFetchTooltip', {
          defaultMessage: 'Unable to fetch terms, error: {errorMessage}',
          values: {
            errorMessage: error.message
          }
        }));
        return;
      }
      if (query && this.lastQuery !== query) {
        // search results returned out of order - ignore results from old query
        return;
      }
      const selectOptions = _lodash.default.get(resp, 'aggregations.termsAgg.buckets', []).map(bucket => {
        return bucket === null || bucket === void 0 ? void 0 : bucket.key;
      });
      if (selectOptions.length === 0 && !query) {
        this.disable((0, _control.noValuesDisableMsg)(fieldName, indexPattern.title));
        return;
      }

      // TODO: terminated_early is missing from response definition.
      // https://github.com/elastic/elasticsearch-js/issues/1289
      this.partialResults = resp.terminated_early || resp.timed_out;
      this.selectOptions = selectOptions;
      this.enable = true;
      this.disabledReason = '';
    });
    this.getSettings = deps.getSettings;
    this.timefilter = deps.data.query.timefilter.timefilter;
    this.searchSource = _searchSource;
  }
  destroy() {
    if (this.abortController) this.abortController.abort();
  }
  hasValue() {
    return typeof this.value !== 'undefined' && this.value.length > 0;
  }
}
exports.ListControl = ListControl;
async function listControlFactory(controlParams, useTimeFilter, deps) {
  const [, {
    data: dataPluginStart
  }] = await deps.core.getStartServices();
  const listControl = new ListControl(controlParams, new _phrase_filter_manager.PhraseFilterManager(controlParams.id, controlParams.fieldName, controlParams.indexPattern, dataPluginStart.dataViews, deps.data.query.filterManager), useTimeFilter, dataPluginStart.search.searchSource, deps);
  await listControl.filterManager.init();
  return listControl;
}