"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeAsReasoningAgent = executeAsReasoningAgent;
var _inferenceCommon = require("@kbn/inference-common");
var _inferenceTracing = require("@kbn/inference-tracing");
var _api = require("@opentelemetry/api");
var _lodash = require("lodash");
var _create_complete_tool_call = require("./create_complete_tool_call");
var _create_reason_tool_call = require("./create_reason_tool_call");
var _markers = require("./markers");
var _planning_tools = require("./planning_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function prepareMessagesForLLM({
  stepsLeft,
  messages,
  canCallTaskTools,
  canCallPlanningTools
}) {
  /**
   * This removes all system tool calls except if it is the last, to compact the
   * conversation and not distract the LLM with tool calls that don't impact the
   * conversation.
   */
  const lastToolMessage = messages.findLast(message => message.role === _inferenceCommon.MessageRole.Tool);
  let next = messages;
  if (lastToolMessage && (0, _planning_tools.isPlanningToolName)(lastToolMessage.name)) {
    const idx = messages.indexOf(lastToolMessage) - 1;
    next = (0, _planning_tools.removeReasonToolCalls)(messages.slice(0, idx)).concat(messages.slice(idx));
  } else {
    next = (0, _planning_tools.removeReasonToolCalls)(messages);
  }
  const lastToolResponse = next.findLast(message => message.role === _inferenceCommon.MessageRole.Tool);
  return next.map(message => {
    if (message === lastToolResponse) {
      return {
        ...lastToolResponse,
        response: {
          ...(typeof lastToolResponse.response === 'string' ? {
            content: lastToolResponse.response
          } : {}),
          stepsLeft
        }
      };
    }
    return message;
  });
}
/**
 * Executes a prompt in a loop in a way that the LLM will use the specified tools
 * to gather context, and then produce a final output, which may or may not include
 * a final tool call.
 *
 * The rules are as follows:
 * - when `reason()` is called, the LLM SHOULD reason about the task or the tool call
 * results
 * - when `reason()` is called, the LLM CAN call another tool
 * - when `complete()` is called, and `finalToolChoice` is NOT specified, the LLM
 * MUST produce a summarization text
 * - when `complete()` is called, and `finalToolChoice` is specified, the LLM MUST
 * call a tool to complete the task, and the LLM SHOULD produce a summarization text
 * - when `finalToolChoice` is specified, and the LLM calls this tool, the task MUST
 * be completed by the orchestrator
 * - when the available number of steps have been exhausted, the LLM MUST produce
 * its final output
 * - if the LLM fails to produce its final output (e.g. by calling an unavailable tool),
 * the orchestrator MUST complete the task
 * - if `finalToolChoice` is specified and is not included as part of the definitive output,
 * the orchestrator MUST fail the task
 */
async function executeAsReasoningAgent(options) {
  const {
    inferenceClient,
    maxSteps = 10,
    toolCallbacks
  } = options;
  async function callTools(toolCalls) {
    return await Promise.all(toolCalls.map(async toolCall => {
      if ((0, _planning_tools.isPlanningToolName)(toolCall.function.name)) {
        throw new Error(`Unexpected planning tool call ${toolCall.function.name}`);
      }
      const callback = toolCallbacks[toolCall.function.name];
      const response = await (0, _inferenceTracing.withExecuteToolSpan)(toolCall.function.name, {
        tool: {
          input: 'arguments' in toolCall.function ? toolCall.function.arguments : undefined,
          toolCallId: toolCall.toolCallId
        }
      }, () => callback(toolCall)).catch(error => {
        var _trace$getActiveSpan;
        (_trace$getActiveSpan = _api.trace.getActiveSpan()) === null || _trace$getActiveSpan === void 0 ? void 0 : _trace$getActiveSpan.recordException(error);
        return {
          response: {
            error,
            data: undefined
          }
        };
      });
      return {
        response: response.response,
        data: response.data,
        name: toolCall.function.name,
        toolCallId: toolCall.toolCallId,
        role: _inferenceCommon.MessageRole.Tool
      };
    }));
  }
  async function innerCallPromptUntil({
    messages: givenMessages,
    stepsLeft,
    temperature
  }) {
    // Append a complete() tool call to force the LLM to generate the final response
    const prevMessages = stepsLeft <= 0 ? givenMessages.concat((0, _create_complete_tool_call.createCompleteToolCall)()) : givenMessages;
    const withoutSystemToolCalls = (0, _planning_tools.removeReasonToolCalls)(prevMessages);
    const consecutiveReasoningSteps = (0, _lodash.takeRightWhile)(withoutSystemToolCalls, msg => {
      var _msg$toolCalls;
      return msg.role === _inferenceCommon.MessageRole.Assistant && !((_msg$toolCalls = msg.toolCalls) !== null && _msg$toolCalls !== void 0 && _msg$toolCalls.length);
    }).length;
    const lastSystemToolCall = prevMessages.findLast(msg => msg.role === _inferenceCommon.MessageRole.Tool && (0, _planning_tools.isPlanningToolName)(msg.name));
    const lastSystemToolCallName = lastSystemToolCall === null || lastSystemToolCall === void 0 ? void 0 : lastSystemToolCall.name;
    const isCompleting = lastSystemToolCallName === 'complete';

    // Nudge the LLM to reason if it has not done after the last tool call
    const mustReason = !isCompleting && lastSystemToolCallName === 'reason' && consecutiveReasoningSteps === 0;
    const canCallTaskTools = !mustReason;
    const canCallPlanningTools = !mustReason && !isCompleting;
    const nextPrompt = {
      ...options.prompt,
      versions: options.prompt.versions.map(version => {
        const {
          tools: promptTools,
          ...rest
        } = version;
        const mergedToolOptions = {
          tools: promptTools
        };
        const nextTools = isCompleting ? mergedToolOptions : {
          tools: {
            ...mergedToolOptions.tools,
            ..._planning_tools.PLANNING_TOOLS
          }
        };
        return {
          ...rest,
          ...nextTools
        };
      })
    };
    const promptOptions = {
      ...(0, _lodash.omit)(options, 'finalToolChoice'),
      prompt: nextPrompt
    };
    const response = await inferenceClient.prompt({
      ...promptOptions,
      stream: false,
      temperature,
      toolChoice: isCompleting ? options.finalToolChoice : undefined,
      prevMessages: prepareMessagesForLLM({
        stepsLeft,
        messages: prevMessages,
        canCallTaskTools,
        canCallPlanningTools
      }),
      stopSequences: [_markers.END_INTERNAL_REASONING_MARKER]
    });
    let content = response.content;

    /**
     * If the LLM hasn't used these markers, we assume it wants to complete its
     * output.
     */

    let completeNextTurn = content && !content.includes(_markers.BEGIN_INTERNAL_REASONING_MARKER) && !content.includes(_markers.END_INTERNAL_REASONING_MARKER) && !response.toolCalls.length;

    /**
     * Remove content after <<<END_INTERNAL>>>. This means that the LLM has combined final output
     * with internal reasoning, and it usually leads the LLM into a loop where it repeats itself.
     */

    const [internalContent, ...externalContentParts] = content.split(_markers.END_INTERNAL_REASONING_MARKER);
    const externalContent = externalContentParts.join(_markers.END_INTERNAL_REASONING_MARKER).trim();

    // use some kind of buffer to allow small artifacts around the markers, like markdown.
    if (externalContent.length && externalContent.length > 25) {
      content = internalContent + _markers.END_INTERNAL_REASONING_MARKER;
      completeNextTurn = true;
    }
    const assistantMessage = {
      role: _inferenceCommon.MessageRole.Assistant,
      content,
      toolCalls: response.toolCalls
    };
    const [systemToolCalls, nonSystemToolCalls] = (0, _lodash.partition)(response.toolCalls, toolCall => (0, _planning_tools.isPlanningToolName)(toolCall.function.name));
    if (systemToolCalls.length && response.toolCalls.length > 1) {
      throw new Error(`When using system tools, only a single tool call is allowed`);
    }
    const finalToolCallName = options.finalToolChoice && typeof options.finalToolChoice === 'object' ? options.finalToolChoice.function : undefined;
    const hasCalledFinalTool = response.toolCalls.some(toolCall => toolCall.function.name === finalToolCallName);
    if (isCompleting || hasCalledFinalTool) {
      // We don't want to send these results back to the LLM, if we are already
      // completing
      return {
        content: response.content,
        tokens: response.tokens,
        toolCalls: response.toolCalls.filter(toolCall => toolCall.function.name === finalToolCallName),
        input: (0, _planning_tools.removeSystemToolCalls)(prevMessages)
      };
    }
    const toolMessagesForNonSystemToolCalls = nonSystemToolCalls.length ? (await callTools(nonSystemToolCalls)).map(toolMessage => {
      return {
        ...toolMessage,
        response: {
          ...(typeof toolMessage.response === 'string' ? {
            content: toolMessage.response
          } : toolMessage.response),
          stepsLeft
        }
      };
    }) : [];
    const systemToolMessages = systemToolCalls.map(systemToolCall => {
      if (systemToolCall.function.name === 'reason') {
        return (0, _create_reason_tool_call.createReasonToolCallResponse)(systemToolCall.toolCallId);
      }
      return (0, _create_complete_tool_call.createCompleteToolCallResponse)(systemToolCall.toolCallId);
    });
    const allToolMessages = [...toolMessagesForNonSystemToolCalls, ...systemToolMessages];
    if (completeNextTurn) {
      return innerCallPromptUntil({
        messages: prevMessages.concat(assistantMessage, ...allToolMessages),
        stepsLeft: 0
      });
    }
    return innerCallPromptUntil({
      messages: prevMessages.concat(assistantMessage, ...allToolMessages, ...(nonSystemToolCalls.length ? (0, _create_reason_tool_call.createReasonToolCall)() : [])),
      stepsLeft: stepsLeft - 1
    });
  }
  return await (0, _inferenceTracing.withActiveInferenceSpan)('reason', () => innerCallPromptUntil({
    // nudge the LLM to go into reasoning mode
    messages: (0, _create_reason_tool_call.createReasonToolCall)(),
    stepsLeft: maxSteps
  }));
}