"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateSaveButtonSaga = exports.updateFieldProperties = exports.syncNodeStyleSaga = exports.syncFieldsSaga = exports.selectedFieldsSelector = exports.selectField = exports.loadFields = exports.liveResponseFieldsSelector = exports.hasFieldsSelector = exports.fieldsSelector = exports.fieldsReducer = exports.fieldMapSelector = exports.deselectField = void 0;
var _typescriptFsa = _interopRequireDefault(require("typescript-fsa"));
var _dist = require("typescript-fsa-reducers/dist");
var _reselect = require("reselect");
var _effects = require("redux-saga/effects");
var _global = require("./global");
var _datasource = require("./datasource");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const actionCreator = (0, _typescriptFsa.default)('x-pack/graph/fields');
const loadFields = exports.loadFields = actionCreator('LOAD_FIELDS');
const updateFieldProperties = exports.updateFieldProperties = actionCreator('UPDATE_FIELD_PROPERTIES');
const selectField = exports.selectField = actionCreator('SELECT_FIELD');
const deselectField = exports.deselectField = actionCreator('DESELECT_FIELD');
const initialFields = {};
const fieldsReducer = exports.fieldsReducer = (0, _dist.reducerWithInitialState)(initialFields).case(_global.reset, () => initialFields).case(_datasource.setDatasource, () => initialFields).case(loadFields, (_currentFields, newFields) => {
  const newFieldMap = {};
  newFields.forEach(field => {
    newFieldMap[field.name] = field;
  });
  return newFieldMap;
}).case(updateFieldProperties, (fields, {
  fieldName,
  fieldProperties
}) => {
  return {
    ...fields,
    [fieldName]: {
      ...fields[fieldName],
      ...fieldProperties
    }
  };
}).case(selectField, (fields, fieldName) => {
  return {
    ...fields,
    [fieldName]: {
      ...fields[fieldName],
      selected: true
    }
  };
}).case(deselectField, (fields, fieldName) => {
  return {
    ...fields,
    [fieldName]: {
      ...fields[fieldName],
      selected: false
    }
  };
}).build();
const fieldMapSelector = state => state.fields;
exports.fieldMapSelector = fieldMapSelector;
const fieldsSelector = exports.fieldsSelector = (0, _reselect.createSelector)(fieldMapSelector, fields => Object.values(fields));
const selectedFieldsSelector = exports.selectedFieldsSelector = (0, _reselect.createSelector)(fieldsSelector, fields => fields.filter(field => field.selected));
const liveResponseFieldsSelector = exports.liveResponseFieldsSelector = (0, _reselect.createSelector)(selectedFieldsSelector, fields => fields.filter(field => field.hopSize && field.hopSize > 0));
const hasFieldsSelector = exports.hasFieldsSelector = (0, _reselect.createSelector)(selectedFieldsSelector, fields => fields.length > 0);

/**
 * Saga making notifying react when fields are selected to re-calculate the state of the save button.
 *
 * Won't be necessary once the workspace is moved to redux
 */
const updateSaveButtonSaga = ({
  notifyReact
}) => {
  function* notify() {
    notifyReact();
  }
  return function* () {
    yield (0, _effects.takeLatest)((0, _helpers.matchesOne)(selectField, deselectField), notify);
  };
};

/**
 * Saga making sure the fields in the store are always synced with the fields
 * known to the workspace.
 *
 * Won't be necessary once the workspace is moved to redux
 */
exports.updateSaveButtonSaga = updateSaveButtonSaga;
const syncFieldsSaga = ({
  getWorkspace
}) => {
  function* syncFields() {
    const workspace = getWorkspace();
    if (!workspace) {
      return;
    }
    const currentState = yield (0, _effects.select)();
    workspace.options.vertex_fields = selectedFieldsSelector(currentState);
  }
  return function* () {
    yield (0, _effects.takeEvery)((0, _helpers.matchesOne)(loadFields, selectField, deselectField, updateFieldProperties), syncFields);
  };
};

/**
 * Saga making sure the field styles (icons and colors) are applied to nodes currently active
 * in the workspace.
 *
 * Won't be necessary once the workspace is moved to redux
 */
exports.syncFieldsSaga = syncFieldsSaga;
const syncNodeStyleSaga = ({
  getWorkspace,
  notifyReact
}) => {
  function* syncNodeStyle(action) {
    const workspace = getWorkspace();
    if (!workspace) {
      return;
    }
    const newColor = action.payload.fieldProperties.color;
    if (newColor) {
      workspace.nodes.forEach(function (node) {
        if (node.data.field === action.payload.fieldName) {
          node.color = newColor;
        }
      });
    }
    const newIcon = action.payload.fieldProperties.icon;
    if (newIcon) {
      workspace.nodes.forEach(function (node) {
        if (node.data.field === action.payload.fieldName) {
          node.icon = newIcon;
        }
      });
    }
    notifyReact();
    const selectedFields = selectedFieldsSelector(yield (0, _effects.select)());
    workspace.options.vertex_fields = selectedFields;
  }
  return function* () {
    yield (0, _effects.takeLatest)(updateFieldProperties.match, syncNodeStyle);
  };
};
exports.syncNodeStyleSaga = syncNodeStyleSaga;