"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRouteRequiredAuthz = void 0;
var _std = require("@kbn/std");
var _common = require("../../../common");
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The authorization requirements needed for an API route. Route authorization requirements are
 * defined either via an `all` object, where all values must be `true` in order for access to be granted,
 * or, by an `any` object, where any value defined that is set to `true` will grant access to the API.
 *
 * The `all` conditions are checked first and if those evaluate to `false`, then `any` conditions are evaluated.
 */
const ROUTE_AUTHZ_REQUIREMENTS = (0, _std.deepFreeze)({
  // Package Policy Update API
  [`put:${_common.PACKAGE_POLICY_API_ROUTES.UPDATE_PATTERN}`]: {
    any: {
      integrations: {
        writeIntegrationPolicies: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            writePolicyManagement: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // Package Policy GET one API
  [`get:${_common.PACKAGE_POLICY_API_ROUTES.INFO_PATTERN}`]: {
    any: {
      integrations: {
        readIntegrationPolicies: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readPolicyManagement: {
              executePackageAction: true
            },
            readTrustedApplications: {
              executePackageAction: true
            },
            readEventFilters: {
              executePackageAction: true
            },
            readHostIsolationExceptions: {
              executePackageAction: true
            },
            readBlocklist: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // Package Policy Bulk GET API
  [`post:${_common.PACKAGE_POLICY_API_ROUTES.BULK_GET_PATTERN}`]: {
    any: {
      integrations: {
        readIntegrationPolicies: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readPolicyManagement: {
              executePackageAction: true
            },
            readTrustedApplications: {
              executePackageAction: true
            },
            readEventFilters: {
              executePackageAction: true
            },
            readHostIsolationExceptions: {
              executePackageAction: true
            },
            readBlocklist: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // Package Policy List API
  [`get:${_common.PACKAGE_POLICY_API_ROUTES.LIST_PATTERN}`]: {
    any: {
      integrations: {
        readIntegrationPolicies: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readPolicyManagement: {
              executePackageAction: true
            },
            readTrustedApplications: {
              executePackageAction: true
            },
            readEventFilters: {
              executePackageAction: true
            },
            readHostIsolationExceptions: {
              executePackageAction: true
            },
            readBlocklist: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // agent status fo policy API
  [`get:${_common.AGENT_API_ROUTES.STATUS_PATTERN}`]: {
    any: {
      fleet: {
        readAgents: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readPolicyManagement: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // EPM Package Info API
  [`get:${_constants.EPM_API_ROUTES.INFO_PATTERN}`]: {
    any: {
      integrations: {
        readPackageInfo: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readSecuritySolution: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  [`get:${_constants.EPM_API_ROUTES.INFO_WITHOUT_VERSION_PATTERN}`]: {
    any: {
      integrations: {
        readPackageInfo: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readSecuritySolution: {
              executePackageAction: true
            }
          }
        }
      }
    }
  },
  // EPM Package Knowledge Base API
  [`get:${_constants.EPM_API_ROUTES.KNOWLEDGE_BASE_PATTERN}`]: {
    any: {
      integrations: {
        readPackageInfo: true
      },
      packagePrivileges: {
        endpoint: {
          actions: {
            readSecuritySolution: {
              executePackageAction: true
            }
          }
        }
      }
    }
  }
});

/**
 * Retrieves the required fleet route authz
 * in order to grant access to the given api route
 * @param routeMethod
 * @param routePath
 */
const getRouteRequiredAuthz = (routeMethod, routePath) => {
  const key = `${routeMethod}:${routePath}`;
  if (typeof ROUTE_AUTHZ_REQUIREMENTS[key] !== 'undefined') {
    return ROUTE_AUTHZ_REQUIREMENTS[key];
  }
  for (const k of Object.keys(ROUTE_AUTHZ_REQUIREMENTS)) {
    if (pathMatchesPattern(k, key)) {
      return ROUTE_AUTHZ_REQUIREMENTS[k];
    }
  }
};
exports.getRouteRequiredAuthz = getRouteRequiredAuthz;
const pathMatchesPattern = (pathPattern, path) => {
  // No path params - pattern is single path
  if (pathPattern === path) {
    return true;
  }

  // If pathPattern has params (`{value}`), then see if `path` matches it
  if (/{.*?}/.test(pathPattern)) {
    const pathParts = path.split(/\//);
    const patternParts = pathPattern.split(/\//);
    if (pathParts.length !== patternParts.length) {
      return false;
    }
    return pathParts.every((part, index) => {
      return part === patternParts[index] || /{.*?}/.test(patternParts[index]);
    });
  }
  return false;
};