"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteActionSecrets = deleteActionSecrets;
exports.extractAndWriteActionSecrets = extractAndWriteActionSecrets;
exports.isActionSecretStorageEnabled = isActionSecretStorageEnabled;
var _app_context = require("../app_context");
var _ = require("..");
var _fleet_server = require("../fleet_server");
var _constants = require("../../constants");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Check if action secret storage is enabled.
 * Returns true if fleet server is standalone (serverless).
 * Otherwise, checks if the minimum fleet server version requirement has been met.
 * If the requirement has been met, updates the settings to enable action secret storage.
 */
async function isActionSecretStorageEnabled(esClient, soClient) {
  var _appContextService$ge, _appContextService$ge2, _appContextService$ge3;
  const logger = _app_context.appContextService.getLogger();

  // if serverless then action secrets will always be supported
  const isFleetServerStandalone = (_appContextService$ge = (_appContextService$ge2 = _app_context.appContextService.getConfig()) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.internal) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.fleetServerStandalone) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
  if (isFleetServerStandalone) {
    logger.trace('Action secrets storage is enabled as fleet server is standalone');
    return true;
  }

  // now check the flag in settings to see if the fleet server requirement has already been met
  // once the requirement has been met, action secrets are always on
  const settings = await _.settingsService.getSettingsOrUndefined(soClient);
  if (settings && settings.action_secret_storage_requirements_met) {
    logger.debug('Action secrets storage requirements already met, turned on in settings');
    return true;
  }

  // otherwise check if we have the minimum fleet server version and enable secrets if so
  if (await (0, _fleet_server.checkFleetServerVersionsForSecretsStorage)(esClient, soClient, _constants.ACTION_SECRETS_MINIMUM_FLEET_SERVER_VERSION)) {
    logger.debug('Enabling action secrets storage as minimum fleet server version has been met');
    try {
      await _.settingsService.saveSettings(soClient, {
        action_secret_storage_requirements_met: true
      });
    } catch (err) {
      // we can suppress this error as it will be retried on the next function call
      logger.warn(`Failed to save settings after enabling action secrets storage: ${err.message}`);
    }
    return true;
  }
  logger.info('Secrets storage is disabled as minimum fleet server version has not been met');
  return false;
}

/**
 * Given a new agent action, extracts the secrets, stores them in saved objects,
 * and returns the action with secrets replaced by references to the saved objects.
 */
async function extractAndWriteActionSecrets(opts) {
  const {
    action,
    esClient,
    secretHashes = {}
  } = opts;
  const secretPaths = getActionSecretPaths(action).filter(path => typeof path.value === 'string');
  const secretRes = await (0, _common.extractAndWriteSOSecrets)({
    soObject: action,
    secretPaths,
    esClient,
    secretHashes
  });
  return {
    actionWithSecrets: secretRes.soObjectWithSecrets,
    secretReferences: secretRes.secretReferences
  };
}

/**
 * Deletes secrets for a given agent action.
 * This function is currently not used, but implemented for completeness.
 */
async function deleteActionSecrets(opts) {
  const {
    action,
    esClient
  } = opts;
  await (0, _common.deleteSOSecrets)(esClient, getActionSecretPaths(action));
}

/**
 * Helper function to get the secret paths for an agent action.
 */
function getActionSecretPaths(action) {
  var _action$secrets, _action$secrets$user_, _action$secrets2;
  const secretPaths = [];
  if (action !== null && action !== void 0 && (_action$secrets = action.secrets) !== null && _action$secrets !== void 0 && (_action$secrets$user_ = _action$secrets.user_info) !== null && _action$secrets$user_ !== void 0 && _action$secrets$user_.password) {
    secretPaths.push({
      path: 'secrets.user_info.password',
      value: action.secrets.user_info.password
    });
  }
  if (action !== null && action !== void 0 && (_action$secrets2 = action.secrets) !== null && _action$secrets2 !== void 0 && _action$secrets2.enrollment_token) {
    secretPaths.push({
      path: 'secrets.enrollment_token',
      value: action.secrets.enrollment_token
    });
  }
  return secretPaths;
}