"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAlertingRuleFromTemplate = createAlertingRuleFromTemplate;
exports.stepCreateAlertingRules = stepCreateAlertingRules;
var _spacesUtils = require("@kbn/spaces-utils");
var _server = require("@kbn/core/server");
var _common = require("../../../../../../common");
var _types = require("../../../../../../common/types");
var _request_utils = require("../../../../request_utils");
var _app_context = require("../../../../app_context");
var _utils = require("../../utils");
var _install = require("../../install");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getRuleId({
  pkgName,
  templateId,
  spaceId
}) {
  return `fleet-${spaceId ? spaceId : _spacesUtils.DEFAULT_SPACE_ID}-${pkgName}-${templateId}`;
}
async function createAlertingRuleFromTemplate(deps, params) {
  const {
    rulesClient,
    logger
  } = deps;
  const {
    pkgName,
    alertTemplateArchiveAsset,
    spaceId
  } = params;
  const ruleId = getRuleId({
    pkgName,
    templateId: alertTemplateArchiveAsset.id,
    spaceId
  });
  try {
    if (!rulesClient) {
      throw new _common.FleetError('Rules client is not available');
    }
    const template = await rulesClient.getTemplate({
      id: alertTemplateArchiveAsset.id
    }).catch(err => {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        return undefined;
      }
      throw err;
    });
    if (!template) {
      throw new _common.FleetError(`Rule template ${alertTemplateArchiveAsset.id} not found`);
    }
    const rule = await rulesClient.get({
      id: ruleId
    }).catch(err => {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        return undefined;
      }
      throw err;
    });
    // Already created
    if (rule) {
      return {
        id: ruleId,
        type: _types.KibanaSavedObjectType.alert,
        deferred: false
      };
    }
    const {
      ruleTypeId,
      id: _id,
      ...rest
    } = template;
    logger.debug(`Creating rule: ${ruleId} for package ${pkgName}`);
    await rulesClient.create({
      data: {
        alertTypeId: ruleTypeId,
        ...rest,
        enabled: true,
        actions: [],
        consumer: 'alerts'
      },
      // what value for consumer will make sense?
      options: {
        id: ruleId
      }
    });
    return {
      id: ruleId,
      type: _types.KibanaSavedObjectType.alert,
      deferred: false
    };
  } catch (e) {
    logger.error(`Error creating rule: ${ruleId} for package ${pkgName}`, {
      error: e
    });
    return {
      id: ruleId,
      type: _types.KibanaSavedObjectType.alert,
      deferred: true
    };
  }
}
async function stepCreateAlertingRules(context) {
  const {
    logger,
    savedObjectsClient,
    packageInstallContext,
    spaceId
  } = context;
  const {
    packageInfo
  } = packageInstallContext;
  const {
    name: pkgName
  } = packageInfo;
  if (pkgName !== _common.FLEET_ELASTIC_AGENT_PACKAGE) {
    return;
  }
  await (0, _utils.withPackageSpan)('Install elastic agent rules', async () => {
    var _appContextService$ge;
    const rulesClient = context.authorizationHeader ? await ((_appContextService$ge = _app_context.appContextService.getAlertingStart()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.getRulesClientWithRequest((0, _request_utils.createKibanaRequestFromAuth)(context.authorizationHeader))) : undefined;
    const assetRefs = [];
    await packageInstallContext.archiveIterator.traverseEntries(async entry => {
      if (!entry.buffer) {
        return;
      }
      const alertTemplate = JSON.parse(entry.buffer.toString('utf8'));
      const ref = await createAlertingRuleFromTemplate({
        rulesClient,
        logger
      }, {
        alertTemplateArchiveAsset: alertTemplate,
        spaceId,
        pkgName
      });
      assetRefs.push(ref);
    }, path => path.match(/\/alerting_rule_template\//) !== null);
    await (0, _install.saveKibanaAssetsRefs)(savedObjectsClient, pkgName, assetRefs, false, true);
  });
}