"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Visitor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _global_visitor_context = require("./global_visitor_context");
var _contexts = require("./contexts");
var _builder = require("../builder");
var _Visitor;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
class Visitor {
  constructor(options = {}) {
    var _options$visitors, _options$data;
    (0, _defineProperty2.default)(this, "ctx", void 0);
    this.options = options;
    this.ctx = new _global_visitor_context.GlobalVisitorContext((_options$visitors = options.visitors) !== null && _options$visitors !== void 0 ? _options$visitors : {}, (_options$data = options.data) !== null && _options$data !== void 0 ? _options$data : {});
  }
  visitors(visitors) {
    Object.assign(this.ctx.methods, visitors);
    return this;
  }
  on(visitor, fn) {
    this.ctx.methods[visitor] = fn;
    return this;
  }

  /**
   * Traverse any AST node given any visitor context.
   *
   * @param node AST node to traverse.
   * @param ctx Traversal context.
   * @returns Result of the visitor callback.
   */
  visit(ctx, input) {
    const node = ctx.node;
    if (node instanceof Array) {
      throw new Error(`Unsupported node type: ${typeof node}`);
    } else if (node && typeof node === 'object') {
      switch (node.type) {
        case 'query':
          {
            this.ctx.assertMethodExists('visitQuery');
            return this.ctx.methods.visitQuery(ctx, input);
          }
        case 'header-command':
          {
            this.ctx.assertMethodExists('visitHeaderCommand');
            return this.ctx.methods.visitHeaderCommand(ctx, input);
          }
        case 'command':
          {
            this.ctx.assertMethodExists('visitCommand');
            return this.ctx.methods.visitCommand(ctx, input);
          }
      }
    }
    throw new Error(`Unsupported node type: ${typeof node}`);
  }

  /**
   * Traverse the root node of ES|QL query with default context.
   *
   * @param node Query node to traverse.
   * @param input Input to pass to the first visitor.
   * @returns The result of the query visitor.
   */
  visitQuery(nodeOrCommands, input) {
    const node = Array.isArray(nodeOrCommands) ? _builder.Builder.expression.query(nodeOrCommands) : nodeOrCommands;
    const queryContext = new _contexts.QueryVisitorContext(this.ctx, node, null);
    return this.visit(queryContext, input);
  }

  /**
   * Traverse starting from known command node with default context.
   *
   * @param node Command node to traverse.
   * @param input Input to pass to the first visitor.
   * @returns The output of the visitor.
   */
  visitCommand(node, input) {
    this.ctx.assertMethodExists('visitCommand');
    return this.ctx.visitCommand(null, node, input);
  }

  /**
   * Traverse starting from known header command node with default context.
   *
   * @param node Header command node to traverse.
   * @param input Input to pass to the first visitor.
   * @returns The output of the visitor.
   */
  visitHeaderCommand(node, input) {
    this.ctx.assertMethodExists('visitHeaderCommand');
    return this.ctx.visitHeaderCommand(null, node, input);
  }

  /**
   * Traverse starting from known expression node with default context.
   *
   * @param node Expression node to traverse.
   * @param input Input to pass to the first visitor.
   * @returns The output of the visitor.
   */
  visitExpression(node, input) {
    this.ctx.assertMethodExists('visitExpression');
    return this.ctx.visitExpression(null, node, input);
  }
}
exports.Visitor = Visitor;
_Visitor = Visitor;
/**
 * Finds the most specific node immediately after the given position. If the
 * position is inside a node, it will return the node itself. If no node is
 * found, it returns `null`.
 *
 * @param ast ES|QL AST
 * @param pos Offset position in the source text
 * @returns The node at or after the given position
 */
(0, _defineProperty2.default)(Visitor, "findNodeAtOrAfter", (ast, pos) => {
  const visitCommand = ctx => {
    for (const child of ctx.arguments()) {
      const {
        location: childLocation
      } = child;
      if (!childLocation) continue;
      const isInsideChild = childLocation.min <= pos && childLocation.max >= pos;
      if (isInsideChild) return ctx.visitExpression(child, undefined);
      const isBeforeChild = childLocation.min > pos;
      if (isBeforeChild) {
        return ctx.visitExpression(child, undefined) || child;
      }
    }
    return null;
  };
  return new _Visitor().on('visitExpression', ctx => {
    const node = ctx.node;
    const location = node.location;
    if (!location) return null;
    const isBefore = location.min > pos;
    let isFirstChild = true;
    for (const child of ctx.arguments()) {
      const {
        location: childLocation
      } = child;
      if (!childLocation) continue;
      if (isFirstChild) {
        isFirstChild = false;
        if (isBefore) {
          const isChildAtOffset = ctx.node.location && ctx.node.location.min < childLocation.min;
          if (isChildAtOffset) return node;
          return ctx.visitExpression(child, undefined) || child;
        }
      }
      const isInsideChild = childLocation.min <= pos && childLocation.max >= pos;
      if (isInsideChild) return ctx.visitExpression(child, undefined);
      const isBeforeChild = childLocation.min > pos;
      if (isBeforeChild) {
        return ctx.visitExpression(child, undefined) || child;
      }
    }
    return null;
  }).on('visitCommand', visitCommand).on('visitHeaderCommand', visitCommand).on('visitQuery', ctx => {
    for (const node of ctx.headerCommands()) {
      const {
        location
      } = node;
      if (!location) continue;
      const isInside = location.min <= pos && location.max >= pos;
      if (isInside) return ctx.visitHeaderCommand(node, undefined);
      const isBefore = location.min > pos;
      if (isBefore) return node;
    }
    for (const node of ctx.commands()) {
      const {
        location
      } = node;
      if (!location) continue;
      const isInside = location.min <= pos && location.max >= pos;
      if (isInside) return ctx.visitCommand(node);
      const isBefore = location.min > pos;
      if (isBefore) return node;
    }
    return null;
  }).visitQuery(ast);
});
/**
 * Finds the most specific node immediately before the given position. If the
 * position is inside a node, it will return the node itself. If no node is
 * found, it returns `null`.
 *
 * @param ast ES|QL AST
 * @param pos Offset position in the source text
 * @returns The node at or before the given position
 */
(0, _defineProperty2.default)(Visitor, "findNodeAtOrBefore", (ast, pos) => {
  const visitCommand = ctx => {
    const nodes = [...ctx.arguments()];
    for (let i = nodes.length - 1; i >= 0; i--) {
      const node = nodes[i];
      if (!node) continue;
      const {
        location
      } = node;
      if (!location) continue;
      const isInside = location.min <= pos && location.max >= pos;
      if (isInside) return ctx.visitExpression(node, undefined);
      const isAfter = location.max < pos;
      if (isAfter) {
        if (ctx.node.location && ctx.node.location.max >= location.max) {
          return ctx.visitExpression(node, undefined) || node;
        }
        return node;
      }
    }
    return null;
  };
  return new _Visitor().on('visitExpression', ctx => {
    const nodeLocation = ctx.node.location;
    const nodes = [...ctx.arguments()];
    if (nodeLocation && nodeLocation.max < pos) {
      const last = nodes[nodes.length - 1];
      if (last && last.location && last.location.max === nodeLocation.max) {
        return ctx.visitExpression(last, undefined) || last;
      } else {
        return ctx.node;
      }
    }
    for (let i = nodes.length - 1; i >= 0; i--) {
      const node = nodes[i];
      if (!node) continue;
      const {
        location
      } = node;
      if (!location) continue;
      const isInside = location.min <= pos && location.max >= pos;
      if (isInside) return ctx.visitExpression(node, undefined);
      const isAfter = location.max < pos;
      if (isAfter) {
        return ctx.visitExpression(node, undefined) || node;
      }
    }
    return null;
  }).on('visitCommand', visitCommand).on('visitHeaderCommand', visitCommand).on('visitQuery', ctx => {
    const nodes = [...ctx.headerCommands(), ...ctx.commands()];
    for (let i = nodes.length - 1; i >= 0; i--) {
      const node = nodes[i];
      if (!node) continue;
      const {
        location
      } = node;
      if (!location) continue;
      const isInside = location.min <= pos && location.max >= pos;
      if (isInside) {
        if (node.type === 'command') return ctx.visitCommand(node);else if (node.type === 'header-command') {
          return ctx.visitHeaderCommand(node, undefined);
        } else return node;
      }
      const isAfter = location.max < pos;
      if (isAfter) {
        if (node.type === 'command') return ctx.visitCommand(node) || node;
        return node;
      }
    }
    return null;
  }).visitQuery(ast);
});