"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.logPluginsStatusChanges = void 0;
var _rxjs = require("rxjs");
var _coreStatusCommon = require("@kbn/core-status-common");
var _log_throttled_buffer = require("./log_throttled_buffer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// let plugins log up to 3 status changes every 30s (extra messages will be throttled / aggregated)
const MAX_MESSAGES_PER_PLUGIN_PER_INTERVAL = 3;
const THROTTLE_INTERVAL_MILLIS = 30000;
const MAX_THROTTLED_MESSAGES = 10;
const logPluginsStatusChanges = ({
  logger,
  plugins$,
  stop$,
  maxMessagesPerPluginPerInterval = MAX_MESSAGES_PER_PLUGIN_PER_INTERVAL,
  throttleIntervalMillis = THROTTLE_INTERVAL_MILLIS,
  maxThrottledMessages = MAX_THROTTLED_MESSAGES
}) => {
  const buffer$ = new _rxjs.Subject();
  const throttled$ = (0, _log_throttled_buffer.createLogThrottledBuffer)({
    buffer$,
    stop$,
    maxThrottledMessages
  });
  const lastMessagesTimestamps = {};
  const direct$ = plugins$.pipe((0, _rxjs.startWith)({}),
  // consider all plugins unavailable by default
  (0, _rxjs.takeUntil)(stop$), (0, _rxjs.pairwise)(), (0, _rxjs.map)(([oldStatus, newStatus]) => getPluginUpdates(oldStatus, newStatus)), (0, _rxjs.concatAll)(), (0, _rxjs.filter)(pluginStatus => {
    const now = Date.now();
    const pluginQuota = lastMessagesTimestamps[pluginStatus.name] || [];
    lastMessagesTimestamps[pluginStatus.name] = pluginQuota;

    // remove timestamps of messages older than the threshold
    while (pluginQuota.length > 0 && pluginQuota[0] < now - throttleIntervalMillis) {
      pluginQuota.shift();
    }
    if (pluginQuota.length >= maxMessagesPerPluginPerInterval) {
      // we're still over quota, throttle the message
      buffer$.next(pluginStatus);
      return false;
    } else {
      // let the message pass through
      pluginQuota.push(now);
      return true;
    }
  }));
  return (0, _rxjs.merge)(direct$, throttled$).subscribe(event => {
    if (typeof event === 'string') {
      logger.warn(event);
    } else {
      const pluginStatus = event;
      const {
        name
      } = pluginStatus;
      const pluginLogger = logger.get(name);
      const message = getPluginStatusMessage(pluginStatus);
      switch (pluginStatus.level) {
        case _coreStatusCommon.ServiceStatusLevels.available:
          pluginLogger.info(message);
          break;
        case _coreStatusCommon.ServiceStatusLevels.degraded:
          pluginLogger.warn(message);
          break;
        default:
          pluginLogger.error(message);
      }
    }
  });
};
exports.logPluginsStatusChanges = logPluginsStatusChanges;
const getPluginUpdates = (previous, next) => Object.entries(next).filter(([name, pluginStatus]) => {
  var _previous$name;
  const currentLevel = pluginStatus.level;
  const previousLevel = (_previous$name = previous[name]) === null || _previous$name === void 0 ? void 0 : _previous$name.level;
  return pluginStatus.reported && currentLevel !== previousLevel;
}).map(([name, pluginStatus]) => ({
  ...pluginStatus,
  name
}));
const getPluginStatusMessage = ({
  name,
  level,
  summary,
  detail,
  repeats = 0
}) => `${name} plugin is now ${level === null || level === void 0 ? void 0 : level.toString()}: ${summary}${detail ? ` | ${detail}` : ''}${repeats > 1 ? ` (repeated ${repeats} times)` : ''}`;