"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.INDEX_PLACEHOLDER = void 0;
exports.findConfiguration = findConfiguration;
exports.isIndexPlaceholder = void 0;
exports.replaceIndexPatterns = replaceIndexPatterns;
var _lodash = require("lodash");
var _esqlUtils = require("@kbn/esql-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INDEX_PLACEHOLDER = exports.INDEX_PLACEHOLDER = '<stream_name_placeholder>';
const isIndexPlaceholder = index => index.startsWith(INDEX_PLACEHOLDER);
exports.isIndexPlaceholder = isIndexPlaceholder;
function findConfiguration(savedObject) {
  const patterns = new Set();
  const fields = {};
  locateConfiguration(savedObject, {
    esqlQuery(query) {
      (0, _esqlUtils.getIndexPatternFromESQLQuery)(query).split(',').forEach(p => patterns.add(p));
      return query;
    },
    indexPattern(pattern) {
      if (pattern.title) {
        pattern.title.split(',').forEach(p => patterns.add(p));
      }
      return pattern;
    },
    field(field) {
      if ('fieldName' in field) {
        const {
          fieldName,
          meta
        } = field;
        if (meta !== null && meta !== void 0 && meta.esType) {
          fields[fieldName] = {
            type: meta.esType
          };
        }
      } else if ('sourceField' in field) {
        const {
          sourceField,
          dataType
        } = field;
        if (sourceField !== '___records___') {
          fields[sourceField] = {
            type: dataType
          };
        }
      }
      return field;
    }
  });
  return {
    patterns: [...patterns],
    fields
  };
}
function replaceIndexPatterns(savedObject, patternReplacements) {
  return locateConfiguration((0, _lodash.cloneDeep)(savedObject), {
    esqlQuery(query) {
      return (0, _esqlUtils.replaceESQLQueryIndexPattern)(query, patternReplacements);
    },
    indexPattern(pattern) {
      var _pattern$title, _pattern$name;
      const updatedPattern = (_pattern$title = pattern.title) === null || _pattern$title === void 0 ? void 0 : _pattern$title.split(',').map(index => {
        var _patternReplacements$;
        return (_patternReplacements$ = patternReplacements[index]) !== null && _patternReplacements$ !== void 0 ? _patternReplacements$ : index;
      }).join(',');

      // data view references may be named after the index patterns they represent,
      // so we attempt to replace index patterns to avoid wrongly named data views
      const updatedName = (_pattern$name = pattern.name) === null || _pattern$name === void 0 ? void 0 : _pattern$name.split(',').map(index => {
        var _patternReplacements$2;
        return (_patternReplacements$2 = patternReplacements[index]) !== null && _patternReplacements$2 !== void 0 ? _patternReplacements$2 : index;
      }).join(',');
      return {
        ...pattern,
        title: updatedPattern,
        name: updatedName
      };
    },
    field(field) {
      return field;
    }
  });
}
function locateConfiguration(content, options) {
  if (content.type === 'index-pattern') {
    content.attributes = options.indexPattern(content.attributes);
  }
  if (content.type === 'dashboard') {
    const attributes = content.attributes;
    const panels = JSON.parse(attributes.panelsJSON).map(panel => traversePanel(panel, options));
    attributes.panelsJSON = JSON.stringify(panels);
  }
  if (content.type === 'lens') {
    content.attributes = traverseLensPanel(content.attributes, options);
  }
  return content;
}
function traversePanel(panel, options) {
  if (panel.type === 'lens') {
    const config = panel.embeddableConfig;
    if (config.query && 'esql' in config.query) {
      config.query.esql = options.esqlQuery(config.query.esql);
    }
    if (config.attributes) {
      traverseLensPanel(config.attributes, options);
    }
  }
  return panel;
}
function traverseLensPanel(panel, options) {
  const state = panel.state;
  if (state.adHocDataViews) {
    state.adHocDataViews = (0, _lodash.mapValues)(state.adHocDataViews, dataView => options.indexPattern(dataView));
  }
  const {
    query: stateQuery,
    datasourceStates: {
      formBased,
      textBased
    }
  } = state;
  if (stateQuery && 'esql' in stateQuery) {
    stateQuery.esql = options.esqlQuery(stateQuery.esql);
  }
  if (formBased) {
    Object.values(formBased.layers).forEach(layer => {
      Object.entries(layer.columns).forEach(([columnId, column]) => {
        layer.columns[columnId] = options.field(column);
      });
    });
  }
  if (textBased) {
    Object.values(textBased.layers).forEach(layer => {
      var _layer$query;
      if ((_layer$query = layer.query) !== null && _layer$query !== void 0 && _layer$query.esql) {
        layer.query.esql = options.esqlQuery(layer.query.esql);
      }
      layer.columns = layer.columns.map(column => options.field(column));
    });
    if ('indexPatternRefs' in textBased) {
      textBased.indexPatternRefs = textBased.indexPatternRefs.map(ref => options.indexPattern(ref));
    }
  }
  return panel;
}