"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAccountsStats = void 0;
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _get_identifier_runtime_mapping = require("../../../../common/runtime_mappings/get_identifier_runtime_mapping");
var _helpers = require("../../../../common/utils/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAccountsStatsQuery = () => ({
  index: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS,
  runtime_mappings: (0, _get_identifier_runtime_mapping.getIdentifierRuntimeMapping)(),
  query: {
    match_all: {}
  },
  aggs: {
    accounts: {
      terms: {
        field: 'asset_identifier',
        order: {
          _count: 'desc'
        },
        size: 100
      },
      aggs: {
        nodes_count: {
          cardinality: {
            field: 'host.name'
          }
        },
        agents_count: {
          cardinality: {
            field: 'agent.id'
          }
        },
        benchmark_id: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.id'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        benchmark_version: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.version'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        benchmark_name: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.name'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        kubernetes_version: {
          top_metrics: {
            metrics: {
              field: 'cloudbeat.kubernetes.version'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        passed_findings_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'result.evaluation': 'passed'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        failed_findings_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'result.evaluation': 'failed'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        resources: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'resource.sub_type': 'Pod'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          },
          aggs: {
            pods_count: {
              cardinality: {
                field: 'resource.id'
              }
            }
          }
        },
        kspm_namespaces_count: {
          filter: {
            term: {
              'rule.benchmark.posture_type': 'kspm'
            }
          },
          aggs: {
            namespaces_count: {
              cardinality: {
                field: 'data_stream.namespace'
              }
            }
          }
        },
        cspm_namespaces_count: {
          filter: {
            term: {
              'rule.benchmark.posture_type': 'cspm'
            }
          },
          aggs: {
            namespaces_count: {
              cardinality: {
                field: 'data_stream.namespace'
              }
            }
          }
        }
      }
    }
  },
  size: 0,
  _source: false
});
const getCspmAccountsStats = (aggregatedResourcesStats, logger) => {
  const accounts = aggregatedResourcesStats.accounts.buckets;
  const cspmAccountsStats = accounts.map(account => ({
    account_id: account.key,
    latest_findings_doc_count: account.doc_count,
    posture_score: (0, _helpers.calculatePostureScore)(account.passed_findings_count.doc_count, account.failed_findings_count.doc_count),
    passed_findings_count: account.passed_findings_count.doc_count,
    failed_findings_count: account.failed_findings_count.doc_count,
    benchmark_name: account.benchmark_name.top[0].metrics['rule.benchmark.name'],
    benchmark_id: account.benchmark_id.top[0].metrics['rule.benchmark.id'],
    benchmark_version: account.benchmark_version.top[0].metrics['rule.benchmark.version'],
    kubernetes_version: account.kubernetes_version.top[0].metrics['cloudbeat.kubernetes.version'],
    agents_count: account.agents_count.value,
    nodes_count: account.nodes_count.value,
    pods_count: account.resources.pods_count.value,
    kspm_namespaces_count: account.kspm_namespaces_count.namespaces_count.value,
    cspm_namespaces_count: account.cspm_namespaces_count.namespaces_count.value
  }));
  return cspmAccountsStats;
};
const getAccountsStats = async (esClient, logger) => {
  try {
    const isIndexExists = await esClient.indices.exists({
      index: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS
    });
    if (isIndexExists) {
      const accountsStatsResponse = await esClient.search(getAccountsStatsQuery());
      const cspmAccountsStats = accountsStatsResponse.aggregations ? getCspmAccountsStats(accountsStatsResponse.aggregations, logger) : [];
      return cspmAccountsStats;
    }
    return [];
  } catch (e) {
    logger.error(`Failed to get account stats ${e}`);
    return [];
  }
};
exports.getAccountsStats = getAccountsStats;