"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RulesSettingsService = exports.DEFAULT_CACHE_INTERVAL_MS = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _types = require("../types");
var _lib = require("../task_runner/lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_CACHE_INTERVAL_MS = exports.DEFAULT_CACHE_INTERVAL_MS = 60000; // 1 minute cache

class RulesSettingsService {
  constructor(options) {
    (0, _defineProperty2.default)(this, "cacheIntervalMs", DEFAULT_CACHE_INTERVAL_MS);
    (0, _defineProperty2.default)(this, "defaultQueryDelaySettings", void 0);
    (0, _defineProperty2.default)(this, "settings", new Map());
    this.options = options;
    if (options.cacheInterval) {
      this.cacheIntervalMs = options.cacheInterval;
    }
    this.defaultQueryDelaySettings = options.isServerless ? _types.DEFAULT_SERVERLESS_QUERY_DELAY_SETTINGS : _types.DEFAULT_QUERY_DELAY_SETTINGS;
  }
  async getSettings(request, spaceId) {
    const now = Date.now();
    if (this.settings.has(spaceId)) {
      const settingsFromLastUpdate = this.settings.get(spaceId);
      const lastUpdated = new Date(settingsFromLastUpdate.lastUpdated).getTime();
      const currentFlappingSettings = settingsFromLastUpdate.flappingSettings;
      const currentQueryDelaySettings = settingsFromLastUpdate.queryDelaySettings;
      if (now - lastUpdated >= this.cacheIntervalMs) {
        // cache expired, refetch settings
        try {
          return await this.fetchSettings(request, spaceId, now);
        } catch (err) {
          // return cached settings on error
          this.options.logger.debug(`Failed to fetch rules settings after cache expiration, using cached settings: ${err.message}`);
          return {
            queryDelaySettings: currentQueryDelaySettings,
            flappingSettings: currentFlappingSettings
          };
        }
      } else {
        return {
          queryDelaySettings: currentQueryDelaySettings,
          flappingSettings: currentFlappingSettings
        };
      }
    } else {
      // nothing in cache, fetch settings
      try {
        return await this.fetchSettings(request, spaceId, now);
      } catch (err) {
        // return default settings on error
        this.options.logger.debug(`Failed to fetch initial rules settings, using default settings: ${err.message}`);
        return {
          queryDelaySettings: this.defaultQueryDelaySettings,
          flappingSettings: _types.DEFAULT_FLAPPING_SETTINGS
        };
      }
    }
  }
  async fetchSettings(request, spaceId, now) {
    const rulesSettingsClient = this.options.getRulesSettingsClientWithRequest(request);
    const queryDelaySettings = await (0, _lib.withAlertingSpan)('alerting:get-query-delay-settings', () => rulesSettingsClient.queryDelay().get());
    const flappingSettings = await (0, _lib.withAlertingSpan)('alerting:get-flapping-settings', () => rulesSettingsClient.flapping().get());
    this.settings.set(spaceId, {
      lastUpdated: now,
      queryDelaySettings,
      flappingSettings
    });
    return {
      flappingSettings,
      queryDelaySettings
    };
  }
}
exports.RulesSettingsService = RulesSettingsService;