"use strict";
/*
 * Copyright Elasticsearch B.V. and contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const commonQueryParams = ['error_trace', 'filter_path', 'human', 'pretty'];
class Security {
    constructor(transport) {
        Object.defineProperty(this, "transport", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "acceptedParams", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        this.transport = transport;
        this.acceptedParams = {
            'security.activate_user_profile': {
                path: [],
                body: [
                    'access_token',
                    'grant_type',
                    'password',
                    'username'
                ],
                query: []
            },
            'security.authenticate': {
                path: [],
                body: [],
                query: []
            },
            'security.bulk_delete_role': {
                path: [],
                body: [
                    'names'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.bulk_put_role': {
                path: [],
                body: [
                    'roles'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.bulk_update_api_keys': {
                path: [],
                body: [
                    'expiration',
                    'ids',
                    'metadata',
                    'role_descriptors'
                ],
                query: []
            },
            'security.change_password': {
                path: [
                    'username'
                ],
                body: [
                    'password',
                    'password_hash'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.clear_api_key_cache': {
                path: [
                    'ids'
                ],
                body: [],
                query: []
            },
            'security.clear_cached_privileges': {
                path: [
                    'application'
                ],
                body: [],
                query: []
            },
            'security.clear_cached_realms': {
                path: [
                    'realms'
                ],
                body: [],
                query: [
                    'usernames'
                ]
            },
            'security.clear_cached_roles': {
                path: [
                    'name'
                ],
                body: [],
                query: []
            },
            'security.clear_cached_service_tokens': {
                path: [
                    'namespace',
                    'service',
                    'name'
                ],
                body: [],
                query: []
            },
            'security.create_api_key': {
                path: [],
                body: [
                    'expiration',
                    'name',
                    'role_descriptors',
                    'metadata'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.create_cross_cluster_api_key': {
                path: [],
                body: [
                    'access',
                    'expiration',
                    'metadata',
                    'name'
                ],
                query: []
            },
            'security.create_service_token': {
                path: [
                    'namespace',
                    'service',
                    'name'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.delegate_pki': {
                path: [],
                body: [
                    'x509_certificate_chain'
                ],
                query: []
            },
            'security.delete_privileges': {
                path: [
                    'application',
                    'name'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.delete_role': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.delete_role_mapping': {
                path: [
                    'name'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.delete_service_token': {
                path: [
                    'namespace',
                    'service',
                    'name'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.delete_user': {
                path: [
                    'username'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.disable_user': {
                path: [
                    'username'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.disable_user_profile': {
                path: [
                    'uid'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.enable_user': {
                path: [
                    'username'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.enable_user_profile': {
                path: [
                    'uid'
                ],
                body: [],
                query: [
                    'refresh'
                ]
            },
            'security.enroll_kibana': {
                path: [],
                body: [],
                query: []
            },
            'security.enroll_node': {
                path: [],
                body: [],
                query: []
            },
            'security.get_api_key': {
                path: [],
                body: [],
                query: [
                    'id',
                    'name',
                    'owner',
                    'realm_name',
                    'username',
                    'with_limited_by',
                    'active_only',
                    'with_profile_uid'
                ]
            },
            'security.get_builtin_privileges': {
                path: [],
                body: [],
                query: []
            },
            'security.get_privileges': {
                path: [
                    'application',
                    'name'
                ],
                body: [],
                query: []
            },
            'security.get_role': {
                path: [
                    'name'
                ],
                body: [],
                query: []
            },
            'security.get_role_mapping': {
                path: [
                    'name'
                ],
                body: [],
                query: []
            },
            'security.get_service_accounts': {
                path: [
                    'namespace',
                    'service'
                ],
                body: [],
                query: []
            },
            'security.get_service_credentials': {
                path: [
                    'namespace',
                    'service'
                ],
                body: [],
                query: []
            },
            'security.get_settings': {
                path: [],
                body: [],
                query: [
                    'master_timeout'
                ]
            },
            'security.get_token': {
                path: [],
                body: [
                    'grant_type',
                    'scope',
                    'password',
                    'kerberos_ticket',
                    'refresh_token',
                    'username'
                ],
                query: []
            },
            'security.get_user': {
                path: [
                    'username'
                ],
                body: [],
                query: [
                    'with_profile_uid'
                ]
            },
            'security.get_user_privileges': {
                path: [],
                body: [],
                query: []
            },
            'security.get_user_profile': {
                path: [
                    'uid'
                ],
                body: [],
                query: [
                    'data'
                ]
            },
            'security.grant_api_key': {
                path: [],
                body: [
                    'api_key',
                    'grant_type',
                    'access_token',
                    'username',
                    'password',
                    'run_as'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.has_privileges': {
                path: [
                    'user'
                ],
                body: [
                    'application',
                    'cluster',
                    'index'
                ],
                query: []
            },
            'security.has_privileges_user_profile': {
                path: [],
                body: [
                    'uids',
                    'privileges'
                ],
                query: []
            },
            'security.invalidate_api_key': {
                path: [],
                body: [
                    'id',
                    'ids',
                    'name',
                    'owner',
                    'realm_name',
                    'username'
                ],
                query: []
            },
            'security.invalidate_token': {
                path: [],
                body: [
                    'token',
                    'refresh_token',
                    'realm_name',
                    'username'
                ],
                query: []
            },
            'security.oidc_authenticate': {
                path: [],
                body: [
                    'nonce',
                    'realm',
                    'redirect_uri',
                    'state'
                ],
                query: []
            },
            'security.oidc_logout': {
                path: [],
                body: [
                    'token',
                    'refresh_token'
                ],
                query: []
            },
            'security.oidc_prepare_authentication': {
                path: [],
                body: [
                    'iss',
                    'login_hint',
                    'nonce',
                    'realm',
                    'state'
                ],
                query: []
            },
            'security.put_privileges': {
                path: [],
                body: [
                    'privileges'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.put_role': {
                path: [
                    'name'
                ],
                body: [
                    'applications',
                    'cluster',
                    'global',
                    'indices',
                    'remote_indices',
                    'remote_cluster',
                    'metadata',
                    'run_as',
                    'description',
                    'transient_metadata'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.put_role_mapping': {
                path: [
                    'name'
                ],
                body: [
                    'enabled',
                    'metadata',
                    'roles',
                    'role_templates',
                    'rules',
                    'run_as'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.put_user': {
                path: [],
                body: [
                    'username',
                    'email',
                    'full_name',
                    'metadata',
                    'password',
                    'password_hash',
                    'roles',
                    'enabled'
                ],
                query: [
                    'refresh'
                ]
            },
            'security.query_api_keys': {
                path: [],
                body: [
                    'aggregations',
                    'aggs',
                    'query',
                    'from',
                    'sort',
                    'size',
                    'search_after'
                ],
                query: [
                    'with_limited_by',
                    'with_profile_uid',
                    'typed_keys'
                ]
            },
            'security.query_role': {
                path: [],
                body: [
                    'query',
                    'from',
                    'sort',
                    'size',
                    'search_after'
                ],
                query: []
            },
            'security.query_user': {
                path: [],
                body: [
                    'query',
                    'from',
                    'sort',
                    'size',
                    'search_after'
                ],
                query: [
                    'with_profile_uid'
                ]
            },
            'security.saml_authenticate': {
                path: [],
                body: [
                    'content',
                    'ids',
                    'realm'
                ],
                query: []
            },
            'security.saml_complete_logout': {
                path: [],
                body: [
                    'realm',
                    'ids',
                    'query_string',
                    'content'
                ],
                query: []
            },
            'security.saml_invalidate': {
                path: [],
                body: [
                    'acs',
                    'query_string',
                    'realm'
                ],
                query: []
            },
            'security.saml_logout': {
                path: [],
                body: [
                    'token',
                    'refresh_token'
                ],
                query: []
            },
            'security.saml_prepare_authentication': {
                path: [],
                body: [
                    'acs',
                    'realm',
                    'relay_state'
                ],
                query: []
            },
            'security.saml_service_provider_metadata': {
                path: [
                    'realm_name'
                ],
                body: [],
                query: []
            },
            'security.suggest_user_profiles': {
                path: [],
                body: [
                    'name',
                    'size',
                    'data',
                    'hint'
                ],
                query: [
                    'data'
                ]
            },
            'security.update_api_key': {
                path: [
                    'id'
                ],
                body: [
                    'role_descriptors',
                    'metadata',
                    'expiration'
                ],
                query: []
            },
            'security.update_cross_cluster_api_key': {
                path: [
                    'id'
                ],
                body: [
                    'access',
                    'expiration',
                    'metadata'
                ],
                query: []
            },
            'security.update_settings': {
                path: [],
                body: [
                    'security',
                    'security-profile',
                    'security-tokens'
                ],
                query: [
                    'master_timeout',
                    'timeout'
                ]
            },
            'security.update_user_profile_data': {
                path: [
                    'uid'
                ],
                body: [
                    'labels',
                    'data'
                ],
                query: [
                    'if_seq_no',
                    'if_primary_term',
                    'refresh'
                ]
            }
        };
    }
    async activateUserProfile(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.activate_user_profile'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/profile/_activate';
        const meta = {
            name: 'security.activate_user_profile'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async authenticate(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.authenticate'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/_authenticate';
        const meta = {
            name: 'security.authenticate'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async bulkDeleteRole(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.bulk_delete_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'DELETE';
        const path = '/_security/role';
        const meta = {
            name: 'security.bulk_delete_role'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async bulkPutRole(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.bulk_put_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/role';
        const meta = {
            name: 'security.bulk_put_role'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async bulkUpdateApiKeys(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.bulk_update_api_keys'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/api_key/_bulk_update';
        const meta = {
            name: 'security.bulk_update_api_keys'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async changePassword(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.change_password'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        let method = '';
        let path = '';
        if (params.username != null) {
            method = 'PUT';
            path = `/_security/user/${encodeURIComponent(params.username.toString())}/_password`;
        }
        else {
            method = 'PUT';
            path = '/_security/user/_password';
        }
        const meta = {
            name: 'security.change_password',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clearApiKeyCache(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.clear_api_key_cache'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_security/api_key/${encodeURIComponent(params.ids.toString())}/_clear_cache`;
        const meta = {
            name: 'security.clear_api_key_cache',
            pathParts: {
                ids: params.ids
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clearCachedPrivileges(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.clear_cached_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_security/privilege/${encodeURIComponent(params.application.toString())}/_clear_cache`;
        const meta = {
            name: 'security.clear_cached_privileges',
            pathParts: {
                application: params.application
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clearCachedRealms(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.clear_cached_realms'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_security/realm/${encodeURIComponent(params.realms.toString())}/_clear_cache`;
        const meta = {
            name: 'security.clear_cached_realms',
            pathParts: {
                realms: params.realms
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clearCachedRoles(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.clear_cached_roles'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_security/role/${encodeURIComponent(params.name.toString())}/_clear_cache`;
        const meta = {
            name: 'security.clear_cached_roles',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async clearCachedServiceTokens(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.clear_cached_service_tokens'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'POST';
        const path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}/credential/token/${encodeURIComponent(params.name.toString())}/_clear_cache`;
        const meta = {
            name: 'security.clear_cached_service_tokens',
            pathParts: {
                namespace: params.namespace,
                service: params.service,
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async createApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.create_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = '/_security/api_key';
        const meta = {
            name: 'security.create_api_key'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async createCrossClusterApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.create_cross_cluster_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/cross_cluster/api_key';
        const meta = {
            name: 'security.create_cross_cluster_api_key'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async createServiceToken(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.create_service_token'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.namespace != null && params.service != null && params.name != null) {
            method = 'PUT';
            path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}/credential/token/${encodeURIComponent(params.name.toString())}`;
        }
        else {
            method = 'POST';
            path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}/credential/token`;
        }
        const meta = {
            name: 'security.create_service_token',
            pathParts: {
                namespace: params.namespace,
                service: params.service,
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async delegatePki(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.delegate_pki'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/delegate_pki';
        const meta = {
            name: 'security.delegate_pki'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deletePrivileges(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.delete_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_security/privilege/${encodeURIComponent(params.application.toString())}/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.delete_privileges',
            pathParts: {
                application: params.application,
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deleteRole(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.delete_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_security/role/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.delete_role',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deleteRoleMapping(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.delete_role_mapping'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_security/role_mapping/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.delete_role_mapping',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deleteServiceToken(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.delete_service_token'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}/credential/token/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.delete_service_token',
            pathParts: {
                namespace: params.namespace,
                service: params.service,
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async deleteUser(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.delete_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'DELETE';
        const path = `/_security/user/${encodeURIComponent(params.username.toString())}`;
        const meta = {
            name: 'security.delete_user',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async disableUser(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.disable_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'PUT';
        const path = `/_security/user/${encodeURIComponent(params.username.toString())}/_disable`;
        const meta = {
            name: 'security.disable_user',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async disableUserProfile(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.disable_user_profile'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'PUT';
        const path = `/_security/profile/${encodeURIComponent(params.uid.toString())}/_disable`;
        const meta = {
            name: 'security.disable_user_profile',
            pathParts: {
                uid: params.uid
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async enableUser(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.enable_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'PUT';
        const path = `/_security/user/${encodeURIComponent(params.username.toString())}/_enable`;
        const meta = {
            name: 'security.enable_user',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async enableUserProfile(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.enable_user_profile'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'PUT';
        const path = `/_security/profile/${encodeURIComponent(params.uid.toString())}/_enable`;
        const meta = {
            name: 'security.enable_user_profile',
            pathParts: {
                uid: params.uid
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async enrollKibana(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.enroll_kibana'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/enroll/kibana';
        const meta = {
            name: 'security.enroll_kibana'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async enrollNode(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.enroll_node'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/enroll/node';
        const meta = {
            name: 'security.enroll_node'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getApiKey(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/api_key';
        const meta = {
            name: 'security.get_api_key'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getBuiltinPrivileges(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_builtin_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/privilege/_builtin';
        const meta = {
            name: 'security.get_builtin_privileges'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getPrivileges(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.application != null && params.name != null) {
            method = 'GET';
            path = `/_security/privilege/${encodeURIComponent(params.application.toString())}/${encodeURIComponent(params.name.toString())}`;
        }
        else if (params.application != null) {
            method = 'GET';
            path = `/_security/privilege/${encodeURIComponent(params.application.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_security/privilege';
        }
        const meta = {
            name: 'security.get_privileges',
            pathParts: {
                application: params.application,
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getRole(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.name != null) {
            method = 'GET';
            path = `/_security/role/${encodeURIComponent(params.name.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_security/role';
        }
        const meta = {
            name: 'security.get_role',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getRoleMapping(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_role_mapping'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.name != null) {
            method = 'GET';
            path = `/_security/role_mapping/${encodeURIComponent(params.name.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_security/role_mapping';
        }
        const meta = {
            name: 'security.get_role_mapping',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getServiceAccounts(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_service_accounts'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.namespace != null && params.service != null) {
            method = 'GET';
            path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}`;
        }
        else if (params.namespace != null) {
            method = 'GET';
            path = `/_security/service/${encodeURIComponent(params.namespace.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_security/service';
        }
        const meta = {
            name: 'security.get_service_accounts',
            pathParts: {
                namespace: params.namespace,
                service: params.service
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getServiceCredentials(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_service_credentials'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = `/_security/service/${encodeURIComponent(params.namespace.toString())}/${encodeURIComponent(params.service.toString())}/credential`;
        const meta = {
            name: 'security.get_service_credentials',
            pathParts: {
                namespace: params.namespace,
                service: params.service
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getSettings(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_settings'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/settings';
        const meta = {
            name: 'security.get_settings'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getToken(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.get_token'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/oauth2/token';
        const meta = {
            name: 'security.get_token'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getUser(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        let method = '';
        let path = '';
        if (params.username != null) {
            method = 'GET';
            path = `/_security/user/${encodeURIComponent(params.username.toString())}`;
        }
        else {
            method = 'GET';
            path = '/_security/user';
        }
        const meta = {
            name: 'security.get_user',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getUserPrivileges(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_user_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = '/_security/user/_privileges';
        const meta = {
            name: 'security.get_user_privileges'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async getUserProfile(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.get_user_profile'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = `/_security/profile/${encodeURIComponent(params.uid.toString())}`;
        const meta = {
            name: 'security.get_user_profile',
            pathParts: {
                uid: params.uid
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async grantApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.grant_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/api_key/grant';
        const meta = {
            name: 'security.grant_api_key'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async hasPrivileges(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.has_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        let method = '';
        let path = '';
        if (params.user != null) {
            method = body != null ? 'POST' : 'GET';
            path = `/_security/user/${encodeURIComponent(params.user.toString())}/_has_privileges`;
        }
        else {
            method = body != null ? 'POST' : 'GET';
            path = '/_security/user/_has_privileges';
        }
        const meta = {
            name: 'security.has_privileges',
            pathParts: {
                user: params.user
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async hasPrivilegesUserProfile(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.has_privileges_user_profile'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = body != null ? 'POST' : 'GET';
        const path = '/_security/profile/_has_privileges';
        const meta = {
            name: 'security.has_privileges_user_profile'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async invalidateApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.invalidate_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'DELETE';
        const path = '/_security/api_key';
        const meta = {
            name: 'security.invalidate_api_key'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async invalidateToken(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.invalidate_token'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'DELETE';
        const path = '/_security/oauth2/token';
        const meta = {
            name: 'security.invalidate_token'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async oidcAuthenticate(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.oidc_authenticate'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/oidc/authenticate';
        const meta = {
            name: 'security.oidc_authenticate'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async oidcLogout(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.oidc_logout'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/oidc/logout';
        const meta = {
            name: 'security.oidc_logout'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async oidcPrepareAuthentication(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.oidc_prepare_authentication'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/oidc/prepare';
        const meta = {
            name: 'security.oidc_prepare_authentication'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async putPrivileges(params, options) {
        var _a;
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.put_privileges'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body = (_a = params.body) !== null && _a !== void 0 ? _a : undefined;
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                // @ts-expect-error
                body = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = '/_security/privilege';
        const meta = {
            name: 'security.put_privileges'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async putRole(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.put_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/role/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.put_role',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async putRoleMapping(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.put_role_mapping'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/role_mapping/${encodeURIComponent(params.name.toString())}`;
        const meta = {
            name: 'security.put_role_mapping',
            pathParts: {
                name: params.name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async putUser(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.put_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/user/${encodeURIComponent(params.username.toString())}`;
        const meta = {
            name: 'security.put_user',
            pathParts: {
                username: params.username
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async queryApiKeys(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.query_api_keys'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = body != null ? 'POST' : 'GET';
        const path = '/_security/_query/api_key';
        const meta = {
            name: 'security.query_api_keys'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async queryRole(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.query_role'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = body != null ? 'POST' : 'GET';
        const path = '/_security/_query/role';
        const meta = {
            name: 'security.query_role'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async queryUser(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.query_user'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = body != null ? 'POST' : 'GET';
        const path = '/_security/_query/user';
        const meta = {
            name: 'security.query_user'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlAuthenticate(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.saml_authenticate'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/saml/authenticate';
        const meta = {
            name: 'security.saml_authenticate'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlCompleteLogout(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.saml_complete_logout'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/saml/complete_logout';
        const meta = {
            name: 'security.saml_complete_logout'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlInvalidate(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.saml_invalidate'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/saml/invalidate';
        const meta = {
            name: 'security.saml_invalidate'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlLogout(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.saml_logout'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/saml/logout';
        const meta = {
            name: 'security.saml_logout'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlPrepareAuthentication(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.saml_prepare_authentication'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'POST';
        const path = '/_security/saml/prepare';
        const meta = {
            name: 'security.saml_prepare_authentication'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async samlServiceProviderMetadata(params, options) {
        const { path: acceptedPath } = this.acceptedParams['security.saml_service_provider_metadata'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                // @ts-expect-error
                querystring[key] = params[key];
            }
        }
        const method = 'GET';
        const path = `/_security/saml/metadata/${encodeURIComponent(params.realm_name.toString())}`;
        const meta = {
            name: 'security.saml_service_provider_metadata',
            pathParts: {
                realm_name: params.realm_name
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async suggestUserProfiles(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.suggest_user_profiles'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = body != null ? 'POST' : 'GET';
        const path = '/_security/profile/_suggest';
        const meta = {
            name: 'security.suggest_user_profiles'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async updateApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.update_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/api_key/${encodeURIComponent(params.id.toString())}`;
        const meta = {
            name: 'security.update_api_key',
            pathParts: {
                id: params.id
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async updateCrossClusterApiKey(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.update_cross_cluster_api_key'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/cross_cluster/api_key/${encodeURIComponent(params.id.toString())}`;
        const meta = {
            name: 'security.update_cross_cluster_api_key',
            pathParts: {
                id: params.id
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async updateSettings(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.update_settings'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        params = params !== null && params !== void 0 ? params : {};
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = '/_security/settings';
        const meta = {
            name: 'security.update_settings'
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
    async updateUserProfileData(params, options) {
        const { path: acceptedPath, body: acceptedBody, query: acceptedQuery } = this.acceptedParams['security.update_user_profile_data'];
        const userQuery = params === null || params === void 0 ? void 0 : params.querystring;
        const querystring = userQuery != null ? { ...userQuery } : {};
        let body;
        const userBody = params === null || params === void 0 ? void 0 : params.body;
        if (userBody != null) {
            if (typeof userBody === 'string') {
                body = userBody;
            }
            else {
                body = { ...userBody };
            }
        }
        for (const key in params) {
            if (acceptedBody.includes(key)) {
                body = body !== null && body !== void 0 ? body : {};
                // @ts-expect-error
                body[key] = params[key];
            }
            else if (acceptedPath.includes(key)) {
                continue;
            }
            else if (key !== 'body' && key !== 'querystring') {
                if (acceptedQuery.includes(key) || commonQueryParams.includes(key)) {
                    // @ts-expect-error
                    querystring[key] = params[key];
                }
                else {
                    body = body !== null && body !== void 0 ? body : {};
                    // @ts-expect-error
                    body[key] = params[key];
                }
            }
        }
        const method = 'PUT';
        const path = `/_security/profile/${encodeURIComponent(params.uid.toString())}/_data`;
        const meta = {
            name: 'security.update_user_profile_data',
            pathParts: {
                uid: params.uid
            }
        };
        return await this.transport.request({ path, method, querystring, body, meta }, options);
    }
}
exports.default = Security;
//# sourceMappingURL=security.js.map