"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DrilldownState = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _useObservable = _interopRequireDefault(require("react-use/lib/useObservable"));
var _rxjs = require("rxjs");
var _use_sync_observable = require("../hooks/use_sync_observable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * An instance of this class represents UI states of a single drilldown which
 * is currently being created or edited.
 */
class DrilldownState {
  constructor({
    factory,
    placeTriggers,
    placeContext,
    name: _name = '',
    triggers: _triggers = [],
    config: _config = {}
  }) {
    /**
     * Drilldown type used to configure this drilldown.
     */
    (0, _defineProperty2.default)(this, "factory", void 0);
    /**
     * Opaque action factory context object excluding the `triggers` attribute.
     */
    (0, _defineProperty2.default)(this, "placeContext", void 0);
    /**
     * User entered name of this drilldown.
     */
    (0, _defineProperty2.default)(this, "name$", void 0);
    /**
     * Whether the `name$` is valid or is in an error state.
     */
    (0, _defineProperty2.default)(this, "nameError$", void 0);
    /**
     * List of all triggers the place which opened the Drilldown Manager supports.
     */
    (0, _defineProperty2.default)(this, "placeTriggers", void 0);
    /**
     * List of all triggers from which the user can pick in UI for this specific
     * drilldown. This is the selection list we show to the user. It is an
     * intersection of all triggers supported by current place with the triggers
     * that the action factory supports.
     */
    (0, _defineProperty2.default)(this, "uiTriggers", void 0);
    /**
     * User selected triggers. (Currently in UI we support user picking just one trigger).
     */
    (0, _defineProperty2.default)(this, "triggers$", void 0);
    /**
     * Error identifier, in case `triggers$` is in an error state.
     */
    (0, _defineProperty2.default)(this, "triggersError$", void 0);
    /**
     * Current action factory (drilldown) configuration, i.e. drilldown
     * configuration object, which will be serialized and persisted in storage.
     */
    (0, _defineProperty2.default)(this, "config$", void 0);
    /**
     * Error identifier, in case `config$` is in an error state.
     */
    (0, _defineProperty2.default)(this, "configError$", void 0);
    /**
     * Whether the drilldown state is in an error and should not be saved. I value
     * is `undefined`, there is no error.
     */
    (0, _defineProperty2.default)(this, "error$", void 0);
    /**
     * Change the name of the drilldown.
     */
    (0, _defineProperty2.default)(this, "setName", name => {
      this.name$.next(name);
    });
    /**
     * Change the list of user selected triggers.
     */
    (0, _defineProperty2.default)(this, "setTriggers", triggers => {
      this.triggers$.next(triggers);
    });
    /**
     * Update the current drilldown configuration.
     */
    (0, _defineProperty2.default)(this, "setConfig", config => {
      this.config$.next(config);
    });
    // Below are convenience React hooks for consuming observables in connected
    // React components.
    (0, _defineProperty2.default)(this, "useName", () => (0, _useObservable.default)(this.name$, this.name$.getValue()));
    (0, _defineProperty2.default)(this, "useTriggers", () => (0, _useObservable.default)(this.triggers$, this.triggers$.getValue()));
    (0, _defineProperty2.default)(this, "useConfig", () => (0, _useObservable.default)(this.config$, this.config$.getValue()));
    (0, _defineProperty2.default)(this, "useError", () => (0, _use_sync_observable.useSyncObservable)(this.error$));
    this.factory = factory;
    this.placeTriggers = placeTriggers;
    this.placeContext = placeContext;
    this.name$ = new _rxjs.BehaviorSubject(_name);
    this.triggers$ = new _rxjs.BehaviorSubject(_triggers);
    this.config$ = new _rxjs.BehaviorSubject(_config);
    const triggersFactorySupports = this.factory.supportedTriggers();
    this.uiTriggers = triggersFactorySupports.filter(trigger => this.placeTriggers.includes(trigger));

    // Pre-select a trigger if there is only one trigger for user to choose from.
    // In case there is only one possible trigger, UI will not display a trigger picker.
    if (this.uiTriggers.length === 1) this.triggers$.next([this.uiTriggers[0]]);
    this.nameError$ = this.name$.pipe((0, _rxjs.map)(currentName => {
      if (!currentName) return 'NAME_EMPTY';
      return undefined;
    }));
    this.triggersError$ = this.triggers$.pipe((0, _rxjs.map)(currentTriggers => {
      if (!currentTriggers.length) return 'NO_TRIGGERS_SELECTED';
      return undefined;
    }));
    this.configError$ = this.config$.pipe((0, _rxjs.map)(conf => {
      if (!this.factory.isConfigValid(conf, this.getFactoryContext())) return 'INVALID_CONFIG';
      return undefined;
    }));
    this.error$ = (0, _rxjs.combineLatest)([this.nameError$, this.triggersError$, this.configError$]).pipe((0, _rxjs.map)(([nameError, configError, triggersError]) => nameError || triggersError || configError || undefined));
  }
  getFactoryContext() {
    return {
      ...this.placeContext,
      triggers: this.triggers$.getValue()
    };
  }

  /**
   * Serialize the current drilldown draft into a serializable action which
   * is persisted to disk.
   */
  serialize() {
    return {
      factoryId: this.factory.id,
      name: this.name$.getValue(),
      config: this.config$.getValue()
    };
  }

  /**
   * Returns a list of all triggers from which user can pick in UI, for this
   * specific drilldown.
   */
  getAllDrilldownTriggers() {
    const triggersFactorySupports = this.factory.supportedTriggers();
    const uiTriggers = triggersFactorySupports.filter(trigger => this.placeTriggers.includes(trigger));
    return uiTriggers;
  }
  isValid() {
    if (!this.name$.getValue()) return false;
    const config = this.config$.getValue();
    if (!config) return false;
    const triggers = this.triggers$.getValue();
    if (triggers.length < 1) return false;
    if (!this.factory.isConfigValid(config, this.getFactoryContext())) return false;
    return true;
  }
}
exports.DrilldownState = DrilldownState;