"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withActiveSpan = withActiveSpan;
var _std = require("@kbn/std");
var _api = require("@opentelemetry/api");
var _sdk = require("@elastic/opentelemetry-node/sdk");
var _rxjs = require("rxjs");
var _defaultTracer = require("@kbn/default-tracer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// gives the desired union

/**
 * Starts an active span in the given or currently active context,
 * unwraps returned promises and observables, and automatically
 * records exceptions and sets the status on the span.
 */
function withActiveSpan(...args) {
  const name = args[0];
  const opts = args.length === 2 ? {} : args[1];
  const ctx = args.length === 4 ? args[2] : _api.context.active();
  const cb = args.length === 2 ? args[1] : args.length === 3 ? args[2] : args[3];
  const tracer = opts.tracer || (0, _defaultTracer.getDefaultTracer)();
  if (!tracer || _sdk.core.isTracingSuppressed(ctx)) {
    return cb();
  }
  return tracer.startActiveSpan(name, opts, ctx, span => {
    try {
      const res = cb(span);
      if ((0, _std.isPromise)(res)) {
        return handlePromise(span, res);
      }
      if ((0, _rxjs.isObservable)(res)) {
        return handleObservable(span, ctx, res);
      }
      updateSpanSafely(span, () => {
        span.setStatus({
          code: _api.SpanStatusCode.OK
        });
        span.end();
      });
      return res;
    } catch (error) {
      updateSpanSafely(span, () => {
        span.recordException(error);
        span.setStatus({
          code: _api.SpanStatusCode.ERROR,
          message: error.message
        });
        span.end();
      });
      throw error;
    }
  });
}
function handleObservable(span, parentContext, source$) {
  const ctx = _api.context.active();
  return new _rxjs.Observable(subscriber => {
    // Make sure anything that happens during this callback uses the context
    // that was active when this function was called. this ensures correct
    // span parenting
    const subscription = _api.context.with(ctx, () => {
      return source$.subscribe({
        next: value => {
          _api.context.with(parentContext, () => {
            subscriber.next(value);
          });
        },
        error: error => {
          _api.context.with(parentContext, () => {
            updateSpanSafely(span, () => {
              span.recordException(error);
              span.setStatus({
                code: _api.SpanStatusCode.ERROR,
                message: error.message
              });
              span.end();
            });
            subscriber.error(error);
          });
        },
        complete: () => {
          _api.context.with(parentContext, () => {
            updateSpanSafely(span, () => {
              span.setStatus({
                code: _api.SpanStatusCode.OK
              });
              span.end();
            });
            subscriber.complete();
          });
        }
      });
    });
    return () => _api.context.with(parentContext, () => subscription.unsubscribe());
  });
}
function handlePromise(span, promise) {
  return promise.then(res => {
    updateSpanSafely(span, () => {
      span.setStatus({
        code: _api.SpanStatusCode.OK
      });
      span.end();
    });
    return res;
  }).catch(error => {
    updateSpanSafely(span, () => {
      span.recordException(error);
      span.setStatus({
        code: _api.SpanStatusCode.ERROR,
        message: error.message
      });
      span.end();
    });
    throw error;
  });
}
function updateSpanSafely(span, cb) {
  if (span.isRecording()) {
    cb();
  }
}