"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskScheduling = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _intervals = require("./lib/intervals");
var _task = require("./task");
var _correct_deprecated_fields = require("./lib/correct_deprecated_fields");
var _retryable_bulk_update = require("./lib/retryable_bulk_update");
var _get_next_run_at = require("./lib/get_next_run_at");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const VERSION_CONFLICT_STATUS = 409;
const BULK_ACTION_SIZE = 100;

/**
 * return type of TaskScheduling.bulkUpdateSchedules method
 */

class TaskScheduling {
  /**
   * Initializes the task manager, preventing any further addition of middleware,
   * enabling the task manipulation methods, and beginning the background polling
   * mechanism.
   */
  constructor(opts) {
    (0, _defineProperty2.default)(this, "store", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "middleware", void 0);
    this.logger = opts.logger;
    this.middleware = opts.middleware;
    this.store = opts.taskStore;
  }

  /**
   * Schedules a task.
   *
   * @param task - The task being scheduled.
   * @returns {Promise<ConcreteTaskInstance>}
   */
  async schedule(taskInstance, options) {
    var _modifiedTask$enabled;
    const {
      taskInstance: modifiedTask
    } = await this.middleware.beforeSave({
      ...(0, _lodash.omit)(options, 'apiKey', 'request'),
      taskInstance: (0, _correct_deprecated_fields.ensureDeprecatedFieldsAreCorrected)(taskInstance, this.logger)
    });
    const traceparent = _elasticApmNode.default.currentTransaction && _elasticApmNode.default.currentTransaction.type !== 'request' ? _elasticApmNode.default.currentTraceparent : '';
    return await this.store.schedule({
      ...modifiedTask,
      traceparent: traceparent || '',
      enabled: (_modifiedTask$enabled = modifiedTask.enabled) !== null && _modifiedTask$enabled !== void 0 ? _modifiedTask$enabled : true
    }, options !== null && options !== void 0 && options.request ? {
      request: options === null || options === void 0 ? void 0 : options.request
    } : undefined);
  }

  /**
   * Bulk schedules a task.
   *
   * @param tasks - The tasks being scheduled.
   * @returns {Promise<ConcreteTaskInstance>}
   */
  async bulkSchedule(taskInstances, options) {
    const traceparent = _elasticApmNode.default.currentTransaction && _elasticApmNode.default.currentTransaction.type !== 'request' ? _elasticApmNode.default.currentTraceparent : '';
    const modifiedTasks = await Promise.all(taskInstances.map(async taskInstance => {
      var _modifiedTask$enabled2;
      const {
        taskInstance: modifiedTask
      } = await this.middleware.beforeSave({
        ...(0, _lodash.omit)(options, 'apiKey', 'request'),
        taskInstance: (0, _correct_deprecated_fields.ensureDeprecatedFieldsAreCorrected)(taskInstance, this.logger)
      });
      return {
        ...modifiedTask,
        traceparent: traceparent || '',
        enabled: (_modifiedTask$enabled2 = modifiedTask.enabled) !== null && _modifiedTask$enabled2 !== void 0 ? _modifiedTask$enabled2 : true
      };
    }));
    return await this.store.bulkSchedule(modifiedTasks, options !== null && options !== void 0 && options.request ? {
      request: options === null || options === void 0 ? void 0 : options.request
    } : undefined);
  }
  async bulkDisable(taskIds, clearStateIdsOrBoolean) {
    return await (0, _retryable_bulk_update.retryableBulkUpdate)({
      taskIds,
      store: this.store,
      getTasks: async ids => await this.bulkGetTasksHelper(ids),
      filter: task => !!task.enabled,
      map: task => ({
        ...task,
        enabled: false,
        ...(Array.isArray(clearStateIdsOrBoolean) && clearStateIdsOrBoolean.includes(task.id) || clearStateIdsOrBoolean === true ? {
          state: {}
        } : {})
      }),
      validate: false
    });
  }
  async bulkEnable(taskIds, runSoon = true) {
    return await (0, _retryable_bulk_update.retryableBulkUpdate)({
      taskIds,
      store: this.store,
      getTasks: async ids => await this.bulkGetTasksHelper(ids),
      filter: task => !task.enabled,
      map: (task, i) => {
        if (runSoon) {
          // Run the first task now. Run all other tasks a random number of ms in the future,
          // with a maximum of 5 minutes or the task interval, whichever is smaller.
          const taskToRun = i === 0 ? {
            ...task,
            runAt: new Date(),
            scheduledAt: new Date()
          } : randomlyOffsetRunTimestamp(task);
          return {
            ...taskToRun,
            enabled: true
          };
        }
        return {
          ...task,
          enabled: true
        };
      },
      validate: false
    });
  }
  async bulkUpdateState(taskIds, stateMapFn) {
    return await (0, _retryable_bulk_update.retryableBulkUpdate)({
      taskIds,
      store: this.store,
      getTasks: async ids => await this.bulkGetTasksHelper(ids),
      filter: () => true,
      map: task => ({
        ...task,
        state: stateMapFn(task.state, task.id)
      }),
      validate: false
    });
  }

  /**
   * Bulk updates schedules for tasks by ids.
   * Only tasks with `idle` status will be updated, as for the tasks which have `running` status,
   * `schedule` and `runAt` will be recalculated after task run finishes
   *
   * @param {string[]} taskIds  - list of task ids
   * @param {IntervalSchedule | RruleSchedule} schedule  - new schedule
   * @returns {Promise<BulkUpdateTaskResult>}
   */
  async bulkUpdateSchedules(taskIds, schedule) {
    return (0, _retryable_bulk_update.retryableBulkUpdate)({
      taskIds,
      store: this.store,
      getTasks: async ids => await this.bulkGetTasksHelper(ids),
      filter: task => task.status === _task.TaskStatus.Idle && !(0, _lodash.isEqual)(task.schedule, schedule),
      map: task => {
        const newRunAtInMs = (0, _get_next_run_at.calculateNextRunAtFromSchedule)({
          schedule,
          startDate: task.scheduledAt
        });
        return {
          ...task,
          schedule,
          runAt: new Date(newRunAtInMs)
        };
      },
      validate: false,
      /**
       * Because the schedule can be converted from Interval to Rrule and vice versa we want to a void a situation
       * where both are defined by passing mergeAttributes: false here.
       */
      mergeAttributes: false
    });
  }
  async bulkGetTasksHelper(taskIds) {
    const batches = await (0, _pMap.default)((0, _lodash.chunk)(taskIds, BULK_ACTION_SIZE), async taskIdsChunk => this.store.bulkGet(taskIdsChunk), {
      concurrency: 10
    });
    return (0, _lodash.flatten)(batches);
  }

  /**
   * Run task.
   *
   * @param taskId - The task being scheduled.
   * @returns {Promise<RunSoonResult>}
   */
  async runSoon(taskId) {
    const task = await this.getNonRunningTask(taskId);
    try {
      await this.store.update({
        ...task,
        status: _task.TaskStatus.Idle,
        scheduledAt: new Date(),
        runAt: new Date()
      }, {
        validate: false
      });
    } catch (e) {
      if (e.statusCode === 409) {
        this.logger.debug(`Failed to update the task (${taskId}) for runSoon due to conflict (409)`);
      } else {
        this.logger.error(`Failed to update the task (${taskId}) for runSoon`);
        throw e;
      }
    }
    return {
      id: task.id
    };
  }

  /**
   * Schedules a task with an Id
   *
   * @param task - The task being scheduled.
   * @returns {Promise<TaskInstanceWithId>}
   */
  async ensureScheduled(taskInstance, options) {
    try {
      return await this.schedule(taskInstance, options);
    } catch (err) {
      if (err.statusCode === VERSION_CONFLICT_STATUS) {
        // check if task specifies a schedule interval
        // if so,try to update the just the schedule
        // only works for interval schedule
        if (taskInstance.schedule && taskInstance.schedule.interval) {
          const result = await this.bulkUpdateSchedules([taskInstance.id], taskInstance.schedule);
          if (result.errors.length && result.errors[0].error.statusCode !== VERSION_CONFLICT_STATUS) {
            throw new Error(`Tried to update schedule for existing task "${taskInstance.id}" but failed with error: ${result.errors[0].error.message}`);
          }
        }
        return taskInstance;
      }
      throw err;
    }
  }
  async getNonRunningTask(taskId) {
    const task = await this.store.get(taskId);
    switch (task.status) {
      case _task.TaskStatus.Claiming:
      case _task.TaskStatus.Running:
        throw Error(`Failed to run task "${taskId}" as it is currently running`);
      case _task.TaskStatus.Unrecognized:
        throw Error(`Failed to run task "${taskId}" with status ${task.status}`);
      case _task.TaskStatus.Failed:
      default:
        return task;
    }
  }
}
exports.TaskScheduling = TaskScheduling;
const randomlyOffsetRunTimestamp = task => {
  var _task$schedule$interv, _task$schedule;
  const now = Date.now();
  const maximumOffsetTimestamp = now + 1000 * 60 * 5; // now + 5 minutes
  const taskIntervalInMs = (0, _intervals.parseIntervalAsMillisecond)((_task$schedule$interv = (_task$schedule = task.schedule) === null || _task$schedule === void 0 ? void 0 : _task$schedule.interval) !== null && _task$schedule$interv !== void 0 ? _task$schedule$interv : '0s');
  const maximumRunAt = Math.min(now + taskIntervalInMs, maximumOffsetTimestamp);

  // Offset between 1 and maximumRunAt ms
  const runAt = new Date(now + Math.floor(Math.random() * (maximumRunAt - now) + 1));
  return {
    ...task,
    runAt,
    scheduledAt: runAt
  };
};