"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NO_ASSIGNED_PARTITIONS_WARNING_INTERVAL = void 0;
exports.claimAvailableTasksMget = claimAvailableTasksMget;
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _wrapped_logger = require("../lib/wrapped_logger");
var _task_type_dictionary = require("../task_type_dictionary");
var _ = require(".");
var _task = require("../task");
var _task_running = require("../task_running");
var _task_claiming = require("../queries/task_claiming");
var _task_events = require("../task_events");
var _query_clauses = require("../queries/query_clauses");
var _mark_available_tasks_as_claimed = require("../queries/mark_available_tasks_as_claimed");
var _result_type = require("../lib/result_type");
var _task_selector_by_capacity = require("./lib/task_selector_by_capacity");
var _get_retry_at = require("../lib/get_retry_at");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Basic operation of this task claimer:
// - search for candidate tasks to run, more than we actually can run
// - initial search returns a slimmer task document for I/O efficiency (no params or state)
// - for each task found, do an mget to get the current seq_no and primary_term
// - if the mget result doesn't match the search result, the task is stale
// - from the non-stale search results, return as many as we can run based on available
//   capacity and the cost of each task type to run

const SIZE_MULTIPLIER_FOR_TASK_FETCH = 4;
async function claimAvailableTasksMget(opts) {
  const apmTrans = _elasticApmNode.default.startTransaction(_task_claiming.TASK_MANAGER_MARK_AS_CLAIMED, _task_running.TASK_MANAGER_TRANSACTION_TYPE);
  try {
    const result = await claimAvailableTasks(opts);
    apmTrans.end('success');
    return result;
  } catch (err) {
    apmTrans.end('failure');
    throw err;
  }
}
async function claimAvailableTasks(opts) {
  const {
    getCapacity,
    claimOwnershipUntil,
    batches,
    events$,
    taskStore,
    taskPartitioner
  } = opts;
  const {
    definitions,
    excludedTaskTypes,
    taskMaxAttempts
  } = opts;
  const logger = (0, _wrapped_logger.createWrappedLogger)({
    logger: opts.logger,
    tags: [claimAvailableTasksMget.name]
  });
  const initialCapacity = getCapacity();
  const stopTaskTimer = (0, _task_events.startTaskTimer)();

  // get a list of candidate tasks to claim, with their version info
  const {
    docs,
    versionMap
  } = await searchAvailableTasks({
    definitions,
    taskTypes: new Set(definitions.getAllTypes()),
    excludedTaskTypePatterns: excludedTaskTypes,
    taskStore,
    events$,
    claimOwnershipUntil,
    getCapacity,
    // set size to accommodate the possibility of retrieving all
    // tasks with the smallest cost, with a size multipler to account
    // for possible conflicts
    size: initialCapacity * _task.TaskCost.Tiny * SIZE_MULTIPLIER_FOR_TASK_FETCH,
    taskMaxAttempts,
    taskPartitioner,
    logger
  });
  if (docs.length === 0) return {
    ...(0, _.getEmptyClaimOwnershipResult)(),
    timing: stopTaskTimer()
  };

  // use mget to get the latest version of each task
  const docLatestVersions = await taskStore.getDocVersions(docs.map(doc => `task:${doc.id}`));

  // filter out stale and missing tasks
  const currentTasks = [];
  const staleTasks = [];
  const missingTasks = [];
  for (const searchDoc of docs) {
    const searchVersion = versionMap.get(searchDoc.id);
    const latestVersion = docLatestVersions.get(`task:${searchDoc.id}`);
    if (!searchVersion || !latestVersion) {
      missingTasks.push(searchDoc);
      continue;
    }
    if (searchVersion.seqNo === latestVersion.seqNo && searchVersion.primaryTerm === latestVersion.primaryTerm) {
      currentTasks.push(searchDoc);
      continue;
    } else {
      staleTasks.push(searchDoc);
      continue;
    }
  }

  // apply limited concurrency limits (TODO: can currently starve other tasks)
  const candidateTasks = (0, _task_selector_by_capacity.selectTasksByCapacity)({
    definitions,
    tasks: currentTasks,
    batches
  });

  // apply capacity constraint to candidate tasks
  const tasksToRun = [];
  const leftOverTasks = [];
  const tasksWithMalformedData = [];
  let capacityAccumulator = 0;
  for (const task of candidateTasks) {
    var _definitions$get$cost, _definitions$get;
    const taskCost = (_definitions$get$cost = (_definitions$get = definitions.get(task.taskType)) === null || _definitions$get === void 0 ? void 0 : _definitions$get.cost) !== null && _definitions$get$cost !== void 0 ? _definitions$get$cost : _task.TaskCost.Normal;
    if (capacityAccumulator + taskCost <= initialCapacity) {
      tasksToRun.push(task);
      capacityAccumulator += taskCost;
    } else {
      leftOverTasks.push(task);
      capacityAccumulator = initialCapacity;
    }
  }

  // build the updated task objects we'll claim
  const now = new Date();
  const taskUpdates = [];
  for (const task of tasksToRun) {
    try {
      var _getRetryAt;
      taskUpdates.push({
        id: task.id,
        version: task.version,
        scheduledAt: task.retryAt != null && new Date(task.retryAt).getTime() < Date.now() ? task.retryAt : task.runAt,
        status: _task.TaskStatus.Running,
        startedAt: now,
        attempts: task.attempts + 1,
        retryAt: (_getRetryAt = (0, _get_retry_at.getRetryAt)(task, definitions.get(task.taskType))) !== null && _getRetryAt !== void 0 ? _getRetryAt : null,
        ownerId: taskStore.taskManagerId
      });
    } catch (error) {
      logger.error(`Error validating task schema ${task.id}:${task.taskType} during claim: ${JSON.stringify(error.message)}`);
      tasksWithMalformedData.push(task);
    }
  }

  // perform the task object updates, deal with errors
  const updatedTasks = {};
  let conflicts = 0;
  let bulkUpdateErrors = 0;
  let bulkGetErrors = 0;
  const updateResults = await taskStore.bulkPartialUpdate(taskUpdates);
  for (const updateResult of updateResults) {
    if ((0, _result_type.isOk)(updateResult)) {
      updatedTasks[updateResult.value.id] = updateResult.value;
    } else {
      const {
        id,
        type,
        error,
        status
      } = updateResult.error;

      // check for 409 conflict errors
      if (status === 409) {
        conflicts++;
      } else {
        logger.error(`Error updating task ${id}:${type} during claim: ${JSON.stringify(error)}`);
        bulkUpdateErrors++;
      }
    }
  }

  // perform an mget to get the full task instance for claiming
  const fullTasksToRun = (await taskStore.bulkGet(Object.keys(updatedTasks))).reduce((acc, task) => {
    if ((0, _result_type.isOk)(task) && task.value.version !== updatedTasks[task.value.id].version) {
      logger.warn(`Task ${task.value.id} was modified during the claiming phase, skipping until the next claiming cycle.`);
      conflicts++;
    } else if ((0, _result_type.isOk)(task)) {
      acc.push(task.value);
    } else {
      const {
        id,
        type,
        error
      } = task.error;
      logger.error(`Error getting full task ${id}:${type} during claim: ${error.message}`);
      bulkGetErrors++;
    }
    return acc;
  }, []);

  // TODO: need a better way to generate stats
  const message = `task claimer claimed: ${fullTasksToRun.length}; stale: ${staleTasks.length}; conflicts: ${conflicts}; missing: ${missingTasks.length}; capacity reached: ${leftOverTasks.length}; updateErrors: ${bulkUpdateErrors}; getErrors: ${bulkGetErrors}; malformed data errors: ${tasksWithMalformedData.length}`;
  logger.debug(message);

  // build results
  const finalResult = {
    stats: {
      tasksUpdated: fullTasksToRun.length,
      tasksConflicted: conflicts,
      tasksClaimed: fullTasksToRun.length,
      tasksLeftUnclaimed: leftOverTasks.length,
      tasksErrors: bulkUpdateErrors + bulkGetErrors + tasksWithMalformedData.length,
      staleTasks: staleTasks.length
    },
    docs: fullTasksToRun,
    timing: stopTaskTimer()
  };
  for (const doc of fullTasksToRun) {
    events$.next((0, _task_events.asTaskClaimEvent)(doc.id, (0, _result_type.asOk)(doc), finalResult.timing));
  }
  return finalResult;
}
let lastPartitionWarningLog;
const NO_ASSIGNED_PARTITIONS_WARNING_INTERVAL = exports.NO_ASSIGNED_PARTITIONS_WARNING_INTERVAL = 60000;
async function searchAvailableTasks({
  definitions,
  taskTypes,
  excludedTaskTypePatterns,
  taskStore,
  getCapacity,
  size,
  taskPartitioner,
  logger
}) {
  const excludedTaskTypes = new Set((0, _.getExcludedTaskTypes)(definitions, excludedTaskTypePatterns));
  const claimPartitions = buildClaimPartitions({
    types: taskTypes,
    excludedTaskTypes,
    getCapacity,
    definitions
  });
  const partitions = await taskPartitioner.getPartitions();
  if (partitions.length === 0 && (lastPartitionWarningLog == null || lastPartitionWarningLog <= Date.now() - NO_ASSIGNED_PARTITIONS_WARNING_INTERVAL)) {
    logger.warn(`Background task node "${taskPartitioner.getPodName()}" has no assigned partitions, claiming against all partitions`);
    lastPartitionWarningLog = Date.now();
  }
  if (partitions.length !== 0 && lastPartitionWarningLog) {
    lastPartitionWarningLog = undefined;
    logger.info(`Background task node "${taskPartitioner.getPodName()}" now claiming with assigned partitions`);
  }
  const sort = (0, _mark_available_tasks_as_claimed.getClaimSort)(definitions);
  const searches = [];

  // not handling removed types yet

  // add search for unlimited types
  if (claimPartitions.unlimitedTypes.length > 0) {
    const queryForUnlimitedTasks = (0, _query_clauses.mustBeAllOf)(
    // Task must be enabled
    _mark_available_tasks_as_claimed.EnabledTask,
    // a task type that's not excluded (may be removed or not)
    (0, _mark_available_tasks_as_claimed.OneOfTaskTypes)('task.taskType', claimPartitions.unlimitedTypes),
    // Either a task with idle status and runAt <= now or
    // status running or claiming with a retryAt <= now.
    (0, _query_clauses.shouldBeOneOf)(_mark_available_tasks_as_claimed.IdleTaskWithExpiredRunAt, _mark_available_tasks_as_claimed.RunningOrClaimingTaskWithExpiredRetryAt),
    // must have a status that isn't 'unrecognized'
    _mark_available_tasks_as_claimed.RecognizedTask);
    const queryUnlimitedTasks = (0, _query_clauses.matchesClauses)(queryForUnlimitedTasks, (0, _query_clauses.filterDownBy)(_mark_available_tasks_as_claimed.InactiveTasks), partitions.length ? (0, _mark_available_tasks_as_claimed.tasksWithPartitions)(partitions) : undefined);
    searches.push({
      query: queryUnlimitedTasks,
      sort,
      // note: we could optimize this to not sort on priority, for this case
      size,
      seq_no_primary_term: true
    });
  }

  // add searches for limited types
  for (const [types, capacity] of claimPartitions.limitedTypes) {
    const queryForLimitedTasks = (0, _query_clauses.mustBeAllOf)(
    // Task must be enabled
    _mark_available_tasks_as_claimed.EnabledTask,
    // Specific task type
    (0, _mark_available_tasks_as_claimed.OneOfTaskTypes)('task.taskType', types.split(',')),
    // Either a task with idle status and runAt <= now or
    // status running or claiming with a retryAt <= now.
    (0, _query_clauses.shouldBeOneOf)(_mark_available_tasks_as_claimed.IdleTaskWithExpiredRunAt, _mark_available_tasks_as_claimed.RunningOrClaimingTaskWithExpiredRetryAt),
    // must have a status that isn't 'unrecognized'
    _mark_available_tasks_as_claimed.RecognizedTask);
    const query = (0, _query_clauses.matchesClauses)(queryForLimitedTasks, (0, _query_clauses.filterDownBy)(_mark_available_tasks_as_claimed.InactiveTasks), partitions.length ? (0, _mark_available_tasks_as_claimed.tasksWithPartitions)(partitions) : undefined);
    searches.push({
      query,
      sort,
      size: capacity * SIZE_MULTIPLIER_FOR_TASK_FETCH,
      seq_no_primary_term: true
    });
  }
  return await taskStore.msearch(searches);
}
function buildClaimPartitions(opts) {
  const result = {
    unlimitedTypes: [],
    limitedTypes: new Map()
  };
  const {
    types,
    excludedTaskTypes,
    getCapacity,
    definitions
  } = opts;
  for (const type of types) {
    const definition = definitions.get(type);
    if (definition == null) continue;
    if (excludedTaskTypes.has(type)) continue;
    if (definition.maxConcurrency == null) {
      result.unlimitedTypes.push(definition.type);
      continue;
    }

    // task type has maxConcurrency defined

    const isSharingConcurrency = (0, _task_type_dictionary.sharedConcurrencyTaskTypes)(type);
    if (isSharingConcurrency) {
      let minCapacity = null;
      for (const sharedType of isSharingConcurrency) {
        const def = definitions.get(sharedType);
        if (def) {
          const capacity = getCapacity(def.type) / def.cost;
          if (minCapacity == null) {
            minCapacity = capacity;
          } else if (capacity < minCapacity) {
            minCapacity = capacity;
          }
        }
      }
      if (minCapacity) {
        result.limitedTypes.set(isSharingConcurrency.join(','), minCapacity);
      }
    } else {
      const capacity = getCapacity(definition.type) / definition.cost;
      if (capacity !== 0) {
        result.limitedTypes.set(definition.type, capacity);
      }
    }
  }
  return result;
}