"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskRunMetricsAggregator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _task_running = require("../task_running");
var _result_type = require("../lib/result_type");
var _task_events = require("../task_events");
var _lib = require("./lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const HDR_HISTOGRAM_MAX = 5400; // 90 minutes
const HDR_HISTOGRAM_BUCKET_SIZE = 10; // 10 seconds
var TaskRunKeys = /*#__PURE__*/function (TaskRunKeys) {
  TaskRunKeys["SUCCESS"] = "success";
  TaskRunKeys["NOT_TIMED_OUT"] = "not_timed_out";
  TaskRunKeys["TOTAL"] = "total";
  TaskRunKeys["TOTAL_ERRORS"] = "total_errors";
  TaskRunKeys["RESCHEDULED_FAILURES"] = "rescheduled_failures";
  TaskRunKeys["USER_ERRORS"] = "user_errors";
  TaskRunKeys["FRAMEWORK_ERRORS"] = "framework_errors";
  return TaskRunKeys;
}(TaskRunKeys || {});
var TaskRunMetricKeys = /*#__PURE__*/function (TaskRunMetricKeys) {
  TaskRunMetricKeys["OVERALL"] = "overall";
  TaskRunMetricKeys["BY_TYPE"] = "by_type";
  return TaskRunMetricKeys;
}(TaskRunMetricKeys || {});
class TaskRunMetricsAggregator {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "counter", new _lib.MetricCounterService(Object.values(TaskRunKeys), TaskRunMetricKeys.OVERALL));
    (0, _defineProperty2.default)(this, "delayHistogram", new _lib.SimpleHistogram(HDR_HISTOGRAM_MAX, HDR_HISTOGRAM_BUCKET_SIZE));
    this.logger = logger;
  }
  initialMetric() {
    return (0, _lodash.merge)(this.counter.initialMetrics(), {
      by_type: {},
      overall: {
        delay: {
          counts: [],
          values: []
        },
        delay_values: []
      }
    });
  }
  collect() {
    return (0, _lodash.merge)(this.counter.collect(), {
      overall: {
        delay: this.delayHistogram.serialize(),
        delay_values: this.delayHistogram.getAllValues()
      }
    });
  }
  reset() {
    this.counter.reset();
    this.delayHistogram.reset();
  }
  processTaskLifecycleEvent(taskEvent) {
    if ((0, _task_events.isTaskRunEvent)(taskEvent)) {
      this.processTaskRunEvent(taskEvent);
      this.logger.debug(() => `Collected metrics after processing lifecycle event - ${JSON.stringify(this.collect())}`);
    } else if ((0, _task_events.isTaskManagerStatEvent)(taskEvent)) {
      this.processTaskManagerStatEvent(taskEvent);
    }
  }
  processTaskRunEvent(taskEvent) {
    const taskRunResult = (0, _result_type.unwrap)(taskEvent.event);
    const {
      task,
      isExpired,
      result
    } = taskRunResult;
    const success = (0, _result_type.isOk)(taskEvent.event);
    const taskType = task.taskType.replaceAll('.', '__');
    const taskTypeGroup = (0, _lib.getTaskTypeGroup)(taskType);

    // increment the total counters
    this.incrementCounters(TaskRunKeys.TOTAL, taskType, taskTypeGroup);

    // increment success counters
    if (success) {
      this.incrementCounters(TaskRunKeys.SUCCESS, taskType, taskTypeGroup);
    } else {
      this.logger.debug(`Incrementing error counter for task ${task.taskType}`);
      // increment total error counts
      this.incrementCounters(TaskRunKeys.TOTAL_ERRORS, taskType, taskTypeGroup);
      if ((0, _task_running.isUserError)(taskRunResult.error)) {
        // increment the user error counters
        this.incrementCounters(TaskRunKeys.USER_ERRORS, taskType, taskTypeGroup);
      } else {
        // increment the framework error counters
        this.incrementCounters(TaskRunKeys.FRAMEWORK_ERRORS, taskType, taskTypeGroup);
      }
      if (result === _task_running.TaskRunResult.RetryScheduled) {
        // increment rescheduled failures
        this.incrementCounters(TaskRunKeys.RESCHEDULED_FAILURES, taskType, taskTypeGroup);
      }
    }

    // increment expired counters
    if (!isExpired) {
      this.incrementCounters(TaskRunKeys.NOT_TIMED_OUT, taskType, taskTypeGroup);
    }
  }
  processTaskManagerStatEvent(taskEvent) {
    if (taskEvent.id === 'runDelay') {
      const delayInSec = Math.round(taskEvent.event.value);
      this.delayHistogram.record(delayInSec);
    }
  }
  incrementCounters(key, taskType, group) {
    this.counter.increment(key, TaskRunMetricKeys.OVERALL);
    this.counter.increment(key, `${TaskRunMetricKeys.BY_TYPE}.${taskType}`);
    if (group) {
      this.counter.increment(key, `${TaskRunMetricKeys.BY_TYPE}.${group}`);
    }
  }
}
exports.TaskRunMetricsAggregator = TaskRunMetricsAggregator;