"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isHighlightedItem = exports.getSidebarItems = exports.getSeriesAndDomain = exports.getQueryMatcher = exports.getLegendItems = exports.getFilterMatcher = exports.getConnectingTime = exports.formatTooltipHeading = exports.formatMillisecond = exports.extractItems = exports.colourPalette = void 0;
var _eui = require("@elastic/eui");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const extractItems = data => {
  // NOTE: This happens client side as the "payload" property is mapped
  // in such a way it can't be queried (or sorted on) via ES.
  return [...data].sort((a, b) => {
    return a.requestSentTime - b.requestSentTime;
  });
};
exports.extractItems = extractItems;
const formatValueForDisplay = (value, points = 3) => {
  return Number(value).toFixed(points);
};
const getColourForMimeType = mimeType => {
  const key = mimeType && _types.MimeTypesMap[mimeType] ? _types.MimeTypesMap[mimeType] : _types.MimeType.Other;
  return colourPalette[key];
};
const getFriendlyTooltipValue = ({
  value,
  timing,
  mimeType
}) => {
  let label = _types.FriendlyTimingLabels[timing];
  if (timing === _types.Timings.Receive && mimeType) {
    const formattedMimeType = _types.MimeTypesMap[mimeType];
    label += ` (${_types.FriendlyMimetypeLabels[formattedMimeType] || mimeType})`;
  }
  return `${label}: ${formatValueForDisplay(value, value > 1 ? 0 : 1)}ms`;
};
const isHighlightedItem = (item, queryMatcher, filterMatcher) => {
  var _queryMatcher, _filterMatcher;
  if (!queryMatcher && !filterMatcher) {
    return true;
  }
  return ((_queryMatcher = queryMatcher === null || queryMatcher === void 0 ? void 0 : queryMatcher(item)) !== null && _queryMatcher !== void 0 ? _queryMatcher : true) && ((_filterMatcher = filterMatcher === null || filterMatcher === void 0 ? void 0 : filterMatcher(item)) !== null && _filterMatcher !== void 0 ? _filterMatcher : true);
};
exports.isHighlightedItem = isHighlightedItem;
const getFriendlyMetadataValue = ({
  value,
  postFix
}) => {
  // value === -1 indicates timing data cannot be extracted
  if (value === undefined || value === -1) {
    return undefined;
  }
  let formattedValue = formatValueForDisplay(value);
  if (postFix) {
    formattedValue = `${formattedValue} ${postFix}`;
  }
  return formattedValue;
};
const getConnectingTime = (connect, ssl) => {
  if (ssl && connect && ssl > 0) {
    return connect - ssl;
  } else {
    return connect;
  }
};
exports.getConnectingTime = getConnectingTime;
const getQueryMatcher = query => {
  if (!query) {
    return item => true;
  }

  /* RegExp below taken from: https://github.com/sindresorhus/escape-string-regexp/blob/main/index.js
   * First, escape all special character to use an exact string match
   * Next, replace escaped '*' with '.' to match any character and support wildcard search */
  const formattedQuery = query.replace(/[|\\{}()[\]^$+*?.]/g, '\\$&');
  const wildcardQuery = formattedQuery.replaceAll('\\*', '.');
  try {
    const regExp = new RegExp(wildcardQuery, 'i');
    return item => {
      var _item$url$search, _item$url;
      return ((_item$url$search = (_item$url = item.url) === null || _item$url === void 0 ? void 0 : _item$url.search(regExp)) !== null && _item$url$search !== void 0 ? _item$url$search : -1) > -1;
    };
  } catch (e) {
    // ignore invalid regex
  }
};
exports.getQueryMatcher = getQueryMatcher;
const getFilterMatcher = filters => {
  if (!(filters !== null && filters !== void 0 && filters.length)) {
    return item => true;
  }
  const filtersByMimeType = filters.reduce((acc, cur) => {
    acc.set(cur, true);
    return acc;
  }, new Map());
  return item => {
    var _MimeTypesMap$item$mi;
    const resolvedMimeType = item.mimeType ? (_MimeTypesMap$item$mi = _types.MimeTypesMap[item.mimeType]) !== null && _MimeTypesMap$item$mi !== void 0 ? _MimeTypesMap$item$mi : _types.MimeType.Other : _types.MimeType.Other;
    return filtersByMimeType.has(resolvedMimeType);
  };
};
exports.getFilterMatcher = getFilterMatcher;
const getSeriesAndDomain = (items, onlyHighlighted = false, dateFormatter, query, activeFilters, markerItems) => {
  const getValueForOffset = item => {
    return item.requestSentTime;
  };
  // The earliest point in time a request is sent or started. This will become our notion of "0".
  let zeroOffset = Infinity;
  items.forEach(i => zeroOffset = Math.min(zeroOffset, getValueForOffset(i)));
  const getValue = (timings, timing) => {
    if (!timings) return;

    // SSL is a part of the connect timing
    if (timing === _types.Timings.Connect) {
      return getConnectingTime(timings.connect, timings.ssl);
    }
    return timings[timing];
  };
  const series = [];
  const metadata = [];
  let totalHighlightedRequests = 0;
  const queryMatcher = getQueryMatcher(query);
  const filterMatcher = getFilterMatcher(activeFilters);
  items.forEach((item, index) => {
    let showTooltip = true;
    const mimeTypeColour = getColourForMimeType(item.mimeType);
    const offsetValue = getValueForOffset(item);
    let currentOffset = offsetValue - zeroOffset;
    const requestStart = currentOffset;
    const isHighlighted = isHighlightedItem(item, queryMatcher, filterMatcher);
    if (isHighlighted) {
      totalHighlightedRequests++;
    }
    if (!item.timings) {
      series.push({
        x: index,
        y0: 0,
        y: 0,
        config: {
          isHighlighted,
          showTooltip: false
        }
      });
      return;
    }
    let timingValueFound = false;
    const networkItemTooltipProps = [];
    _types.TIMING_ORDER.forEach(timing => {
      const value = getValue(item.timings, timing);
      const colour = timing === _types.Timings.Receive ? mimeTypeColour : colourPalette[timing];
      if (value !== null && value !== undefined && value >= 0) {
        timingValueFound = true;
        const y = currentOffset + value;
        const tooltipProps = {
          value: getFriendlyTooltipValue({
            value: y - currentOffset,
            timing,
            mimeType: item.mimeType
          }),
          colour
        };
        networkItemTooltipProps.push(tooltipProps);
        series.push({
          x: index,
          y0: currentOffset,
          y,
          config: {
            id: index,
            colour,
            isHighlighted
          }
        });
        currentOffset = y;
      }
    });

    /* if no specific timing values are found, use the total time
     * if total time is not available use 0, set showTooltip to false,
     * and omit tooltip props */
    if (!timingValueFound) {
      showTooltip = false;
      const total = item.timings.total;
      const hasTotal = total !== -1;
      if (hasTotal) {
        networkItemTooltipProps.push({
          value: getFriendlyTooltipValue({
            value: total,
            timing: _types.Timings.Receive,
            mimeType: item.mimeType
          }),
          colour: mimeTypeColour
        });
      }
      series.push({
        x: index,
        y0: hasTotal ? currentOffset : 0,
        y: hasTotal ? currentOffset + item.timings.total : 0,
        config: {
          isHighlighted,
          colour: hasTotal ? mimeTypeColour : ''
        }
      });
    }
    metadata.push(formatMetadata({
      item,
      index,
      showTooltip,
      requestStart,
      dateFormatter,
      networkItemTooltipProps
    }));
  });
  const yValues = series.map(serie => serie.y);
  const domain = {
    min: 0,
    max: Math.max(...yValues)
  };
  let filteredSeries = series;
  if (onlyHighlighted) {
    filteredSeries = series.filter(item => item.config.isHighlighted);
  }
  if (markerItems && markerItems.length > 0) {
    // set domain to include marker items, whichever has higher time value
    const highestMarkerTime = Math.max(...markerItems.map(item => item.offset));
    domain.max = Math.max(domain.max, highestMarkerTime);
  }
  return {
    series: filteredSeries,
    domain,
    metadata,
    totalHighlightedRequests
  };
};
exports.getSeriesAndDomain = getSeriesAndDomain;
const formatHeaders = headers => {
  if (typeof headers === 'undefined') {
    return undefined;
  }
  return Object.keys(headers).map(key => ({
    name: key,
    value: `${headers[key]}`
  }));
};
const formatMetadata = ({
  item,
  index,
  requestStart,
  dateFormatter,
  showTooltip,
  networkItemTooltipProps
}) => {
  const {
    certificates,
    ip,
    mimeType,
    requestHeaders,
    responseHeaders,
    url,
    resourceSize,
    transferSize,
    status
  } = item;
  const {
    dns,
    connect,
    ssl,
    wait,
    receive,
    total
  } = item.timings || {};
  const contentDownloaded = receive && receive > 0 ? receive : total;
  return {
    x: index,
    url,
    networkItemTooltipProps,
    showTooltip,
    requestHeaders: formatHeaders(requestHeaders),
    responseHeaders: formatHeaders(responseHeaders),
    certificates: certificates ? [{
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateIssuer],
      value: certificates.issuer
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateIssueDate],
      value: certificates.validFrom ? dateFormatter(certificates.validFrom) : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateExpiryDate],
      value: certificates.validTo ? dateFormatter(certificates.validTo) : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.CertificateSubject],
      value: certificates.subjectName
    }] : undefined,
    details: [{
      name: _types.FriendlyFlyoutLabels[_types.Metadata.Status],
      value: status ? `${status}` : undefined
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.MimeType],
      value: mimeType
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.RequestStart],
      value: getFriendlyMetadataValue({
        value: requestStart,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Dns],
      value: getFriendlyMetadataValue({
        value: dns,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Connect],
      value: getFriendlyMetadataValue({
        value: getConnectingTime(connect, ssl),
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Ssl],
      value: getFriendlyMetadataValue({
        value: ssl,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Wait],
      value: getFriendlyMetadataValue({
        value: wait,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyTimingLabels[_types.Timings.Receive],
      value: getFriendlyMetadataValue({
        value: contentDownloaded,
        postFix: 'ms'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.ResourceSize],
      value: getFriendlyMetadataValue({
        value: resourceSize ? resourceSize / 1000 : undefined,
        postFix: 'KB'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.TransferSize],
      value: getFriendlyMetadataValue({
        value: transferSize ? transferSize / 1000 : undefined,
        postFix: 'KB'
      })
    }, {
      name: _types.FriendlyFlyoutLabels[_types.Metadata.IP],
      value: ip
    }]
  };
};
const getSidebarItems = (items, onlyHighlighted, query, activeFilters) => {
  const queryMatcher = getQueryMatcher(query);
  const filterMatcher = getFilterMatcher(activeFilters);
  const sideBarItems = items.map((item, index) => {
    const isHighlighted = isHighlightedItem(item, queryMatcher, filterMatcher);
    const offsetIndex = index + 1;
    const {
      url,
      status,
      method
    } = item;
    return {
      url,
      status,
      method,
      isHighlighted,
      offsetIndex,
      index
    };
  });
  if (onlyHighlighted) {
    return sideBarItems.filter(item => item.isHighlighted);
  }
  return sideBarItems;
};
exports.getSidebarItems = getSidebarItems;
const getLegendItems = () => {
  let timingItems = [];
  Object.values(_types.Timings).forEach(timing => {
    // The "receive" timing is mapped to a mime type colour, so we don't need to show this in the legend
    if (timing === _types.Timings.Receive) {
      return;
    }
    timingItems = [...timingItems, {
      name: _types.FriendlyTimingLabels[timing],
      color: TIMING_PALETTE[timing]
    }];
  });
  let mimeTypeItems = [];
  Object.values(_types.MimeType).forEach(mimeType => {
    mimeTypeItems = [...mimeTypeItems, {
      name: _types.FriendlyMimetypeLabels[mimeType],
      color: MIME_TYPE_PALETTE[mimeType]
    }];
  });
  return [...timingItems, ...mimeTypeItems];
};

// Timing colour palette
exports.getLegendItems = getLegendItems;
const SAFE_PALETTE = (0, _eui.euiPaletteColorBlind)({
  rotations: 2
});
const buildTimingPalette = () => {
  const palette = Object.values(_types.Timings).reduce((acc, value) => {
    switch (value) {
      case _types.Timings.Blocked:
        acc[value] = SAFE_PALETTE[11];
        break;
      case _types.Timings.Dns:
        acc[value] = SAFE_PALETTE[10];
        break;
      case _types.Timings.Connect:
        acc[value] = SAFE_PALETTE[13];
        break;
      case _types.Timings.Ssl:
        acc[value] = SAFE_PALETTE[14];
        break;
      case _types.Timings.Send:
        acc[value] = SAFE_PALETTE[19];
        break;
      case _types.Timings.Wait:
        acc[value] = SAFE_PALETTE[9];
        break;
      case _types.Timings.Receive:
        acc[value] = SAFE_PALETTE[15];
        break;
    }
    return acc;
  }, {});
  return palette;
};
const TIMING_PALETTE = buildTimingPalette();

// MimeType colour palette

const buildMimeTypePalette = () => {
  const palette = Object.values(_types.MimeType).reduce((acc, value) => {
    switch (value) {
      case _types.MimeType.Html:
        acc[value] = SAFE_PALETTE[1];
        break;
      case _types.MimeType.Script:
        acc[value] = SAFE_PALETTE[7];
        break;
      case _types.MimeType.Stylesheet:
        acc[value] = SAFE_PALETTE[3];
        break;
      case _types.MimeType.Image:
        acc[value] = SAFE_PALETTE[4];
        break;
      case _types.MimeType.Media:
        acc[value] = SAFE_PALETTE[5];
        break;
      case _types.MimeType.Font:
        acc[value] = SAFE_PALETTE[2];
        break;
      case _types.MimeType.XHR:
        acc[value] = SAFE_PALETTE[0];
        break;
      case _types.MimeType.Other:
        acc[value] = SAFE_PALETTE[6];
        break;
    }
    return acc;
  }, {});
  return palette;
};
const MIME_TYPE_PALETTE = buildMimeTypePalette();
const colourPalette = exports.colourPalette = {
  ...TIMING_PALETTE,
  ...MIME_TYPE_PALETTE
};
const formatTooltipHeading = (index, fullText) => isNaN(index) ? fullText : `${index}. ${fullText}`;
exports.formatTooltipHeading = formatTooltipHeading;
const formatMillisecond = (ms, {
  maxMillis = 1000,
  digits
}) => {
  if (ms < 0) {
    return '--';
  }
  if (ms < maxMillis) {
    return `${ms.toFixed(digits !== null && digits !== void 0 ? digits : 0)} ms`;
  }
  return `${(ms / 1000).toFixed(digits !== null && digits !== void 0 ? digits : 2)} s`;
};
exports.formatMillisecond = formatMillisecond;