"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateFeatureRoute = exports.listFeaturesRoute = exports.identifyFeaturesRoute = exports.getFeatureRoute = exports.featureRoutes = exports.describeStreamRoute = exports.deleteFeatureRoute = exports.bulkFeaturesRoute = void 0;
var _zod = require("@kbn/zod");
var _streamsSchema = require("@kbn/streams-schema");
var _streamlang = require("@kbn/streamlang");
var _streamsAi = require("@kbn/streams-ai");
var _rxjs = require("rxjs");
var _create_server_route = require("../../../create_server_route");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _constants = require("../../../../../common/constants");
var _assert_significant_events_access = require("../../../utils/assert_significant_events_access");
var _run_feature_identification = require("../../../../lib/streams/feature/run_feature_identification");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const dateFromString = _zod.z.string().transform(input => new Date(input));
const getFeatureRoute = exports.getFeatureRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/features/{featureName}',
  options: {
    access: 'internal',
    summary: 'Get a feature for a stream',
    description: 'Fetches the specified feature'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      featureName: _streamsSchema.streamObjectNameSchema
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      name,
      featureName
    } = params.path;
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${name}, insufficient privileges`);
    }
    const feature = await featureClient.getFeature(name, featureName);
    return {
      feature
    };
  }
});
const deleteFeatureRoute = exports.deleteFeatureRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'DELETE /internal/streams/{name}/features/{featureName}',
  options: {
    access: 'internal',
    summary: 'Delete a feature for a stream',
    description: 'Deletes the specified feature'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      featureName: _streamsSchema.streamObjectNameSchema
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      name,
      featureName
    } = params.path;
    const {
      write
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!write) {
      throw new _security_error.SecurityError(`Cannot delete feature for stream ${name}, insufficient privileges`);
    }
    return await featureClient.deleteFeature(name, featureName);
  }
});
const updateFeatureRoute = exports.updateFeatureRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /internal/streams/{name}/features/{featureName}',
  options: {
    access: 'internal',
    summary: 'Updates a feature for a stream',
    description: 'Updates the specified feature'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      featureName: _streamsSchema.streamObjectNameSchema
    }),
    body: _zod.z.object({
      description: _zod.z.string(),
      filter: _streamlang.conditionSchema
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      path: {
        name,
        featureName
      },
      body: {
        description,
        filter
      }
    } = params;
    const {
      write
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!write) {
      throw new _security_error.SecurityError(`Cannot update features for stream ${name}, insufficient privileges`);
    }
    return await featureClient.updateFeature(name, {
      name: featureName,
      description,
      filter
    });
  }
});
const listFeaturesRoute = exports.listFeaturesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/features',
  options: {
    access: 'internal',
    summary: 'Lists all features for a stream',
    description: 'Fetches all features for the specified stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      name
    } = params.path;
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${name}, insufficient privileges`);
    }
    const {
      hits: features
    } = await featureClient.getFeatures(name);
    return {
      features
    };
  }
});
const bulkFeaturesRoute = exports.bulkFeaturesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/features/_bulk',
  options: {
    access: 'internal',
    summary: 'Bulk changes to features',
    description: 'Add or delete features in bulk for a given stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      operations: _zod.z.array(_zod.z.union([_zod.z.object({
        index: _zod.z.object({
          feature: _streamsSchema.featureSchema
        })
      }), _zod.z.object({
        delete: _zod.z.object({
          feature: _zod.z.object({
            name: _streamsSchema.streamObjectNameSchema
          })
        })
      })]))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      path: {
        name
      },
      body: {
        operations
      }
    } = params;
    const {
      write
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!write) {
      throw new _security_error.SecurityError(`Cannot update features for stream ${name}, insufficient privileges`);
    }
    return await featureClient.bulk(name, operations);
  }
});
const identifyFeaturesRoute = exports.identifyFeaturesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/features/_identify',
  options: {
    access: 'internal',
    summary: 'Identify features in a stream',
    description: 'Identify features in a stream with an LLM'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      connectorId: _zod.z.string(),
      from: dateFromString,
      to: dateFromString
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server,
    logger
  }) => {
    const {
      featureClient,
      scopedClusterClient,
      licensing,
      uiSettingsClient,
      streamsClient,
      inferenceClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      path: {
        name
      },
      query: {
        connectorId,
        from: start,
        to: end
      }
    } = params;
    const {
      write
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!write) {
      throw new _security_error.SecurityError(`Cannot update features for stream ${name}, insufficient privileges`);
    }
    const [{
      hits
    }, stream] = await Promise.all([featureClient.getFeatures(name), streamsClient.getStream(name)]);
    const {
      features
    } = await (0, _run_feature_identification.runFeatureIdentification)({
      start: start.getTime(),
      end: end.getTime(),
      esClient: scopedClusterClient.asCurrentUser,
      inferenceClient: inferenceClient.bindTo({
        connectorId
      }),
      logger,
      stream,
      features: hits
    });
    return {
      type: 'identified_features',
      features
    };
  }
});
const describeStreamRoute = exports.describeStreamRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/_describe_stream',
  options: {
    access: 'internal',
    summary: 'Generate a stream description',
    description: 'Generate a stream description based on data in the stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      connectorId: _zod.z.string(),
      from: dateFromString,
      to: dateFromString
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      scopedClusterClient,
      licensing,
      uiSettingsClient,
      streamsClient,
      inferenceClient
    } = await getScopedClients({
      request
    });
    await (0, _assert_significant_events_access.assertSignificantEventsAccess)({
      server,
      licensing,
      uiSettingsClient
    });
    const {
      path: {
        name
      },
      query: {
        connectorId,
        from: start,
        to: end
      }
    } = params;
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot generate stream description for ${name}, insufficient privileges`);
    }
    const stream = await streamsClient.getStream(name);
    return (0, _rxjs.from)((0, _streamsAi.generateStreamDescription)({
      stream,
      esClient: scopedClusterClient.asCurrentUser,
      inferenceClient: inferenceClient.bindTo({
        connectorId
      }),
      start: start.valueOf(),
      end: end.valueOf()
    })).pipe((0, _rxjs.map)(description => {
      return {
        type: 'stream_description',
        description
      };
    }));
  }
});
const featureRoutes = exports.featureRoutes = {
  ...getFeatureRoute,
  ...deleteFeatureRoute,
  ...updateFeatureRoute,
  ...listFeaturesRoute,
  ...bulkFeaturesRoute,
  ...identifyFeaturesRoute,
  ...describeStreamRoute
};