"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerStreamsUsageCollector = registerStreamsUsageCollector;
var _streamsSchema = require("@kbn/streams-schema");
var _utils = require("./utils");
var _register_collector = require("./register_collector");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Reader abstraction to hide storage/client specifics from the collector

function createFetchFunction(logger, getReader) {
  return async function fetchStreamsUsageStats(_) {
    try {
      const reader = await getReader();
      const allStreamsData = await reader.readAllManagedStreams();
      const {
        classicStreamsMetrics,
        wiredStreamsCount
      } = processStreamsData(allStreamsData);
      return {
        classic_streams: classicStreamsMetrics,
        wired_streams: {
          count: wiredStreamsCount
        }
      };
    } catch (error) {
      logger.error('Failed to collect Streams telemetry data', error);
      throw error;
    }
  };
  function processStreamsData(streamDefinitions) {
    let changedCount = 0;
    let withProcessingCount = 0;
    let withFieldsCount = 0;
    let withChangedRetentionCount = 0;
    let wiredCount = 0;
    for (const definition of streamDefinitions) {
      if (_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
        // Ignore group streams for current metrics
        continue;
      }
      if (_streamsSchema.Streams.WiredStream.Definition.is(definition)) {
        wiredCount++;
      } else if (_streamsSchema.Streams.ClassicStream.Definition.is(definition)) {
        // Presence of a classic stream in storage implies it has been stored/changed
        changedCount++;
        if ((0, _utils.hasProcessingSteps)(definition)) {
          withProcessingCount++;
        }
        if ((0, _utils.hasFieldOverrides)(definition)) {
          withFieldsCount++;
        }
        if ((0, _utils.hasChangedRetention)(definition.ingest.lifecycle)) {
          withChangedRetentionCount++;
        }
      }
    }
    return {
      classicStreamsMetrics: {
        changed_count: changedCount,
        with_processing_count: withProcessingCount,
        with_fields_count: withFieldsCount,
        with_changed_retention_count: withChangedRetentionCount
      },
      wiredStreamsCount: wiredCount
    };
  }
}

/**
 * Registers the Streams usage statistics collector
 */
function registerStreamsUsageCollector(usageCollection, logger, getReader) {
  (0, _register_collector.registerStreamsUsageCollector)(usageCollection, {
    isReady: () => true,
    fetch: createFetchFunction(logger, getReader)
  });
}