"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ProcessorSuggestionsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _template_semantics = require("../helpers/template_semantics");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PROCESSOR_ENDPOINT_ID = 'ingest.put_pipeline';
function isRecord(value) {
  return typeof value === 'object' && value !== null;
}
function isProcessorEntry(value) {
  return isRecord(value) && Object.values(value).every(definition => definition === undefined || isRecord(definition));
}
function extractProcessorEntries(spec) {
  var _processorArray$;
  const endpoints = spec.endpoints;
  if (!isRecord(endpoints)) return [];
  const ingestEndpoint = endpoints[PROCESSOR_ENDPOINT_ID];
  if (!isRecord(ingestEndpoint)) return [];
  const rules = ingestEndpoint.data_autocomplete_rules;
  if (!isRecord(rules)) return [];
  const processorArray = rules.processors;
  if (!Array.isArray(processorArray) || processorArray.length === 0) return [];
  const oneOf = (_processorArray$ = processorArray[0]) === null || _processorArray$ === void 0 ? void 0 : _processorArray$.__one_of;
  if (!Array.isArray(oneOf)) return [];
  return oneOf.filter(isProcessorEntry);
}
class ProcessorSuggestionsService {
  constructor() {
    (0, _defineProperty2.default)(this, "fetcher", void 0);
  }
  setConsoleStart(consoleStart) {
    this.fetcher = () => consoleStart.getSpecJson();
  }
  async getSuggestions() {
    if (!this.fetcher) {
      return [];
    }
    try {
      const spec = this.fetcher();
      return this.buildSuggestions(spec);
    } catch {
      return [];
    }
  }
  buildSuggestions(spec) {
    const entries = extractProcessorEntries(spec);
    return entries.map(entry => {
      const name = Object.keys(entry)[0];
      if (!name) {
        return undefined;
      }
      const def = entry[name];
      const template = def === null || def === void 0 ? void 0 : def.__template;
      return {
        name,
        template
      };
    }).filter(suggestion => suggestion !== undefined);
  }
  buildPropertiesByProcessor(spec) {
    const processorEntries = extractProcessorEntries(spec);
    const propertiesByProcessorMap = {};
    for (const entry of processorEntries) {
      const processorName = Object.keys(entry)[0];
      if (!processorName) continue;
      const processorDefinition = entry[processorName];
      if (!processorDefinition || typeof processorDefinition !== 'object') continue;
      const propertySuggestions = [];
      for (const [propertyName, propertyRule] of Object.entries(processorDefinition)) {
        // Filter out metadata keys like __template, __one_of, __doc, etc.
        if (propertyName.startsWith('__')) continue;
        const rawTemplate = this.extractTemplateFromRule(propertyRule);
        const normalizedTemplate = this.normalizeToJsonValue(rawTemplate);
        propertySuggestions.push({
          name: propertyName,
          template: normalizedTemplate
        });
      }
      propertiesByProcessorMap[processorName] = propertySuggestions;
    }
    return propertiesByProcessorMap;
  }
  extractTemplateFromRule(rule) {
    return (0, _template_semantics.getTemplateFromRule)(rule);
  }
  normalizeToJsonValue(template) {
    if (template === undefined) return undefined;
    if (template === null) return null;
    const t = typeof template;
    if (t === 'string' || t === 'number' || t === 'boolean') return template;
    if (Array.isArray(template)) return template;
    if (isRecord(template)) return template;
    return undefined;
  }
  async getAllSuggestions() {
    if (!this.fetcher) {
      return {
        processors: [],
        propertiesByProcessor: {}
      };
    }
    try {
      const spec = this.fetcher();
      return {
        processors: this.buildSuggestions(spec),
        propertiesByProcessor: this.buildPropertiesByProcessor(spec)
      };
    } catch {
      return {
        processors: [],
        propertiesByProcessor: {}
      };
    }
  }
}
exports.ProcessorSuggestionsService = ProcessorSuggestionsService;